package ru.yandex.direct.grid.processing.service.communication;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.communication.container.web.CommunicationDataContainer;
import ru.yandex.direct.communication.container.web.CommunicationLinkInfo;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.container.web.CommunicationMessageButton;
import ru.yandex.direct.communication.container.web.CommunicationMessageContent;
import ru.yandex.direct.communication.container.web.CommunicationMessageGroup;
import ru.yandex.direct.communication.container.web.CommunicationTargetObject;
import ru.yandex.direct.communication.facade.ActionTarget;
import ru.yandex.direct.communication.facade.impl.actions.SubResult;
import ru.yandex.direct.communication.service.CommunicationChannelService;
import ru.yandex.direct.core.entity.communication.model.LinkPath;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.banner.GdAdFilter;
import ru.yandex.direct.grid.processing.model.campaign.GdCampaignFilter;
import ru.yandex.direct.grid.processing.model.communication.DataContainerType;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationActionResult;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationActionSubResult;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationActionTarget;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationDataContainer;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationLinkInfo;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessage;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageButton;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageButtonAction;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageButtonActionType;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageButtonStyle;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageButtonViewType;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageContent;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageGroup;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageGroupList;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageList;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageSeverity;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationMessageStatuses;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationTargetObject;
import ru.yandex.direct.grid.processing.model.communication.GdCommunicationTargetObjectType;
import ru.yandex.direct.grid.processing.model.communication.GdEntityType;
import ru.yandex.direct.grid.processing.model.communication.GdLinkPath;
import ru.yandex.direct.grid.processing.model.communication.SlotName;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupFilter;
import ru.yandex.direct.grid.processing.model.shortener.GdShortFilterUnion;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
class CommunicationInfoService {
    private final CommunicationChannelService communicationChannelService;
    private final GridShortenerService gridShortenerService;

    @Autowired
    public CommunicationInfoService(
            CommunicationChannelService communicationChannelService,
            GridShortenerService gridShortenerService
    ) {
        this.communicationChannelService = communicationChannelService;
        this.gridShortenerService = gridShortenerService;
    }

    private GdCommunicationTargetObject toGdCommunicationTargetObject(CommunicationTargetObject object) {
        return new GdCommunicationTargetObject()
                .withId(object.getId())
                .withType(GdCommunicationTargetObjectType.fromSource(object.getType()));
    }

    private GdCommunicationMessageButton toGdCommunicationMessageButton(CommunicationMessageButton button) {
        return new GdCommunicationMessageButton()
                .withId(button.getId())
                .withTitle(button.getTitle())
                .withConfirmText(button.getConfirmText())
                .withStyle(GdCommunicationMessageButtonStyle.fromSource(button.getStyle()))
                .withAction(GdCommunicationMessageButtonAction.fromSource(button.getAction()))
                .withViewType(GdCommunicationMessageButtonViewType.fromSource(button.getViewType()))
                .withActionType(GdCommunicationMessageButtonActionType.fromSource(button.getActionType()))
                .withLinkInfo(toGdLinkInfo(button.getLinkInfo()))
                .withHref(button.getHref())
                .withDisabled(button.isDisabled());
    }

    private GdCommunicationLinkInfo toGdLinkInfo(CommunicationLinkInfo linkInfo) {
        if (linkInfo == null) {
            return null;
        }
        return new GdCommunicationLinkInfo()
                .withPath(GdLinkPath.fromSource(linkInfo.getPath()))
                .withEntityType(GdEntityType.fromSource(linkInfo.getEntityType()))
                .withEntityIds(linkInfo.getEntityIds())
                .withCampaignId(linkInfo.getCampaignId())
                .withFilterKey(getFilterKey(linkInfo));
    }

    private String getFilterKey(CommunicationLinkInfo linkInfo) {
        if (linkInfo == null ||
                linkInfo.getPath() != LinkPath.GRID ||
                linkInfo.getEntityType() == null ||
                linkInfo.getClientId() == null ||
                linkInfo.getEntityIds() == null ||
                linkInfo.getEntityIds().isEmpty()
        ) {
            return null;
        }
        GdShortFilterUnion filterUnion = new GdShortFilterUnion();
        switch (linkInfo.getEntityType()) {
            case CAMPAIGN:
                filterUnion.setCampaignFilter(new GdCampaignFilter()
                        .withCampaignIdIn(new HashSet<>(linkInfo.getEntityIds())));
                break;
            case GROUP:
                filterUnion.setAdGroupFilter(new GdAdGroupFilter()
                        .withCampaignIdIn(Set.of(linkInfo.getCampaignId()))
                        .withAdGroupIdIn(new HashSet<>(linkInfo.getEntityIds())));
                break;
            case BANNER:
                filterUnion.setAdFilter(new GdAdFilter()
                        .withCampaignIdIn(Set.of(linkInfo.getCampaignId()))
                        .withAdIdIn(new HashSet<>(linkInfo.getEntityIds())));
                break;
            default:
                return null;
        }
        return gridShortenerService.saveFilter(linkInfo.getClientId(), filterUnion, null).getFilterKey();
    }

    private GdCommunicationMessageContent toGdCommunicationMessageContent(CommunicationMessageContent content) {
        if (content == null) {
            return null;
        }
        return new GdCommunicationMessageContent()
                .withText(content.getText())
                .withTitle(content.getTitle())
                .withImageUrl(content.getImageUrl())
                .withButtons(StreamEx.of(content.getButtons()).map(
                        button -> toGdCommunicationMessageButton(button)).toList())
                .withDataContainer(toGdDataContainer(content.getDataContainer()))
                .withApplyDateTime(content.getApplyDateTime())
                .withParams(content.getParams());
    }

    private GdCommunicationDataContainer toGdDataContainer(CommunicationDataContainer dataContainer) {
        return new GdCommunicationDataContainer()
                .withType(DataContainerType.fromSource(dataContainer.getType()))
                .withFieldName(dataContainer.getFieldName())
                .withText(dataContainer.getText())
                .withSubData(mapList(dataContainer.getSubData(), this::toGdDataContainer));
    }

    private GdCommunicationMessage toGdCommunicationMessage(CommunicationMessage message) {
        var slotStr = message.getSlot().getName();
        var slotEnum = SlotName.fromTypedValue(slotStr);
        return new GdCommunicationMessage()
                .withClientId(message.getClientId())
                .withSlot(slotEnum)
                .withActualSlots(List.of(slotStr))
                .withActualSlotNames(List.of(slotEnum))
                .withSeverity(GdCommunicationMessageSeverity.fromSource(message.getSeverity()))
                .withStatus(GdCommunicationMessageStatuses.fromSource(message.getStatus()))
                .withIsOptional(!message.isDisableForEdit())
                .withDisableForEdit(message.isDisableForEdit())
                .withName(nvl(message.getName(), ""))
                .withGroupName(nvl(message.getGroupName(), ""))
                .withMajorVersion(message.getMajorVersion())
                .withMinorVersion(message.getMinorVersion())
                .withInventoryAdditionalData(message.getInventoryAdditionalData())
                .withContent(toGdCommunicationMessageContent(message.getContent()))
                .withEventId(message.getEventId())
                .withEventVersionId(message.getEventVersionId())
                .withTargetObject(toGdCommunicationTargetObject(message.getTargetObject()));
    }

    private GdCommunicationMessageGroup toGdCommunicationMessageGroup(CommunicationMessageGroup group) {
        return new GdCommunicationMessageGroup()
                .withMessages(StreamEx.of(group.getMessages()).map(
                        message -> toGdCommunicationMessage(message)).toList())
                .withName(nvl(group.getName(), ""))
                .withTitle(group.getContent().getTitle())
                .withText(group.getContent().getText())
                .withImageUrl(group.getContent().getImageUrl())
                .withContent(toGdCommunicationMessageContent(group.getContent()));
    }

    public GdCommunicationMessageList getCommunicationMessageList(
            ClientId clientId,
            Long operatorUid,
            List<String> slots,
            List<GdCommunicationTargetObject> targetObjects,
            String interfaceLanguage
    ) {
        var messages = communicationChannelService.getCommunicationMessage(clientId,
                operatorUid, new HashSet<>(slots), StreamEx.of(targetObjects).map(o -> o.getId()).toList(),
                interfaceLanguage);

        var result = StreamEx.of(messages)
                .map(message -> toGdCommunicationMessage(message))
                .toList();

        return new GdCommunicationMessageList()
                .withMessages(result);
    }

    public GdCommunicationMessageGroupList getCommunicationMessageGroupList(
            ClientId clientId,
            Long operatorUid,
            List<String> slots,
            List<GdCommunicationTargetObject> targetObjects,
            String interfaceLanguage) {
        var groups = communicationChannelService.getCommunicationMessageGroup(clientId,
                operatorUid, new HashSet<>(slots), StreamEx.of(targetObjects).map(o -> o.getId()).toList(),
                interfaceLanguage);

        var result = StreamEx.of(groups)
                .map(group -> toGdCommunicationMessageGroup(group))
                .toList();

        return new GdCommunicationMessageGroupList()
                .withGroups(result);
    }

    public GdCommunicationActionResult confirmAction(
            ClientId clientId,
            Long operatorUid,
            Long requestId,
            Long buttonId,
            String slot,
            List<GdCommunicationActionTarget> targets,
            String lang) {
        var result = communicationChannelService.buttonAction(
                buttonId, requestId, clientId, operatorUid, slot, lang,
                mapList(targets, CommunicationInfoService::toCoreTarget));
        return new GdCommunicationActionResult()
                .withSuccessObjectIds(result.getSuccessObjects())
                .withFailedObjectIds(result.getFailedObjects())
                .withSubResults(mapList(result.getSubResults(), CommunicationInfoService::convertSubResult));
    }

    private static GdCommunicationActionSubResult convertSubResult(SubResult result) {
        return new GdCommunicationActionSubResult()
                .withObjectId(result.getObjectId())
                .withTitle(result.getTitle())
                .withText(result.getText());
    }

    private static ActionTarget toCoreTarget(GdCommunicationActionTarget target) {
        return new ActionTarget(
                target.getTargetObjectId(),
                target.getEventId(),
                target.getEventVersionId(),
                target.getMajorDataVersion(),
                target.getMinorDataVersion(),
                target.getInventoryAdditionalData()
        );
    }

    public GdCommunicationMessageList getAutoAppliedRecommendationList(
            ClientId clientId,
            Long operatorUid,
            List<String> slots,
            List<GdCommunicationTargetObject> targetObjects,
            String interfaceLanguage
    ) {
        var messages = communicationChannelService.getAutoAppliedRecommendation(clientId,
                operatorUid, new HashSet<>(slots), StreamEx.of(targetObjects).map(o -> o.getId()).toList(),
                interfaceLanguage);

        var result = StreamEx.of(messages)
                .map(message -> toGdCommunicationMessage(message))
                .toList();

        return new GdCommunicationMessageList()
                .withMessages(result);
    }
}
