package ru.yandex.direct.grid.processing.service.constant;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportParametersService;
import ru.yandex.direct.core.entity.banner.type.button.BannerWithButtonHelper;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.internalads.service.PlaceService;
import ru.yandex.direct.core.entity.internalads.service.TemplatePlaceService;
import ru.yandex.direct.core.entity.internalads.service.TemplateResourceService;
import ru.yandex.direct.core.entity.page.service.PageService;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesRepository;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionShortcutService;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.repository.GeoTimezoneRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.constants.GdBusinessUnit;
import ru.yandex.direct.grid.processing.model.constants.GdButtonAction;
import ru.yandex.direct.grid.processing.model.constants.GdButtonActionDetailed;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlaceInfo;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlacesInfoContainer;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlace;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlacesContainer;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResource;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResourcesContainer;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStation;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStationsContainer;
import ru.yandex.direct.grid.processing.model.constants.GdOsVersions;
import ru.yandex.direct.grid.processing.model.constants.GdTargetingCategory;
import ru.yandex.direct.grid.processing.model.constants.GdTimezoneGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUcRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalPageInfo;
import ru.yandex.direct.grid.processing.service.freelancer.FreelancerDataService;
import ru.yandex.direct.grid.processing.service.offlinereport.OfflineReportValidationService;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.regions.GeoTree;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.relevancematch.Constants.DISABLED_RELEVANCE_MATCH_CATEGORIES;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdRole;
import static ru.yandex.direct.grid.processing.service.constant.ConstantsConverter.toMetroPredicate;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class ConstantDataService {
    private final ClientGeoService clientGeoService;
    private final FreelancerDataService freelancerDataService;
    private final Supplier<LocalDate> agencyOfflineReportMaximumAvailableDateCached;
    private final Supplier<LocalDate> agencyKpiOfflineReportMaximumAvailableDateCached;
    private final Supplier<LocalDate> agencyKpiOfflineReportMinimumAvailableDateCached;
    private final TemplatePlaceService templatePlaceService;
    private final TemplateResourceService templateResourceService;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final GeoTimezoneRepository geoTimezoneRepository;
    private final TranslationService translationService;
    private final String turbolandingUnifiedApiUrl;
    private final String metrikaUiUrlPattern;
    private final TargetingCategoriesRepository targetingCategoriesRepository;
    private final BannerWithButtonHelper newBannerWithButtonHelper;
    private final PageService pageService;
    private final PlaceService placeService;
    private final ProductService productService;
    private final CrowdTestSettingsService crowdTestSettingsService;
    private final FeatureService featureService;
    private final String balanceDomain;

    @Autowired
    public ConstantDataService(@Value("${turbo_landings.url}") String turbolandingUnifiedApiUrl,
                               @Value("${metrika.metrika_ui_url_pattern}") String metrikaUiUrlPattern,
                               @Value("${balance.domain}") String balanceDomain,
                               ClientGeoService clientGeoService,
                               FreelancerDataService freelancerDataService,
                               AgencyOfflineReportParametersService agencyOfflineReportParametersService,
                               TemplatePlaceService templatePlaceService,
                               TemplateResourceService templateResourceService,
                               SspPlatformsRepository sspPlatformsRepository,
                               GeoTimezoneRepository geoTimezoneRepository,
                               TranslationService translationService,
                               TargetingCategoriesRepository targetingCategoriesRepository,
                               BannerWithButtonHelper newBannerWithButtonHelper,
                               PageService pageService,
                               PlaceService placeService,
                               OfflineReportValidationService offlineReportValidationService,
                               ProductService productService,
                               CrowdTestSettingsService crowdTestSettingsService,
                               FeatureService featureService) {
        this.clientGeoService = clientGeoService;
        this.freelancerDataService = freelancerDataService;
        this.agencyOfflineReportMaximumAvailableDateCached = Suppliers.memoizeWithExpiration(
                agencyOfflineReportParametersService::getMaximumAvailableDate, 10, TimeUnit.MINUTES);
        this.agencyKpiOfflineReportMaximumAvailableDateCached = Suppliers.memoizeWithExpiration(
                offlineReportValidationService::getAgencyKpiMaximumAvailableDate, 10, TimeUnit.MINUTES);
        this.agencyKpiOfflineReportMinimumAvailableDateCached = Suppliers.memoizeWithExpiration(
                offlineReportValidationService::getAgencyKpiMinimumAvailableDate, 10, TimeUnit.MINUTES);
        this.templatePlaceService = templatePlaceService;
        this.templateResourceService = templateResourceService;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.geoTimezoneRepository = geoTimezoneRepository;
        this.translationService = translationService;
        this.turbolandingUnifiedApiUrl = turbolandingUnifiedApiUrl;
        this.metrikaUiUrlPattern = metrikaUiUrlPattern;
        this.targetingCategoriesRepository = targetingCategoriesRepository;
        this.newBannerWithButtonHelper = newBannerWithButtonHelper;
        this.pageService = pageService;
        this.placeService = placeService;
        this.productService = productService;
        this.crowdTestSettingsService = crowdTestSettingsService;
        this.featureService = featureService;
        this.balanceDomain = balanceDomain;
    }


    /**
     * Получить список станций метрополитена
     *
     * @param operatorClientId идентификатор оператора
     * @param input            Входные параметры для получения станций метрополитена
     * @return список станций метрополитена отсортированный по названию
     */
    List<GdMetroStation> getMetroStations(ClientId operatorClientId, GdMetroStationsContainer input) {
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(operatorClientId);

        return StreamEx.of(geoTree.getMetroMap().values())
                .filter(toMetroPredicate(input.getFilter()))
                .map(ConstantsConverter::toGdMetroStation)
                .sorted(Comparator.comparing(GdMetroStation::getMetroStationName))
                .toList();
    }

    /**
     * Получить справочник бизнес юнитов, добавив фиктивный "не бизнес-юнит" с id=-1, означающий все не БЮ продукты.
     *
     * @return список бизнес юнитов
     */
    List<GdBusinessUnit> getBusinessUnits() {
        Map<Long, String> businessUnits = productService.getBusinessUnits();

        return StreamEx.of(businessUnits.entrySet())
                .map(e -> new GdBusinessUnit()
                        .withBusinessUnitId(e.getKey())
                        .withBusinessUnitName(e.getValue()))
                .toList();
    }

    /**
     * Для внутренней рекламы. Получить какие шаблоны возможно размещать на каком-либо месте.
     *
     * @param input Входные параметры
     * @return список связей место-шаблон
     */
    List<GdInternalTemplatePlace> getInternalTemplatePlaces(GdInternalTemplatePlacesContainer input) {
        List<TemplatePlace> templatePlaceList =
                templatePlaceService.getVisibleTemplatesByPlaceIds(input.getFilter().getPlacesIds());
        return mapList(templatePlaceList, ConstantsConverter::toGdInternalTemplatePlace);
    }

    /**
     * Для внутренней рекламы. Получить информацию по плейсам
     *
     * @param input Входные параметры
     * @return список информации по плейсам
     */
    List<GdInternalAdPlaceInfo> getInternalPlacesInfo(GdInternalAdPlacesInfoContainer input) {
        List<InternalAdPlaceInfo> placesInfo =
                placeService.getPlaceInfoForValidPlaceByIds(input.getFilter().getPlacesIds());
        return mapList(placesInfo, ConstantsConverter::toGdInternalAdPlaceInfo);
    }

    /**
     * Для внутренней рекламы. Получить какие ресурсы нужны шаблону.
     *
     * @param input Входные параметры
     * @return список ресурсов
     */
    List<GdInternalTemplateResource> getInternalTemplateResources(GdInternalTemplateResourcesContainer input) {
        var templateResourceMap =
                templateResourceService.getReadonlyByTemplateIds(input.getFilter().getTemplateIds());
        return mapList(templateResourceMap.values(), ConstantsConverter::toGdInternalTemplateResource);
    }

    /**
     * Для внутренней рекламы получить список площадок
     */
    public List<GdInternalPageInfo> getInternalPagesInfo(User operator) {
        if (!canShowInternalPagesInfo(operator)) {
            return Collections.emptyList();
        }

        List<Page> internalAdPages = pageService.getAllInternalAdPages();
        return StreamEx.of(internalAdPages)
                .map(ConstantsConverter::toGdInternalPageInfo)
                .sortedBy(GdInternalPageInfo::getPageId)
                .toList();
    }

    /**
     * Отдаем площадки внутренней рекламы только тем у кого есть доступ
     */
    private static boolean canShowInternalPagesInfo(User operator) {
        return isInternalAdRole(operator) || hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.SUPERREADER);
    }

    /**
     * Получить максимальную допустимую дату для заказа агентсткого офлайн-отчета
     *
     * @return дата кешированная на 10 минут
     */
    LocalDate getAgencyOfflineReportMaximumAvailableDate() {
        return agencyOfflineReportMaximumAvailableDateCached.get();
    }

    /**
     * Получить максимальную допустимую дату для заказа офлайн-отчета по квартальным KPI
     *
     * @return дата кешированная на 10 минут
     */
    LocalDate getAgencyKpiOfflineReportMaximumAvailableDate() {
        return agencyKpiOfflineReportMaximumAvailableDateCached.get();
    }

    /**
     * Получить минимальную допустимую дату для заказа офлайн-отчета по квартальным KPI
     *
     * @return дата кешированная на 10 минут
     */
    LocalDate getAgencyKpiOfflineReportMinimumAvailableDate() {
        return agencyKpiOfflineReportMinimumAvailableDateCached.get();
    }

    /**
     * Количество специалистов-фрилансеров в системе
     */
    Integer getFreelancersCount() {
        // Это значение как и соседнеене agencyOfflineReportMaximumAvailableDate не очень константно
        // Если появится ещё один такой системный атрибут, стоит задуматься о вынесении системных параметров
        // в отдельный top-level контейнер
        return freelancerDataService.getFreelancersCount();
    }

    Set<String> getAllSupplySidePlatforms() {
        return Set.copyOf(sspPlatformsRepository.getAllSspPlatforms());
    }

    List<GdTimezoneGroup> getTimezoneGroups() {
        Collection<GeoTimezone> geoTimezones = geoTimezoneRepository.getGeoTimezonesByTimezoneIds(emptyList());
        return ConstantsConverter.toGdTimezoneGroups(geoTimezones, translationService.getLocale());
    }

    String getTurbolandingApiUrl() {
        YandexDomain domain = HostUtils.getYandexDomain().orElse(YandexDomain.RU);
        String turbolandingApiUrl;
        if (crowdTestSettingsService.isCrowdTestRequest(domain)) { //если запрос из crowdtest, то подменить url турбы
            turbolandingApiUrl = crowdTestSettingsService.getCrowdTestTurbolandingApiUrl(domain);
        } else {
            turbolandingApiUrl = turbolandingUnifiedApiUrl;
            if (turbolandingApiUrl.endsWith(YandexDomain.RU.getYandexDomain()) && domain != YandexDomain.RU) {
                return turbolandingApiUrl.replace(YandexDomain.RU.getYandexDomain(),
                        domain.getYandexDomain());
            }
        }
        return turbolandingApiUrl;
    }

    String getMetrikaUiUrl() {
        return getMetrikaUrlByPattern(metrikaUiUrlPattern);
    }

    List<GdOsVersions> getOsVersions() {
        return ConstantsConverter.toGdOsVersions();
    }

    List<GdTargetingCategory> getTargetingCategories() {
        List<TargetingCategory> targetingCategories = targetingCategoriesRepository.getAll();
        return mapList(targetingCategories, ConstantsConverter::toGdTargetingCategory);
    }

    List<GdButtonActionDetailed> getButtonActions(ClientId clientId) {
        var clientFeatures = featureService.getEnabledForClientId(clientId);
        return mapList(newBannerWithButtonHelper.getAllowedButtonActions(clientFeatures),
                action -> new GdButtonActionDetailed()
                        .withAction(GdButtonAction.fromSource(action))
                        .withName(newBannerWithButtonHelper.translate(action)));
    }

    /**
     * @param urlPattern шаблон для url, связанного с Метрикой,
     *                   шаблонизированный {@code metrika_name} и {@code yandex_domain}
     *                   (!) Валидации на существование этих полей в шаблоне нет
     * @return url, с подставленными значениями в зависимости от доменной зоны
     */
    private String getMetrikaUrlByPattern(String urlPattern) {
        YandexDomain domain = HostUtils.getYandexDomain().orElse(YandexDomain.RU);
        String metrikaName;
        if (domain == YandexDomain.COM || domain == YandexDomain.TR) {
            metrikaName = "metrica";
        } else {
            metrikaName = "metrika";
        }
        var paramsMap = Map.of(
                "metrika_name", metrikaName,
                "yandex_domain", domain.getYandexDomain());
        var substitutor = new StrSubstitutor(paramsMap, "%(", ")");
        return substitutor.replace(urlPattern);
    }

    /**
     * Дефолтные ret_cond_id шорткатов условий ретаргетинга
     */
    List<Long> getShortcutRetargetingConditionIds() {
        return RetargetingConditionShortcutService.RETARGETING_CONDITION_SHORTCUT_DEFAULT_IDS;
    }

    String getBalanceDomainUrl() {
        YandexDomain domain = HostUtils.getYandexDomain().orElse(YandexDomain.RU);
        String balanceApiUrl;
        if (crowdTestSettingsService.isCrowdTestRequest(domain)) { //если запрос из crowdtest, то подменить url баланса
            balanceApiUrl = crowdTestSettingsService.getCrowdTestBalanceApiUrl(domain);
        } else {
            balanceApiUrl = balanceDomain;
            if (balanceApiUrl.endsWith(YandexDomain.RU.getYandexDomain()) && domain != YandexDomain.RU) {
                return balanceApiUrl.replace(YandexDomain.RU.getYandexDomain(),
                        domain.getYandexDomain());
            }
        }
        return balanceApiUrl;
    }

    List<GdUcRelevanceMatchCategory> getAutotargetingCategories() {
        return StreamEx.of(GdRelevanceMatchCategory.values())
                .map(category -> new GdUcRelevanceMatchCategory()
                        .withCategory(category)
                        .withChecked(true)
                        .withDisabled(DISABLED_RELEVANCE_MATCH_CATEGORIES.contains(
                                RelevanceMatchCategory.fromTypedValue(category.getTypedValue()))))
                .toList();
    }
}
