package ru.yandex.direct.grid.processing.service.constant;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstantsService;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill;
import ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.constants.GdAlertData;
import ru.yandex.direct.grid.processing.model.constants.GdBrandLiftConstants;
import ru.yandex.direct.grid.processing.model.constants.GdBusinessUnit;
import ru.yandex.direct.grid.processing.model.constants.GdBusinessUnitsData;
import ru.yandex.direct.grid.processing.model.constants.GdButtonActionDetailed;
import ru.yandex.direct.grid.processing.model.constants.GdCampaignDefaultValues;
import ru.yandex.direct.grid.processing.model.constants.GdConfigurationConstants;
import ru.yandex.direct.grid.processing.model.constants.GdCurrencyDescription;
import ru.yandex.direct.grid.processing.model.constants.GdDefaultValues;
import ru.yandex.direct.grid.processing.model.constants.GdDirectConstants;
import ru.yandex.direct.grid.processing.model.constants.GdDirectConstantsCampaignValidationData;
import ru.yandex.direct.grid.processing.model.constants.GdDirectConstantsValidationData;
import ru.yandex.direct.grid.processing.model.constants.GdFreelancerSkillType;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlaceInfo;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlacesInfoContainer;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlacesInfoData;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlace;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlacesContainer;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlacesData;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResource;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResourcesContainer;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResourcesData;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStation;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStationsContainer;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStationsData;
import ru.yandex.direct.grid.processing.model.constants.GdOsVersions;
import ru.yandex.direct.grid.processing.model.constants.GdStatLimits;
import ru.yandex.direct.grid.processing.model.constants.GdStrategyConstants;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictData;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictName;
import ru.yandex.direct.grid.processing.model.constants.GdTargetingCategory;
import ru.yandex.direct.grid.processing.model.constants.GdTimezoneGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUcRelevanceMatchCategory;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalPageInfo;
import ru.yandex.direct.grid.processing.model.product.GdProductPageGroupTags;
import ru.yandex.direct.grid.processing.model.product.GdProductTargetTags;
import ru.yandex.direct.grid.processing.model.strategy.defaults.GdPackageStrategiesDefaults;
import ru.yandex.direct.grid.processing.service.client.converter.BusinessCategoriesService;
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider;
import ru.yandex.direct.grid.processing.service.freelancer.FreelancerConverter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.mediascope.MediascopeClientConfig;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BRAND_LIFT_MIN_DAYS_CAMPAIGN_INTERVAL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.BY_ALL_GOALS_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.COUNT_AD_SHOWS_FOR_PRICE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_DAYS_STRATEGY_INTERVAL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_CARD_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DAYS_STRATEGY_INTERVAL;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DAY_BUDGET_DAILY_CHANGE_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_SUFFICIENT_GOAL_CONVERSION_COUNT;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.PAY_FOR_CONVERSION_SUM_TO_AVG_CPA_MIN_RATIO;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill.allFreelancerSkills;
import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.NEW_CARD_PAY_LIMIT;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.getCampaignDefaultValues;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getAdGroupValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getCalloutValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getCampaignValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getCashbackValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getKeywordsValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getMinusWordsLibValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getMinusWordsValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getPromoExtensionValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getSitelinkValidationData;
import static ru.yandex.direct.grid.processing.service.constant.ValidationConstantsService.getVcardValidationData;
import static ru.yandex.direct.grid.processing.service.goal.GoalConstant.MAX_GOAL_COUNT_FOR_FAST_STATISTIC_REQUEST;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.StringUtils.parseLongsFromString;


/**
 * Сервис, поставляющий данные о константных значениях Директа
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ConstantGraphQlService {  //TODO добавлять новые сервисы лучше в ConstantDataService

    static final String INTERNAL_PAGES_INFO_RESOLVER = "internalPagesInfo";

    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final FreelancerConverter freelancerConverter;
    private final String canvasUiDomain;
    private final String spravIFrameUrl;
    private final String surveysUiUrlPattern;
    private final String mediascopePartnerId;
    private final String infoblockTeasersUrl;
    private final String serverPushesHost;
    private final ConstantDataService constantDataService;
    private final ConstantStringDictsService constantStringDictsService;
    private final FeatureService featureService;
    private final BusinessCategoriesService businessCategoriesService;
    private final GridContextProvider gridContextProvider;
    private final ValidationConstantsService validationConstantsService;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final CampaignConstantsService campaignConstantsService;

    @SuppressWarnings("checkstyle:ParameterNumber")
    @Autowired
    public ConstantGraphQlService(
            @Value("${canvas.ui_domain}") String canvasUiDomain,
            @Value("${organizations-iframe.url}") String spravIFrameUrl,
            @Value("${brand_survey.ui_url_pattern}") String surveysUiUrlPattern,
            @Value("${common_links.infoblock_teasers_url}") String infoblockTeasersUrl,
            @Value("${xiva_client.xiva.host}") String serverPushesHost,
            MediascopeClientConfig mediascopeClientConfig,
            PpcPropertiesSupport ppcPropertiesSupport,
            FreelancerConverter freelancerConverter,
            ConstantDataService constantDataService,
            ConstantStringDictsService constantStringDictsService,
            FeatureService featureService, BusinessCategoriesService businessCategoriesService,
            GridContextProvider gridContextProvider, ValidationConstantsService validationConstantsService,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            CampaignConstantsService campaignConstantsService) {
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.freelancerConverter = freelancerConverter;
        this.canvasUiDomain = canvasUiDomain;
        this.spravIFrameUrl = spravIFrameUrl;
        this.surveysUiUrlPattern = surveysUiUrlPattern;
        this.constantDataService = constantDataService;
        this.constantStringDictsService = constantStringDictsService;
        this.featureService = featureService;
        this.mediascopePartnerId = mediascopeClientConfig.getClientId();
        this.infoblockTeasersUrl = infoblockTeasersUrl;
        this.serverPushesHost = serverPushesHost;
        this.businessCategoriesService = businessCategoriesService;
        this.gridContextProvider = gridContextProvider;
        this.validationConstantsService = validationConstantsService;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.campaignConstantsService = campaignConstantsService;
    }

    /**
     * GraphQL запрос на получение констант директа
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "constants")
    public GdDirectConstants getDirectConstantsFetcher() {
        GdStrategyConstants strategyConstants = new GdStrategyConstants()
                .withMaxDailyBudgetChangesPerDay(MAX_DAY_BUDGET_DAILY_CHANGE_COUNT)
                .withByAllGoalsGoalId(BY_ALL_GOALS_GOAL_ID)
                .withMeaningfulGoalsOptimizationGoalId(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
                .withEngagedSessionGoalId(ENGAGED_SESSION_GOAL_ID)
                .withMinSufficientGoalConversionCount(MIN_SUFFICIENT_GOAL_CONVERSION_COUNT)
                .withPayForConversionSumToAvgCpaMinRatio(PAY_FOR_CONVERSION_SUM_TO_AVG_CPA_MIN_RATIO)
                .withPayForConversionAvgCpaWarningRatioDefaultValue(PAY_FOR_CONVERSION_AVG_CPA_WARNING_RATIO_DEFAULT_VALUE)
                .withMaxDaysStrategyInterval(MAX_DAYS_STRATEGY_INTERVAL)
                .withDefaultDaysStrategyInterval(DEFAULT_DAYS_STRATEGY_INTERVAL)
                .withCountAdShowsForPrice(COUNT_AD_SHOWS_FOR_PRICE);
        GdBrandLiftConstants brandLiftConstants = new GdBrandLiftConstants()
                .withMinDaysCampaignBrandLiftInterval(BRAND_LIFT_MIN_DAYS_CAMPAIGN_INTERVAL);
        return new GdDirectConstants()
                .withStrategyConstants(strategyConstants)
                .withBrandLiftConstants(brandLiftConstants);
    }

    /**
     * GraphQL подзапрос на получение валютных констант директа
     *
     * @param directConstants Родительский объект запроса
     * @param codes           коды валют, по которым нужно получить константы, если список не передан или пуст,
     *                        константы отдаются по всем валютам
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "currencyConstants")
    public List<@GraphQLNonNull GdCurrencyDescription> getCurrencyConstants(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLArgument(name = "codes") @Nullable List<@GraphQLNonNull CurrencyCode> codes) {
        EntryStream<String, Currency> currencyEntryStream = EntryStream.of(Currencies.getCurrencies());
        if (codes != null && !codes.isEmpty()) {
            Set<String> allowedCodes = codes.stream()
                    .map(Enum::name)
                    .collect(Collectors.toSet());
            currencyEntryStream = currencyEntryStream.filterKeys(allowedCodes::contains);
        }
        ClientId clientId = context.getSubjectUser().getClientId();
        Set<String> enabledFeaturesForClient = featureService.getEnabledForClientId(clientId);
        boolean useNewAutopayCardLimit = ppcPropertiesSupport.find(NEW_CARD_PAY_LIMIT.getName())
                .orElse("0").equals("1");
        var defaultCpmFrontpagePrice = cpmYndxFrontpageCurrencyService.getDefaultCpmFrontpagePrice();
        return currencyEntryStream
                .mapKeyValue((code, currency) ->
                        ConstantsConverter.currencyDescription(enabledFeaturesForClient,
                                useNewAutopayCardLimit, code, currency,
                                defaultCpmFrontpagePrice))
                .toList();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "defaultValues")
    public GdDefaultValues defaultValues(
            @GraphQLContext GdDirectConstants directConstants) {
        return new GdDefaultValues();
    }

    // TODO(dimitrovsd): удалить резолвер в пользу резолвера campaigns
    @GraphQLNonNull
    @GraphQLQuery(name = "campaign")
    public GdCampaignDefaultValues campaignDefaultValues(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDefaultValues defaultValues) {
        return getCampaignDefaultValues(
                featureService.getEnabledForClientId(context.getOperator().getClientId()),
                featureService.getEnabledForClientId(context.getSubjectUser().getClientId()),
                context.getQueriedClient().getIsProStategyViewEnabled(),
                campaignConstantsService.getDefaultAttributionModel()
        );
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "campaigns")
    public List<GdCampaignDefaultValues> campaignsDefaultValues(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDefaultValues defaultValues,
            @GraphQLArgument(name = "campaignTypes") @Nullable List<@GraphQLNonNull GdCampaignType> campaignTypes) {
        // TODO: будет лучше перенести зависящие от клиента данные в клиентские константы (DIRECT-133820)
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        checkNotNull(context.getQueriedClient(), "must be called in client context");

        return getCampaignDefaultValues(campaignTypes,
                featureService.getEnabledForClientId(context.getOperator().getClientId()),
                featureService.getEnabledForClientId(context.getSubjectUser().getClientId()),
                context.getQueriedClient().getIsProStategyViewEnabled(),
                campaignConstantsService.getDefaultAttributionModel());
    }

    /**
     * GraphQL запрос на получение полного списка услуг которые могут предоставляться фрилансерами.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "freelancerSkillTypes")
    public List<@GraphQLNonNull GdFreelancerSkillType> freelancerSkillTypes(
            @GraphQLContext GdDirectConstants directConstants) {
        List<FreelancerSkill> allSkills = allFreelancerSkills();
        return freelancerConverter.convertFreelancerSkillListToGd(allSkills);
    }

    /**
     * GraphQL запрос на получение полного списка категорий клиентского бизнеса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "businessCategories")
    public List<@GraphQLNonNull String> businessCategories(
            @GraphQLContext GdDirectConstants directConstants) {
        return businessCategoriesService.getAllDirectBusinessCategories();
    }

    /**
     * GraphQL запрос на получение списка параметров конфигурации.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "configuration")
    public GdConfigurationConstants configuration(@GraphQLContext GdDirectConstants directConstants) {
        return new GdConfigurationConstants()
                .withTurbolandingUnifiedApiUrl(constantDataService.getTurbolandingApiUrl())
                .withMetrikaUiUrl(constantDataService.getMetrikaUiUrl())
                .withCanvasUiDomain(canvasUiDomain)
                .withBalanceDomain(constantDataService.getBalanceDomainUrl())
                .withSpravIFrameUrl(spravIFrameUrl)
                .withSurveysUiUrl(String.format(surveysUiUrlPattern,
                        HostUtils.getYandexDomain().orElse(YandexDomain.RU)))
                .withMediascopePartnerId(mediascopePartnerId)
                .withInfoblockTeasersUrl(infoblockTeasersUrl)
                .withServerPushesHost(serverPushesHost);
    }

    /**
     * GraphQL подзапрос на получение данных для отображения аварийной плашки
     * Если плашку отображать не надо, то тексты не отдаются, чтобы не смущать продвинутых пользователей
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "alertData")
    public GdAlertData alertData(@GraphQLContext GdDirectConstants directConstants) {

        Map<String, String> ppcPropertiesData = ppcPropertiesSupport.getByNames(Arrays.asList(
                PpcPropertyEnum.NEW_INTERFACE_ALERT_ENABLED.getName(),
                PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_RU.getName(),
                PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_EN.getName(),
                PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_CLIENT_IDS.getName()
        ));
        GdAlertData alertData = new GdAlertData()
                .withEnabled(nvl(ppcPropertiesData.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_ENABLED.getName()),
                        YesNo.NO.name()).equals(YesNo.YES.name()));
        if (alertData.getEnabled()) {
            var clientId = gridContextProvider.getGridContext().getSubjectUser().getClientId().asLong();
            var clientIds = ppcPropertiesData.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_CLIENT_IDS.getName());
            if (clientIds == null || parseLongsFromString(clientIds).contains(clientId)) {
                alertData
                        .withTextRu(ppcPropertiesData.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_RU.getName()))
                        .withTextEn(ppcPropertiesData.get(PpcPropertyEnum.NEW_INTERFACE_ALERT_TEXT_EN.getName()));
            } else {
                alertData.setEnabled(false);
            }
        }
        return alertData;
    }

    /**
     * GraphQL запрос на получение констант необходимых для фронтовой валидации объектов
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "validation")
    public GdDirectConstantsValidationData getDirectValidationConstants(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstants directConstants) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");

        ClientId clientId = context.getSubjectUser().getClientId();

        return new GdDirectConstantsValidationData()
                .withCampaignConstants(getCampaignValidationData(featureService.getEnabledForClientId(clientId)))
                .withAdGroupConstants(getAdGroupValidationData())
                .withAdConstants(validationConstantsService.getAdValidationData(clientId))
                .withVcardConstants(getVcardValidationData())
                .withCalloutConstants(getCalloutValidationData())
                .withSitelinkConstants(getSitelinkValidationData())
                .withLibMinusKeywordsConstants(getMinusWordsLibValidationData())
                .withMinusKeywordsConstants(getMinusWordsValidationData())
                .withKeywordsConstants(getKeywordsValidationData())
                .withCashbackConstants(getCashbackValidationData())
                .withPromoExtensionConstants(getPromoExtensionValidationData());
    }

    /**
     * GraphQL запрос на получение названий всех используемых площадок необходимых для фронтовой валидации объектов
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "supplySidePlatforms")
    public Set<@GraphQLNonNull String> getAllSupplySidePlatforms(
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstantsCampaignValidationData campaignConstants) {
        return constantDataService.getAllSupplySidePlatforms();
    }

    /**
     * GraphQL запрос на получение списка станций метрополитена
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "metroStations",
            description = "Станции метрополитена (только для Москвы, Санкт-Петербурга и Киева)")
    public GdMetroStationsData getMetroStations(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetroStationsContainer input) {
        List<GdMetroStation> metroStations =
                constantDataService.getMetroStations(context.getOperator().getClientId(), input);

        return new GdMetroStationsData()
                .withRowset(metroStations);
    }

    /**
     * GraphQL запрос на получение списка бизнес-юнитов
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "businessUnits",
            description = "Бизнес юниты для прайсовых пакетов")
    public GdBusinessUnitsData getBusinessUnits(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDirectConstants directConstants) {
        List<GdBusinessUnit> businessUnits = constantDataService.getBusinessUnits();

        return new GdBusinessUnitsData().withRowset(businessUnits);
    }

    /**
     * GraphQL запрос на получение набора связей места с шаблонами для внутренней рекламы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "internalTemplatePlaces",
            description = "Набор связей места с шаблонами для внутренней рекламы")
    @SuppressWarnings("unused")
    public GdInternalTemplatePlacesData getInternalTemplatePlaces(
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdInternalTemplatePlacesContainer input) {
        List<GdInternalTemplatePlace> templatePlaces = constantDataService.getInternalTemplatePlaces(input);
        return new GdInternalTemplatePlacesData().withRowset(templatePlaces);
    }

    /**
     * GraphQL запрос на получение информации по плейсам внутренней рекламы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "internalAdPlacesInfo",
            description = "Набор информации по плейсам внутренней рекламы")
    @SuppressWarnings("unused")
    public GdInternalAdPlacesInfoData getInternalAdPlacesInfo(
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdInternalAdPlacesInfoContainer input) {
        List<GdInternalAdPlaceInfo> placesInfo = constantDataService.getInternalPlacesInfo(input);
        return new GdInternalAdPlacesInfoData().withRowset(placesInfo);
    }

    /**
     * GraphQL запрос на получение набора ресурсов для шаблонов внутренней рекламы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "internalTemplateResources",
            description = "Набор ресурсов для шаблонов внутренней рекламы")
    @SuppressWarnings("unused")
    public GdInternalTemplateResourcesData getInternalTemplateResources(
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdInternalTemplateResourcesContainer input) {
        List<GdInternalTemplateResource> templateResources = constantDataService.getInternalTemplateResources(input);
        return new GdInternalTemplateResourcesData().withRowset(templateResources);
    }

    /**
     * GraphQL запрос на получение площадок внутренней рекламы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = INTERNAL_PAGES_INFO_RESOLVER, description = "Площадки внутренней рекламы")
    public List<@GraphQLNonNull GdInternalPageInfo> getInternalPagesInfo(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getInternalPagesInfo(context.getOperator());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "stringDicts",
            description = "Строковые словари")
    @SuppressWarnings("unused")
    public List<GdStringDictData> getStringDicts(
            @GraphQLContext GdDirectConstants directConstants,
            @GraphQLNonNull @GraphQLArgument(name = "dictNames") List<GdStringDictName> dictNames
    ) {
        return constantStringDictsService.get(dictNames);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "timezoneGroups", description = "Доступные группы таймзон")
    @SuppressWarnings("unused")
    public List<@GraphQLNonNull GdTimezoneGroup> getTimezoneGroups(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getTimezoneGroups();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "agencyOfflineReportMaximumDate",
            description = "Максимально допустимая конечная дата для заказа агентсткого офлайн-отчета")
    @SuppressWarnings("unused")
    public LocalDate getAgencyOfflineReportMaximumAvailableDate(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getAgencyOfflineReportMaximumAvailableDate();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "agencyKpiOfflineReportMaximumDate",
            description = "Максимально допустимая конечная дата для заказа офлайн-отчета по квартальным KPI")
    @SuppressWarnings("unused")
    public LocalDate getAgencyKpiOfflineReportMaximumAvailableDate(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getAgencyKpiOfflineReportMaximumAvailableDate();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "agencyKpiOfflineReportMinimumDate",
            description = "Минимально допустимая начальная дата для заказа офлайн-отчета по квартальным KPI")
    @SuppressWarnings("unused")
    public LocalDate getAgencyKpiOfflineReportMinimumAvailableDate(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getAgencyKpiOfflineReportMinimumAvailableDate();
    }

    @GraphQLQuery(name = "freelancersCount",
            description = "Количество частных специалистов по настройке Директа")
    public Integer getFreelancersCount(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getFreelancersCount();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "osVersions",
            description = "Доступные версии ОС")
    public List<@GraphQLNonNull GdOsVersions> getOsVersions(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getOsVersions();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "targetingCategories",
            description = "Список существующих в словаре категорий интересов")
    public List<@GraphQLNonNull GdTargetingCategory> getTargetingCategories(@GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getTargetingCategories();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "bsPageGroupTags",
            description = "теги дополнительных площадок для группы")
    public List<GdProductPageGroupTags> getBsPageGroupTags(@GraphQLContext GdDirectConstants directConstants) {
        return Arrays.asList(GdProductPageGroupTags.values());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "bsTargetTags",
            description = "теги ограничения площадок для группы")
    public List<GdProductTargetTags> getBsTargetTags(@GraphQLContext GdDirectConstants directConstants) {
        return Arrays.asList(GdProductTargetTags.values());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "gdStatLimits",
            description = "граничные значения для запроса статистики")
    public GdStatLimits getGdStatLimits(@GraphQLContext GdDirectConstants directConstants) {
        return new GdStatLimits()
                .withMaxFromDays(GridValidationService.MAX_FROM_STAT_PERIOD.toDays())
                .withMaxToDays(GridValidationService.MAX_TO_STAT_PERIOD.toDays());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "buttonActions",
            description = "возможные значения действий на кнопке")
    public List<GdButtonActionDetailed> getButtonActions(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstants directConstants) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        return constantDataService.getButtonActions(context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "maxGoalCountForFastStatisticRequest")
    public Long getMaxGoalCountForFastStatisticRequest(
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstants directConstants) {
        return MAX_GOAL_COUNT_FOR_FAST_STATISTIC_REQUEST;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "shortcutRetargetingConditionIds",
            description = "Дефолтные ret_cond_id шорткатов, которые нужно пересоздать при первом использовании")
    @SuppressWarnings("unused")
    public List<Long> getRetargetingConditionShortcutIds(
            @SuppressWarnings("unused") @GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getShortcutRetargetingConditionIds();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "maxCardCount",
            description = "Максимальное количество привязанных карточек (ограничение баланса)")
    public Integer getMaxCardCount(@GraphQLContext GdDirectConstants directConstants) {
        return MAX_CARD_COUNT;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "autotargetingCategories",
            description = "Категории автотаргетинга")
    public List<GdUcRelevanceMatchCategory> getAutotargetingCategories(
            @GraphQLContext GdDirectConstants directConstants) {
        return constantDataService.getAutotargetingCategories();
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "packageStrategies")
    public GdPackageStrategiesDefaults packageStrategiesDefaults(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdDefaultValues defaultValues) {
        checkNotNull(context.getSubjectUser(), "expecting non null subjectUser");
        checkNotNull(context.getQueriedClient(), "must be called in client context");

        return PackageStrategyDefaultValuesUtils.getPackageStrategiesDefaults();
    }

}
