package ru.yandex.direct.grid.processing.service.constant;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroupadditionaltargeting.DistribSoftConstants;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.InterfaceLangConstants;
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.UaTraitsConstants;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictBaseEntry;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictData;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictEntry;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictEntryWithLongKey;
import ru.yandex.direct.grid.processing.model.constants.GdStringDictName;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Map.entry;

@Service
@ParametersAreNonnullByDefault
public class ConstantStringDictsService {
    private static final Map<GdStringDictName, Map<Long, String>> DICTS = Map.ofEntries(
            entry(GdStringDictName.UA_TRAITS_BROWSER_BASE, UaTraitsConstants.BROWSER_BASE),
            entry(GdStringDictName.UA_TRAITS_BROWSER_ENGINE, UaTraitsConstants.BROWSER_ENGINE),
            entry(GdStringDictName.UA_TRAITS_BROWSER_NAME, UaTraitsConstants.BROWSER_NAME),
            entry(GdStringDictName.UA_TRAITS_DEVICE_VENDOR, UaTraitsConstants.DEVICE_VENDOR),
            entry(GdStringDictName.UA_TRAITS_OS_FAMILY, UaTraitsConstants.OS_FAMILY),
            entry(GdStringDictName.UA_TRAITS_OS_NAME, UaTraitsConstants.OS_NAME),
            entry(GdStringDictName.DISTRIB_SOFT, DistribSoftConstants.DISTRIB_SOFT)
    );

    private static final Map<GdStringDictName, Set<String>> STRING_DICTS = Map.ofEntries(
            entry(GdStringDictName.INTERFACE_LANG, InterfaceLangConstants.INTERFACE_LANG)
    );

    private static GdStringDictData convert(GdStringDictName dictName, Map<Long, String> dicts) {
        List<GdStringDictBaseEntry> entries = EntryStream.of(dicts)
                .mapKeyValue((key, value) -> new GdStringDictEntryWithLongKey().withKey(key).withValue(value))
                .map(GdStringDictBaseEntry.class::cast)
                .toImmutableList();
        return new GdStringDictData()
                .withDictName(dictName)
                .withEntries(entries);
    }

    private static GdStringDictData convert(GdStringDictName dictName, Set<String> stringDicts) {
        List<GdStringDictBaseEntry> entries = stringDicts.stream()
                .map(value -> new GdStringDictEntry().withValue(value))
                .map(GdStringDictBaseEntry.class::cast)
                .collect(Collectors.toList());
        return new GdStringDictData()
                .withDictName(dictName)
                .withEntries(entries);
    }

    public List<GdStringDictData> get(List<GdStringDictName> dictNames) {
        checkArgument(Sets.difference(new HashSet<>(dictNames), Sets.union(DICTS.keySet(), STRING_DICTS.keySet())).isEmpty(),
                "Argument 'dictName' has unknown value");

        return ListUtils.union(
                StreamEx.of(dictNames)
                        .mapToEntry(DICTS::get)
                        .nonNullValues()
                        .mapKeyValue(ConstantStringDictsService::convert)
                        .toImmutableList(),
                StreamEx.of(dictNames)
                        .mapToEntry(STRING_DICTS::get)
                        .nonNullValues()
                        .mapKeyValue(ConstantStringDictsService::convert)
                        .toImmutableList());
    }
}
