package ru.yandex.direct.grid.processing.service.constant;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.internalads.model.InternalAdPlaceInfo;
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource;
import ru.yandex.direct.core.entity.internalads.model.TemplatePlace;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory;
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone;
import ru.yandex.direct.core.entity.timetarget.model.GroupType;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.grid.processing.model.constants.GdCurrencyDescription;
import ru.yandex.direct.grid.processing.model.constants.GdInternalAdPlaceInfo;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplatePlace;
import ru.yandex.direct.grid.processing.model.constants.GdInternalTemplateResource;
import ru.yandex.direct.grid.processing.model.constants.GdMetroCity;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStation;
import ru.yandex.direct.grid.processing.model.constants.GdMetroStationsFilter;
import ru.yandex.direct.grid.processing.model.constants.GdOsVersions;
import ru.yandex.direct.grid.processing.model.constants.GdTargetingCategory;
import ru.yandex.direct.grid.processing.model.constants.GdTimezone;
import ru.yandex.direct.grid.processing.model.constants.GdTimezoneGroup;
import ru.yandex.direct.grid.processing.model.internalad.GdInternalPageInfo;
import ru.yandex.direct.regions.Metro;

import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignWithStrategyValidationUtils.getAutobudgetPayForConversionAvgCpaWarning;
import static ru.yandex.direct.core.validation.constraints.MobileContentConstraints.OS_VERSIONS;
import static ru.yandex.direct.currency.MoneyUtils.getMaxAutopayCard;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.MIN_RECOMMENDED_CPV_AUTOBUDGET_MULTIPLICAND;
import static ru.yandex.direct.grid.processing.service.constant.DefaultValuesUtils.MIN_RECOMMENDED_CPV_PERIOD;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class ConstantsConverter {
    private static final BigDecimal THREE = BigDecimal.valueOf(3);
    private static final BigDecimal SIX = BigDecimal.valueOf(6);

    private ConstantsConverter() {
    }

    public static GdCurrencyDescription currencyDescription(Set<String> enabledFeatures,
                                                            boolean useNewAutopayCardLimit,
                                                            String cur, Currency desc,
                                                            Map<CurrencyCode, BigDecimal> defaultCpmFrontpagePriceMap) {
        return new GdCurrencyDescription()
                .withCode(CurrencyCode.valueOf(cur))
                .withMinDailyBudget(desc.getMinDayBudget())
                .withMaxDailyBudget(desc.getMaxDailyBudgetAmount())
                .withAuctionStep(desc.getAuctionStep())
                .withAutobudgetAvgCpaWarning(desc.getAutobudgetAvgCpaWarning())
                .withAutobudgetPayForConversionAvgCpaWarning(
                        getAutobudgetPayForConversionAvgCpaWarning(enabledFeatures, desc))
                .withAutobudgetAvgPriceWarning(desc.getAutobudgetAvgPriceWarning())
                .withAutobudgetMaxPriceWarning(desc.getAutobudgetMaxPriceWarning())
                .withAutobudgetSumWarning(desc.getAutobudgetSumWarning())
                .withBigRate(desc.getBigRate())
                .withDefaultAutobudget(desc.getDefaultAutobudget())
                .withDefaultPrice(desc.getDefaultPrice())
                .withDefaultAvgCpv(desc.getDefaultAvgCpv())
                .withDefaultCpmPrice(desc.getDefaultCpmPrice())
                .withDefaultCpmFrontpagePrice(
                        defaultCpmFrontpagePriceMap.get(desc.getCode()))
                .withDirectDefaultPay(desc.getDirectDefaultPay())
                .withMaxAutobudget(desc.getMaxAutobudget())
                .withMaxAutobudgetBid(desc.getMaxAutobudgetBid())
                .withMaxAutopayCard(getMaxAutopayCard(desc, useNewAutopayCardLimit))
                .withMaxAutopayRemaining(desc.getMaxAutopayRemaining())
                .withMaxAutopayYamoney(desc.getMaxAutopayYamoney())
                .withMaxClientArchive(desc.getMaxClientArchive())
                .withMiddleCpmPrice(desc.getMaxCpmPrice().divide(BigDecimal.TEN, RoundingMode.HALF_UP))
                .withMiddleCpmFrontpagePrice(desc.getMaxCpmPrice().divide(SIX, RoundingMode.HALF_UP))
                .withMiddleAvgCpv(desc.getMiddleAvgCpv())
                .withMaxAvgCpv(desc.getMaxAvgCpv())
                .withMaxCpmPrice(desc.getMaxCpmPrice())
                .withMaxCpmFrontpagePrice(desc.getMaxCpmPrice())
                .withMaxDailyBudgetForPeriod(desc.getMaxDailyBudgetForPeriod())
                .withMaxPrice(desc.getMaxPrice())
                .withMaxShowBid(desc.getMaxShowBid())
                .withMaxTopaySuggest(desc.getMaxTopaySuggest())
                .withMinRecommendedCpvPeriod(MIN_RECOMMENDED_CPV_PERIOD)
                .withMinRecommendedCpvAutobudget(desc.getMinAutobudget().multiply(
                        MIN_RECOMMENDED_CPV_AUTOBUDGET_MULTIPLICAND))
                .withMinAutobudget(desc.getMinAutobudget())
                .withMinAutobudgetForEcom(desc.getMinAutobudget().multiply(THREE))
                .withMinAutobudgetAvgCpa(desc.getMinAutobudgetAvgCpa())
                .withMinAutobudgetAvgCpm(desc.getMinAutobudgetAvgCpm())
                .withMinAutobudgetAvgPrice(desc.getMinAutobudgetAvgPrice())
                .withMinAutobudgetBid(desc.getMinAutobudgetBid())
                .withMinAutopay(desc.getMinAutopay())
                .withMinCpcCpaPerformance(desc.getMinCpcCpaPerformance())
                .withMinAvgCpv(desc.getMinAvgCpv())
                .withMinCpmPrice(desc.getMinCpmPrice())
                .withMinCpmFrontpagePrice(desc.getMinCpmFrontpagePrice())
                .withMinDailyBudgetForPeriod(desc.getMinDailyBudgetForPeriod())
                .withMinImagePrice(desc.getMinImagePrice())
                .withMinPay(desc.getMinPay())
                .withMinPrice(desc.getMinPrice())
                .withMinPriceForMfa(desc.getMinPriceForMfa())
                .withMinSumInterpreteAsPayment(desc.getMinSumInterpreteAsPayment())
                .withMinTransferMoney(desc.getMinTransferMoney())
                .withMinWalletDayBudget(desc.getMinWalletDayBudget())
                .withMoneymeterMaxMiddleSum(desc.getMoneymeterMaxMiddleSum())
                .withMoneymeterMiddlePriceMin(desc.getMoneymeterMiddlePriceMin())
                .withMoneymeterTypicalMiddleSumIntervalBegin(desc.getMoneymeterTypicalMiddleSumIntervalBegin())
                .withMoneymeterTypicalMiddleSumIntervalEnd(desc.getMoneymeterTypicalMiddleSumIntervalEnd())
                .withRecommendedSumToPay(desc.getRecommendedSumToPay())
                .withMinAutobudgetClicksBundle(desc.getMinAutobudgetClicksBundle())
                .withAutobudgetClicksBundleWarning(desc.getAutobudgetClicksBundleWarning())
                .withIsoNumCode(desc.getIsoNumCode())
                .withPayForConversionMinReservedSumDefaultValue(desc.getPayForConversionMinReservedSumDefaultValue())
                .withMaxAutobudgetClicksBundle(desc.getMaxAutobudgetClicksBundle())
                .withPrecisionDigitCount(desc.getPrecisionDigitCount())
                .withBrandSurveyBudgetThreshold(desc.getBrandSurveyBudgetThreshold())
                .withBrandSurveyBudgetThresholdDaily(desc.getBrandSurveyBudgetThresholdDaily())
                .withMonetOutLimit(desc.getMoneyOutLimit())
                .withTouchWeekBudgetSumFirst(desc.getTouchWeekBudgetSumFirst())
                .withTouchWeekBudgetSumSecond(desc.getTouchWeekBudgetSumSecond())
                .withTouchWeekBudgetSumThird(desc.getTouchWeekBudgetSumThird())
                .withRecommendationSumMin(desc.getRecommendationSumMin())
                .withRecommendationSumMid(desc.getRecommendationSumMid())
                .withRecommendationSumMax(desc.getRecommendationSumMax())
                .withRecommendationSumWarn(desc.getRecommendationSumWarn())
                .withPayWithCashLimit(desc.getPayWithCashLimit());
    }

    static Predicate<Metro> toMetroPredicate(GdMetroStationsFilter filter) {
        Set<Long> metroGeoRegionIdIn =
                ifNotNull(filter.getMetroCityIn(), metroCityIn -> mapSet(metroCityIn, GdMetroCity::getRegionId));

        return metroGeoRegionIdIn == null
                ? metro -> true
                : metro -> metroGeoRegionIdIn.contains(metro.getParent().getId());
    }

    static GdMetroStation toGdMetroStation(Metro metro) {
        return new GdMetroStation()
                .withGeoRegionId(Math.toIntExact(metro.getParent().getId()))
                .withMetroStationId(Math.toIntExact(metro.getId()))
                .withMetroStationName(metro.getName());
    }

    static GdInternalTemplatePlace toGdInternalTemplatePlace(TemplatePlace templatePlace) {
        return new GdInternalTemplatePlace()
                .withPlaceId(templatePlace.getPlaceId())
                .withTemplateId(templatePlace.getTemplateId());
    }

    static GdInternalAdPlaceInfo toGdInternalAdPlaceInfo(InternalAdPlaceInfo placeInfo) {
        return new GdInternalAdPlaceInfo()
                .withPlaceId(placeInfo.getId())
                .withFullDescription(placeInfo.getFullDescription())
                .withIsModerated(placeInfo.isModerated());
    }

    static GdInternalTemplateResource toGdInternalTemplateResource(ReadOnlyDirectTemplateResource templateResource) {
        return new GdInternalTemplateResource()
                .withId(templateResource.getId())
                .withTemplateId(templateResource.getTemplateId())
                .withDescription(templateResource.getDescription())
                .withResourceType(templateResource.getResourceType())
                .withPosition(templateResource.getPosition())
                .withOptionsRequired(templateResource.isRequired())
                .withOptionsBananaUrl(templateResource.isBananaUrl())
                .withOptionsBananaImage(templateResource.isBananaImage());
    }

    static List<GdTimezoneGroup> toGdTimezoneGroups(Collection<GeoTimezone> geoTimezones, Locale locale) {
        return StreamEx.of(geoTimezones)
                .sorted(Comparator.comparing(GeoTimezone::getGroupType))
                .mapToEntry(GeoTimezone::getGroupType, geoTimezone -> toGdTimezone(geoTimezone, locale))
                .collapseKeys()
                .mapKeyValue(ConstantsConverter::toGdTimezoneGroup)
                .toList();
    }

    private static GdTimezoneGroup toGdTimezoneGroup(GroupType type, List<GdTimezone> timezones) {
        return new GdTimezoneGroup()
                .withGroupNick(type.getTypedValue())
                .withTimezones(sortTimezones(type, timezones));
    }

    static GdTargetingCategory toGdTargetingCategory(TargetingCategory targetingCategory) {
        return new GdTargetingCategory()
                .withTargetingCategoryId(targetingCategory.getTargetingCategoryId())
                .withParentId(targetingCategory.getParentId())
                .withName(targetingCategory.getName())
                .withOriginalName(targetingCategory.getOriginalName())
                .withImportId(targetingCategory.getImportId())
                .withAvailable(targetingCategory.isAvailable());
    }

    /**
     * Сортирует таймзоны по сдвигам для России и по именам для всех остальных
     */
    private static List<GdTimezone> sortTimezones(GroupType type, List<GdTimezone> timezones) {
        if (type == GroupType.RUSSIA) {
            timezones.sort(Comparator.comparingInt(GdTimezone::getOffsetSeconds));
        } else {
            timezones.sort(Comparator.comparing(GdTimezone::getName));
        }
        return timezones;
    }

    private static GdTimezone toGdTimezone(GeoTimezone timezone, Locale locale) {
        Instant now = Instant.now();
        ZoneId mskZone = ZoneId.of("Europe/Moscow");
        ZoneId zone = timezone.getTimezone();

        int offsetSeconds = zone.getRules().getOffset(now).getTotalSeconds();
        int mskOffsetSeconds = mskZone.getRules().getOffset(now).getTotalSeconds();

        String mskOffset = formatOffset(offsetSeconds, mskOffsetSeconds);
        String gmtOffset = formatOffset(offsetSeconds, ZoneId.of("Z").getRules().getOffset(now).getTotalSeconds());
        String offsetStr = getOffsetString(timezone.getGroupType(), mskOffset, gmtOffset, mskZone.equals(zone));

        return new GdTimezone()
                .withId(timezone.getTimezoneId())
                .withTimezone(timezone.getTimezone().getId())
                .withOffsetSeconds(offsetSeconds)
                .withMskOffset(mskOffset)
                .withGmtOffset(gmtOffset)
                .withOffsetStr(offsetStr)
                .withName(getName(locale, timezone, offsetStr));
    }

    static List<GdOsVersions> toGdOsVersions() {
        List<GdOsVersions> gdOsVersions = new ArrayList<>();
        for (Map.Entry<OsType, Set<String>> operationSystemEntry : OS_VERSIONS.entrySet()) {
            for (String version : operationSystemEntry.getValue()) {
                @SuppressWarnings("ConstantConditions")
                GdOsVersions item = new GdOsVersions()
                        .withOsType(OsType.toSource(operationSystemEntry.getKey()).toString())
                        .withVersions(version);
                gdOsVersions.add(item);
            }
        }
        return gdOsVersions;
    }

    /**
     * Собирает строку сдвига из MSK и GMT свдигов. Для России мы показываем
     * только сдвиг от Москвы (кроме самой Москвы — для неё ничего), для мира
     * GMT сдвиг, а для СНГ — оба.
     */
    private static String getOffsetString(GroupType group, String mskOffset, String gmtOffset, boolean isMskZone) {
        switch (group) {
            case RUSSIA:
                return isMskZone ? "" : "(MSK " + mskOffset + ")";
            case CIS:
                return "(MSK " + mskOffset + ", GMT " + gmtOffset + ")";
            case WORLD:
                return "(GMT " + gmtOffset + ")";
        }
        return "";
    }

    private static String getName(Locale locale, GeoTimezone timezone, String suffix) {
        String baseName;
        switch (locale.getLanguage()) {
            case "en":
                baseName = timezone.getNameEn();
                break;
            case "tr":
                baseName = timezone.getNameTr();
                break;
            case "uk":
                baseName = timezone.getNameUa();
                break;
            default:
                baseName = timezone.getNameRu();
        }
        return baseName + (suffix.isEmpty() ? "" : " " + suffix);
    }

    private static String formatOffset(int offsetSeconds, int baseOffsetSeconds) {
        int diff = offsetSeconds - baseOffsetSeconds;
        int hours = diff / (60 * 60);
        int minutes = Math.abs(diff - hours * 60 * 60) / 60;
        return String.format("%s%02d:%02d", diff >= 0 ? "+" : "-", Math.abs(hours), minutes);
    }

    static GdInternalPageInfo toGdInternalPageInfo(Page page) {
        return new GdInternalPageInfo()
                .withPageId(page.getId())
                .withName(page.getName())
                .withDescription(page.getDescription());
    }

}
