package ru.yandex.direct.grid.processing.service.constant;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.GdTimeInterval;
import ru.yandex.direct.grid.model.campaign.GdBroadMatch;
import ru.yandex.direct.grid.model.campaign.GdCampaignAttributionModel;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdCpmYndxFrontpageCampaignShowType;
import ru.yandex.direct.grid.model.campaign.GdMeaningfulGoal;
import ru.yandex.direct.grid.model.campaign.facelift.GdCampaignAdditionalData;
import ru.yandex.direct.grid.model.campaign.notification.GdCampaignCheckPositionInterval;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudget;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetPeriod;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyAvgCpcPerFilter;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyAvgCpm;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyManual;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyRoi;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyType;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyOptimizeClicks;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyOptimizeConversions;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyOptimizeInstalls;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdBroadMatchRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignEmailSettingsRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignNotificationRequest;
import ru.yandex.direct.grid.processing.model.campaign.mutation.GdCampaignSmsSettingsRequest;
import ru.yandex.direct.grid.processing.model.constants.GdCampaignDefaultValues;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.DEFAULT_CPI_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.RESERVE_RETURN_MAX;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdAttributionModel;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdDayBudgetShowMode;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdEshowsRate;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdEshowsVideoType;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdImpressionStandardTime;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdTimeTarget;
import static ru.yandex.direct.grid.model.utils.GridTimeUtils.toGdTimeInterval;
import static ru.yandex.direct.grid.processing.service.campaign.CampaignDataConverter.toGdCampaignCheckPositionInterval;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.DEFAULT_TIMEZONE;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.UA_DEFAULT_TIMEZONE;
import static ru.yandex.direct.libs.timetarget.TimeTargetUtils.defaultTimeTarget;
import static ru.yandex.direct.utils.CommonUtils.nvl;


@ParametersAreNonnullByDefault
public class DefaultValuesUtils {

    // любые изменения в стратегии CPV потребуют времени для обучения стратегии (от 14 дней)
    public static final long MIN_RECOMMENDED_CPV_PERIOD = 14L;
    // коэффициент 168 получен делением рекомендованного 2-недельного бюджета в рублях для CPV - 50000
    // на минимальный недельный автобюджет в рублях - 300
    public static final BigDecimal MIN_RECOMMENDED_CPV_AUTOBUDGET_MULTIPLICAND = BigDecimal.valueOf(168);

    public static final Set<GdCampaignType> SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES =
            Set.of(
                    GdCampaignType.TEXT,
                    GdCampaignType.MCBANNER,
                    GdCampaignType.DYNAMIC,
                    GdCampaignType.CONTENT_PROMOTION,
                    GdCampaignType.CPM_PRICE,
                    GdCampaignType.INTERNAL_FREE,
                    GdCampaignType.INTERNAL_DISTRIB,
                    GdCampaignType.INTERNAL_AUTOBUDGET,
                    GdCampaignType.CPM_BANNER,
                    GdCampaignType.PERFORMANCE,
                    GdCampaignType.MOBILE_CONTENT,
                    GdCampaignType.CPM_YNDX_FRONTPAGE
            );

    public static final GdCampaignCheckPositionInterval DEFAULT_CAMPAIGN_CHECK_POSITION_INTERVAL =
            toGdCampaignCheckPositionInterval(CampaignConstants.DEFAULT_CAMPAIGN_CHECK_POSITION_INTERVAL);
    public static final GdTimeInterval DEFAULT_SMS_TIME_INTERVAL =
            toGdTimeInterval(CampaignConstants.DEFAULT_SMS_TIME_INTERVAL);
    private static final Map<YandexDomain, Long> DOMAIN_TO_TIMEZONE = Map.of(
            YandexDomain.RU, DEFAULT_TIMEZONE,
            YandexDomain.UA, UA_DEFAULT_TIMEZONE
    );
    private static final int DEFAULT_CPM_STRATEGY_PERIOD = 29;

    private DefaultValuesUtils() {
    }

    static GdCampaignDefaultValues getCampaignDefaultValues(Set<String> operatorFeatures, Set<String> clientFeatures,
                                                            Boolean isProStrategyViewEnabled,
                                                            CampaignAttributionModel defaultAttributionModel) {
        return getCampaignDefaultValues(singletonList(GdCampaignType.TEXT), operatorFeatures, clientFeatures,
                isProStrategyViewEnabled, defaultAttributionModel)
                .get(0);
    }

    static List<GdCampaignDefaultValues> getCampaignDefaultValues(@Nullable Collection<GdCampaignType> campaignTypes,
                                                                  Set<String> operatorFeatures,
                                                                  Set<String> clientFeatures,
                                                                  @Nullable Boolean isProStrategyViewEnabled,
                                                                  CampaignAttributionModel defaultAttributionModel) {

        final boolean isPro = nvl(isProStrategyViewEnabled, Boolean.TRUE);

        return StreamEx.of(nvl(campaignTypes, SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES))
                .map(type -> getCampaignDefaultValues(type, operatorFeatures, clientFeatures, isPro,
                        defaultAttributionModel))
                .toList();
    }

    private static GdCampaignFlatStrategy getDefaultStrategy(GdCampaignType campaignType,
                                                             Set<String> operatorFeatures,
                                                             Set<String> clientFeatures,
                                                             boolean isProViewEnabled) {
        LocalDate now = LocalDate.now();
        var defaultStrategy = new GdCampaignStrategyManual()
                .withPlatform(GdCampaignPlatform.BOTH)
                .withIsAutoBudget(false)
                .withType(GdCampaignStrategyType.DEFAULT)
                .withStrategyType(GdStrategyType.DEFAULT)
                .withSeparateBidding(true);

        switch (campaignType) {
            case TEXT:
                if (operatorFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_ROI.getName())) {
                    return new GdCampaignStrategyRoi()
                            .withPlatform(GdCampaignPlatform.BOTH)
                            .withIsAutoBudget(true)
                            .withType(GdCampaignStrategyType.ROI)
                            .withStrategyType(GdStrategyType.ROI)
                            .withReserveReturn(RESERVE_RETURN_MAX);
                } else if (operatorFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_AVG_CPA.getName())) {
                    return new GdStrategyOptimizeConversions()
                            .withPlatform(GdCampaignPlatform.BOTH)
                            .withIsAutoBudget(true)
                            .withType(GdCampaignStrategyType.AVG_CPA)
                            .withStrategyType(GdStrategyType.OPTIMIZE_CONVERSIONS);
                } else if (operatorFeatures.contains(FeatureName.DEFAULT_AUTOBUDGET_AVG_CLICK_WITH_WEEK_BUDGET.getName())) {
                    return new GdStrategyOptimizeClicks()
                            .withPlatform(GdCampaignPlatform.BOTH)
                            .withIsAutoBudget(true)
                            .withType(GdCampaignStrategyType.AVG_CLICK)
                            .withStrategyType(GdStrategyType.OPTIMIZE_CLICKS)
                            .withBudget(new GdCampaignBudget()
                                    .withSum(BigDecimal.ZERO)
                                    .withPeriod(GdCampaignBudgetPeriod.WEEK)
                                    .withShowMode(GdCampaignBudgetShowMode.DEFAULT));
                }
                return defaultStrategy;
            case INTERNAL_AUTOBUDGET:
                return new GdStrategyOptimizeClicks()
                        .withPlatform(GdCampaignPlatform.BOTH)
                        .withIsAutoBudget(true)
                        .withType(GdCampaignStrategyType.AVG_CLICK)
                        .withStrategyType(GdStrategyType.OPTIMIZE_CLICKS);
            case MOBILE_CONTENT:
                if (clientFeatures.contains(FeatureName.CPA_PAY_FOR_CONVERSIONS_MOBILE_APPS_ALLOWED.getName())) {
                    return new GdStrategyOptimizeInstalls()
                            .withPlatform(GdCampaignPlatform.BOTH)
                            .withIsAutoBudget(true)
                            .withType(GdCampaignStrategyType.AVG_CPA)
                            .withStrategyType(GdStrategyType.OPTIMIZE_CONVERSIONS)
                            .withGoalId(DEFAULT_CPI_GOAL_ID)
                            .withPayForConversion(true);
                } else {
                    return new GdStrategyOptimizeClicks()
                            .withPlatform(GdCampaignPlatform.BOTH)
                            .withIsAutoBudget(true)
                            .withType(GdCampaignStrategyType.AVG_CLICK)
                            .withStrategyType(GdStrategyType.OPTIMIZE_CLICKS);
                }
            case CPM_YNDX_FRONTPAGE:
                return defaultCpmMaxImpressionsCustomPeriodStrategy(now);
            case CPM_BANNER:
                return defaultCpmMaxImpressionsCustomPeriodStrategy(now);
            case CPM_DEALS:
                return new GdCampaignStrategyManual()
                        .withType(GdCampaignStrategyType.CPM_DEFAULT)
                        .withStrategyType(GdStrategyType.CPM_DEFAULT)
                        .withIsAutoBudget(false)
                        .withPlatform(GdCampaignPlatform.CONTEXT)
                        .withSeparateBidding(true);
            case PERFORMANCE:
                return new GdCampaignStrategyAvgCpcPerFilter()
                        .withPlatform(GdCampaignPlatform.CONTEXT)
                        .withIsAutoBudget(true)
                        .withType(GdCampaignStrategyType.AVG_CPC_PER_FILTER)
                        .withStrategyType(GdStrategyType.AVG_CPC_PER_FILTER);
            case DYNAMIC:
                return createDefaultDynamicStrategy(clientFeatures, isProViewEnabled);
            default:
                return defaultStrategy;
        }
    }

    private static GdCampaignFlatStrategy createDefaultDynamicStrategy(Set<String> clientFeatures,
                                                                       boolean isProViewEnabled) {
        if (clientFeatures.contains(FeatureName.SIMPLIFIED_STRATEGY_VIEW_ENABLED.getName()) && !isProViewEnabled) {
            return createDynamicOptimizeClicksStrategy();
        }
        return createDynamicManualStrategy();
    }

    private static GdCampaignStrategyManual createDynamicManualStrategy() {
        return new GdCampaignStrategyManual()
                .withPlatform(GdCampaignPlatform.SEARCH)
                .withIsAutoBudget(false)
                .withType(GdCampaignStrategyType.DEFAULT)
                .withStrategyType(GdStrategyType.DEFAULT);
    }

    private static GdStrategyOptimizeClicks createDynamicOptimizeClicksStrategy() {
        return new GdStrategyOptimizeClicks()
                .withPlatform(GdCampaignPlatform.SEARCH)
                .withIsAutoBudget(true)
                .withType(GdCampaignStrategyType.AVG_CLICK)
                .withStrategyType(GdStrategyType.OPTIMIZE_CLICKS);
    }

    private static GdCampaignStrategyAvgCpm defaultCpmMaxImpressionsCustomPeriodStrategy(LocalDate start) {
        GdCampaignBudget budget = new GdCampaignBudget()
                .withPeriod(GdCampaignBudgetPeriod.CUSTOM)
                .withStart(start)
                .withFinish(start.plusDays(DEFAULT_CPM_STRATEGY_PERIOD))
                .withAutoProlongation(true)
                .withSum(BigDecimal.ZERO)
                .withShowMode(GdCampaignBudgetShowMode.DEFAULT);

        return new GdCampaignStrategyAvgCpm()
                .withPlatform(GdCampaignPlatform.CONTEXT)
                .withIsAutoBudget(true)
                .withType(GdCampaignStrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD)
                .withStrategyType(GdStrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD)
                .withAvgCpm(new BigDecimal(100))
                .withBudget(budget)
                .withDailyChangeCount(0L);
    }

    private static GdCampaignDefaultValues getCampaignDefaultValues(GdCampaignType campaignType,
                                                                    Set<String> availableFeatures,
                                                                    Set<String> clientFeatures,
                                                                    boolean isProViewEnabled,
                                                                    CampaignAttributionModel defaultAttributionModel) {
        checkState(SUPPORTED_CAMPAIGN_TYPES_WITH_DEFAULT_VALUES.contains(campaignType));
        var advancedGeoTargeting = clientFeatures.contains(FeatureName.ADVANCED_GEOTARGETING.getName());
        GdCampaignDefaultValues campaignDefaultValues = new GdCampaignDefaultValues()
                .withCampaignType(campaignType)
                .withDefaultWarningBalance(CampaignConstants.DEFAULT_CAMPAIGN_WARNING_BALANCE)
                .withDefaultCheckPositionInterval(DEFAULT_CAMPAIGN_CHECK_POSITION_INTERVAL)
                .withDefaultSmsTimeInterval(DEFAULT_SMS_TIME_INTERVAL)
                .withDefaultMeaningfulGoalId(ENGAGED_SESSION_GOAL_ID)
                .withCampaignName("Новая")
                .withStartDate(LocalDate.now())
                .withHasExtendedGeoTargeting(true)
                .withUseCurrentRegion(true)
                .withUseRegularRegion(advancedGeoTargeting)
                .withStrategy(getDefaultStrategy(campaignType, availableFeatures, clientFeatures, isProViewEnabled))
                .withBidModifiers(Collections.emptyList())
                .withBroadMatch(new GdBroadMatch().withBroadMatchFlag(true)
                        .withBroadMatchGoalId(null)
                        .withBroadMatchLimit(CampaignConstants.BROAD_MATCH_LIMIT_DEFAULT))
                .withHasSiteMonitoring(false)
                .withMetrikaCounters(Collections.emptySet())
                .withMenaningfulGoals(List.of(new GdMeaningfulGoal().withGoalId(ENGAGED_SESSION_GOAL_ID)))
                .withAttributionModel(toGdAttributionModel(defaultAttributionModel))
                .withCommonVcard(null)
                .withOpenStat(false)
                .withClientDialogId(null)
                .withMarkMetrikaCunters(true)
                .withDisabledPlaces(Collections.emptySet())
                .withDisabledIps(Collections.emptySet())
                .withHasTitleSubstitute(true)
                .withContentLanguage(null)
                .withTurboAppsEnabled(false)
                .withDefaultBroadMatchLimit(CampaignConstants.BROAD_MATCH_LIMIT_DEFAULT)
                .withDefaultBroadMatchAllGoalsId(CampaignConstants.BROAD_MATCH_ALL_GOALS_ID)
                .withDefaultContextLimit(CampaignConstants.DEFAULT_CONTEXT_LIMIT)
                .withDefaultHasEnableCpcHold(CampaignConstants.DEFAULT_ENABLE_CPC_HOLD)
                .withDefaultEnableCompanyInfo(CampaignConstants.DEFAULT_ENABLE_COMPANY_INFO)
                .withDefaultIsAloneTrafaretAllowed(CampaignConstants.DEFAULT_IS_ALONE_TRAFARET_ALLOWED)
                .withDefaultHasTurboSmarts(CampaignConstants.DEFAULT_HAS_TURBO_SMARTS)
                .withDefaultRequireFiltrationByDontShowDomains(
                        CampaignConstants.DEFAULT_REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS)
                .withDefaultAddMetrikaTagToUrl(CampaignConstants.DEFAULT_ADD_METRIKA_TAG_TO_URL)
                .withDefaultAddOpenstatTagToUrl(CampaignConstants.DEFAULT_ADD_OPENSTAT_TAG_TO_URL)
                .withDefaultExcludePausedCompetingAds(CampaignConstants.DEFAULT_EXCLUDE_PAUSED_COMPETING_ADS)
                .withDefaultMeaningfulGoalId(CampaignConstants.ENGAGED_SESSION_GOAL_ID)
                .withDefaultEshowsBannerRate(toGdEshowsRate(CampaignConstants.DEFAULT_ESHOWS_BANNER_RATE))
                .withDefaultEshowsVideoRate(toGdEshowsRate(CampaignConstants.DEFAULT_ESHOWS_VIDEO_RATE))
                .withDefaultEshowsVideoType(toGdEshowsVideoType(CampaignConstants.DEFAULT_ESHOWS_VIDEO_TYPE))
                .withDefaultTimeTarget(defaultGdTimeTarget())
                .withDefaultDayBudget(CampaignConstants.DEFAULT_DAY_BUDGET)
                .withDefaultDayBudgetShowMode(toGdDayBudgetShowMode(CampaignConstants.DEFAULT_DAY_BUDGET_SHOW_MODE))
                .withDefaultImpressionStandardTime(
                        toGdImpressionStandardTime(CampaignConstants.DEFAULT_IMPRESSION_STANDARD_TIME))
                .withDefaultAdditionalData(
                        new GdCampaignAdditionalData()
                                .withHref("")
                                .withBusinessCategory("")
                                .withCompanyName(""))
                .withIsUniversalCamp(false)
                .withDefaultCalltrackingSettingsId(null)
                .withIsRecommendationsManagementEnabled(CampaignConstants.DEFAULT_IS_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .withIsPriceRecommendationsManagementEnabled(
                        CampaignConstants.DEFAULT_IS_PRICE_RECOMMENDATIONS_MANAGEMENT_ENABLED)
                .withDefaultCampAimType(null);

        // TODO(dimitrovsd): переделать на TypeSupport'ы
        switch (campaignType) {
            case CONTENT_PROMOTION:
            case MCBANNER:
                campaignDefaultValues.getStrategy().setPlatform(GdCampaignPlatform.SEARCH);
                break;
            case CPM_PRICE:
                campaignDefaultValues.withEndDate(LocalDate.now());
                campaignDefaultValues.withPricePackageId(0L);
                break;
            case CPM_YNDX_FRONTPAGE:
                campaignDefaultValues.withAllowedFrontpageType(
                        ImmutableSet.of(GdCpmYndxFrontpageCampaignShowType.FRONTPAGE,
                                GdCpmYndxFrontpageCampaignShowType.FRONTPAGE_MOBILE,
                                GdCpmYndxFrontpageCampaignShowType.BROWSER_NEW_TAB));
                break;
            case PERFORMANCE:
                campaignDefaultValues.withIsWwManagedOrder(false);
            case INTERNAL_FREE:
            case INTERNAL_DISTRIB:
            case INTERNAL_AUTOBUDGET:
                campaignDefaultValues.setAttributionModel(GdCampaignAttributionModel.LAST_SIGNIFICANT_CLICK);
                break;
        }

        return campaignDefaultValues;
    }

    public static GdTimeTarget defaultGdTimeTarget() {
        YandexDomain yandexDomain = HostUtils.getYandexDomain().orElse(YandexDomain.RU);
        return toGdTimeTarget(defaultTimeTarget())
                .withUseWorkingWeekends(yandexDomain != YandexDomain.TR)
                .withIdTimeZone(DOMAIN_TO_TIMEZONE.getOrDefault(yandexDomain, DEFAULT_TIMEZONE));
    }

    public static GdCampaignNotificationRequest defaultGdCampaignNotification(String userEmail) {
        return new GdCampaignNotificationRequest()
                .withSmsSettings(new GdCampaignSmsSettingsRequest()
                        .withSmsTime(DEFAULT_SMS_TIME_INTERVAL)
                        .withEnableEvents(emptySet()))
                .withEmailSettings(new GdCampaignEmailSettingsRequest().withEmail(userEmail));
    }

    public static GdBroadMatchRequest defaultGdBroadMatch() {
        return new GdBroadMatchRequest()
                .withBroadMatchFlag(false)
                .withBroadMatchLimit(CampaignConstants.BROAD_MATCH_LIMIT_DEFAULT);
    }
}
