package ru.yandex.direct.grid.processing.service.contentpromotion;

import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionMetaWithStatus;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionService;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.contentpromotion.GdGetContentPromotionMetaInput;
import ru.yandex.direct.grid.processing.model.contentpromotion.GdGetContentPromotionMetaPayload;
import ru.yandex.direct.grid.processing.service.contentpromotion.validation.ContentPromotionMetaValidationService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.grid.processing.model.contentpromotion.GdContentPromotionType.toSource;
import static ru.yandex.direct.grid.processing.service.contentpromotion.ContentPromotionConverter.convertContentPromotionMeta;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис, возвращающий метаданные о продвигаемом контенте
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class ContentPromotionGraphQlService {

    private final ContentPromotionService contentPromotionService;
    private final ContentPromotionMetaValidationService contentPromotionMetaValidationService;
    private final GridValidationResultConversionService validationResultConverter;;

    @Autowired
    public ContentPromotionGraphQlService(ContentPromotionService contentPromotionService,
                                          ContentPromotionMetaValidationService contentPromotionMetaValidationService,
                                          GridValidationResultConversionService validationResultConverter) {
        this.contentPromotionService = contentPromotionService;
        this.contentPromotionMetaValidationService = contentPromotionMetaValidationService;
        this.validationResultConverter = validationResultConverter;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getContentPromotionMeta")
    public GdGetContentPromotionMetaPayload getContentPromotionMeta(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGetContentPromotionMetaInput input) {
        contentPromotionMetaValidationService.validateGetContentPromotionMetaInput(input);

        ContentPromotionContentType contentPromotionType = checkNotNull(toSource(input.getContentPromotionType()));

        String requestId = UUID.randomUUID().toString().toUpperCase();
        ContentPromotionMetaWithStatus metaWithStatus = contentPromotionService.getMeta(input.getUrl(),
                requestId,
                contentPromotionType,
                input.getCampaignId(),
                input.getAdGroupId(),
                checkNotNull(context.getSubjectUser()).getLogin());

        GdValidationResult validationResult = validationResultConverter.buildGridValidationResultIfErrorsOrWarnings(
                metaWithStatus.getValidationResult(),
                path(field(GdGetContentPromotionMetaInput.URL)));

        return new GdGetContentPromotionMetaPayload()
                .withContentPromotionMeta(convertContentPromotionMeta(contentPromotionType, metaWithStatus.getMeta()))
                .withValidationResult(validationResult);
    }
}
