package ru.yandex.direct.grid.processing.service.conversioncenter

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.direct.core.configuration.CoreConfiguration
import ru.yandex.direct.core.entity.conversionsource.model.ConversionAction
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionValue
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceSettings
import ru.yandex.direct.core.entity.conversionsource.model.Destination
import ru.yandex.direct.core.entity.conversionsource.model.MetrikaGoalSelection
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingInfo
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingStatus
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode
import ru.yandex.direct.core.entity.retargeting.converter.GoalConverter
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.currency.Money
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionStats
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionValue
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionValueDynamic
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionValueFixed
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionValueNotSet
import ru.yandex.direct.grid.processing.model.goal.GdConversionActionValueType
import ru.yandex.direct.grid.processing.model.goal.GdConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourceSettingsUnion
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourceStats
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourceType
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourceTypeCode
import ru.yandex.direct.grid.processing.model.goal.GdExternalConversionAction
import ru.yandex.direct.grid.processing.model.goal.GdExternalConversionActionInfo
import ru.yandex.direct.grid.processing.model.goal.GdFtpConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdFtpConversionSourceSettings
import ru.yandex.direct.grid.processing.model.goal.GdGoogleSheetsConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdGoogleSheetsConversionSourceSettings
import ru.yandex.direct.grid.processing.model.goal.GdLinkConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdLinkConversionSourceSettings
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaConversionAction
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaGoalSelection
import ru.yandex.direct.grid.processing.model.goal.GdSFtpConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdSFtpConversionSourceSettings
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSourceConversionAction
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSourceConversionActionValue
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSourceConversionActionValueType
import ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoal
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.metrika.client.model.request.UserCountersExtendedFilter
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.utils.CommonUtils.min
import ru.yandex.direct.utils.crypt.Encrypter
import ru.yandex.direct.utils.mapToSet

@Component
class ConversionCenterConverter(
    private val metrikaClient: MetrikaClient,
    private val rbacService: RbacService,
    @Qualifier(CoreConfiguration.CONVERSION_CENTER_ENCRYPTER_BEAN_NAME) private val encrypter: Encrypter
) {
    companion object {
        private val LOGGER = LoggerFactory.getLogger(ConversionCenterConverter::class.java)
    }

    fun fromGdMetrikaGoalSelection(gdMetrikaGoalSelections: List<GdMetrikaGoalSelection>): List<MetrikaGoalSelection> {
        return gdMetrikaGoalSelections.map { fromGdMetrikaGoalSelection(it) }
    }

    fun toGdConversionSourceTypes(conversionSourceTypes: List<ConversionSourceType>): List<GdConversionSourceType> {
        val gdConversionSourceType = conversionSourceTypes.map {
            toGdConversionSourceType(it)
        }
        return gdConversionSourceType
    }

    fun toGdConversionSources(
        clientId: ClientId,
        conversionSources: List<ConversionSource>,
        goalStat: Map<Long, ConversionsStat>,
    ): List<GdConversionSource> {
        val counterIds = conversionSources.mapToSet { it.counterId }
        val filter = UserCountersExtendedFilter().withCounterIds(counterIds.toList())
        val uids = rbacService.getClientRepresentativesUids(clientId).toList()
        val availableCounterIds = metrikaClient.getUsersCountersNumExtended2(uids, filter).users
            .map { it.counters }
            .flatten()
            .map { it.id }
            .toSet()
        val metrikaGoalsById = metrikaClient.getMassCountersGoalsFromMetrika(availableCounterIds).values
            .map(GoalConverter::fromCounterGoals)
            .flatten()
            .associateBy { it.id }
        val availableSources = conversionSources.map {
            if (!availableCounterIds.contains(it.counterId.toInt())) {
                it.copy(actions = emptyList())
            } else {
                it
            }
        }
        return availableSources.mapNotNull { toGdConversionSource(it, metrikaGoalsById, goalStat) }
    }

    private fun fromGdMetrikaGoalSelection(gdMetrikaGoalSelection: GdMetrikaGoalSelection): MetrikaGoalSelection {
        return MetrikaGoalSelection(
            gdMetrikaGoalSelection.metrikaCounterId,
            gdMetrikaGoalSelection.goalId,
            gdMetrikaGoalSelection.value?.let { ConversionActionValue.Fixed(Money.valueOf(it.cost, it.currency)) },
            gdMetrikaGoalSelection.isSelected
        )
    }

    private fun toGdConversionSourceType(conversionSourceType: ConversionSourceType): GdConversionSourceType {
        return GdConversionSourceType()
            .withId(conversionSourceType.id)
            .withName(conversionSourceType.name)
            .withDescription(conversionSourceType.description)
            .withIconUrl(conversionSourceType.iconUrl)
            .withActivationUrl(conversionSourceType.activationUrl)
            .withIsDraft(conversionSourceType.isDraft)
            .withPosition(conversionSourceType.position)
            .withCode(conversionSourceType.code.toGrid())
            .withIsEditable(conversionSourceType.isEditable)
            .withDescriptionEn(conversionSourceType.descriptionEn)
            .withNameEn(conversionSourceType.nameEn)
    }

    private fun toGdConversionSource(
        source: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdConversionSource? = when (source.typeCode) {
        ConversionSourceTypeCode.METRIKA -> toGdMetrikaConversionSource(source, metrikaGoalsById, goalStat)
        ConversionSourceTypeCode.LINK -> toGdLinkConversionSource(source, metrikaGoalsById, goalStat)
        ConversionSourceTypeCode.FTP -> toGdFtpConversionSource(source, metrikaGoalsById, goalStat)
        ConversionSourceTypeCode.GOOGLE_SHEETS -> toGdGoogleSheetsConversionSource(source, metrikaGoalsById, goalStat)
        ConversionSourceTypeCode.SFTP -> toGdSFtpConversionSource(source, metrikaGoalsById, goalStat)
        else -> {
            LOGGER.error("Conversion source type ${source.typeCode} is not implemented yet.")
            null
        }
    }

    private fun toGdMetrikaConversionSource(
        conversionSource: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdConversionSource {
        return GdMetrikaConversionSource()
            .withId(conversionSource.id)
            .withName(conversionSource.name)
            .withType(conversionSource.typeCode.toGrid())
            .withConversionActions(toGdMetrikaConversionAction(conversionSource.actions, metrikaGoalsById, goalStat))
            .withCounterId(conversionSource.counterId)
            .withStats(computeAggregatedStat(conversionSource.actions, goalStat))
            .withProcessingIsRunning(false) // договорились с фронтом, пока функциональность не реализована будет false
            .withErrors(emptyList())
    }

    private fun toGdMetrikaConversionAction(
        conversionAction: List<ConversionAction>,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): List<GdMetrikaConversionAction> {
        return conversionAction.filter {
            if (metrikaGoalsById[it.goalId] == null) {
                LOGGER.error("Could not find conversion action goal with id ${it.goalId} in Metrika")
            }
            metrikaGoalsById[it.goalId] != null
        }.map {
            GdMetrikaConversionAction()
                .withName(it.name)
                .withValue(it.value.toGrid())
                .withGoal(toGdGoal(metrikaGoalsById[it.goalId]!!, null))
                .withStats(goalStat[it.goalId]?.let { conversions ->
                    GdConversionActionStats()
                        .withConversionVisitsCount(conversions.visitsCount)
                        .withAttributionRatio(conversions.attributionRatio)
                })
        }
    }

    private fun toGdLinkConversionSource(
        source: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdLinkConversionSource {
        return GdLinkConversionSource()
            .withId(source.id)
            .withName(source.name)
            .withType(source.typeCode.toGrid())
            .withConversionActions(toGdExternalConversionAction(source.actions, metrikaGoalsById, goalStat))
            .withSettings((source.settings as ConversionSourceSettings.Link).toGrid())
            .withCounterId(source.counterId)
            .withStats(computeAggregatedStat(source.actions, goalStat))
            .withUpdateFileReminder(source.updatePeriodHours > 0)
            .withUpdateFileReminderDaysCount(source.updatePeriodHours.takeIf { it > 0 }?.let { it / 24 })
            .withProcessingIsRunning(false) // договорились с фронтом, пока функциональность не реализована будет false
            .withErrors(emptyList())
    }

    private fun toGdSFtpConversionSource(
        source: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdSFtpConversionSource {
        val settings = source.settings as ConversionSourceSettings.SFtp
        return GdSFtpConversionSource()
            .withId(source.id)
            .withName(source.name)
            .withType(source.typeCode.toGrid())
            .withConversionActions(toGdExternalConversionAction(source.actions, metrikaGoalsById, goalStat))
            .withSettings(settings.toGrid())
            .withCounterId(source.counterId)
            .withStats(computeAggregatedStat(source.actions, goalStat))
            .withUpdateFileReminder(source.updatePeriodHours > 0)
            .withUpdateFileReminderDaysCount(source.updatePeriodHours.takeIf { it > 0 }?.let { it / 24 })
            .withProcessingIsRunning(false) // договорились с фронтом, пока функциональность не реализована будет false
            .withErrors(emptyList())
    }

    private fun toGdFtpConversionSource(
        source: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdFtpConversionSource {
        val settings = source.settings as ConversionSourceSettings.Ftp
        return GdFtpConversionSource()
            .withId(source.id)
            .withName(source.name)
            .withType(source.typeCode.toGrid())
            .withConversionActions(toGdExternalConversionAction(source.actions, metrikaGoalsById, goalStat))
            .withSettings(settings.toGrid())
            .withCounterId(source.counterId)
            .withStats(computeAggregatedStat(source.actions, goalStat))
            .withUpdateFileReminder(source.updatePeriodHours > 0)
            .withUpdateFileReminderDaysCount(source.updatePeriodHours.takeIf { it > 0 }?.let { it / 24 })
            .withProcessingIsRunning(false) // договорились с фронтом, пока функциональность не реализована будет false
            .withErrors(emptyList())
    }

    private fun toGdGoogleSheetsConversionSource(
        source: ConversionSource,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdGoogleSheetsConversionSource {
        return GdGoogleSheetsConversionSource()
            .withId(source.id)
            .withName(source.name)
            .withType(source.typeCode.toGrid())
            .withConversionActions(toGdExternalConversionAction(source.actions, metrikaGoalsById, goalStat))
            .withSettings((source.settings as ConversionSourceSettings.GoogleSheets).toGrid())
            .withCounterId(source.counterId)
            .withStats(computeAggregatedStat(source.actions, goalStat))
            .withUpdateFileReminder(source.updatePeriodHours > 0)
            .withUpdateFileReminderDaysCount(source.updatePeriodHours.takeIf { it > 0 }?.let { it / 24 })
            .withProcessingIsRunning(false) // договорились с фронтом, пока функциональность не реализована будет false
            .withErrors(emptyList())
    }

    private fun toGdExternalConversionAction(
        conversionAction: List<ConversionAction>,
        metrikaGoalsById: Map<Long, Goal>,
        goalStat: Map<Long, ConversionsStat>,
    ): List<GdExternalConversionAction> {
        return conversionAction.map {
            GdExternalConversionAction()
                .withName(it.name)
                .withValue(it.value.toGrid())
                .withGoal(
                    if (metrikaGoalsById[it.goalId] != null) {
                        toGdGoal(metrikaGoalsById[it.goalId]!!, null)
                    } else {
                        null
                    }
                )
                .withStats(goalStat[it.goalId]?.let { conversions ->
                    GdConversionActionStats()
                        .withConversionVisitsCount(conversions.visitsCount)
                        .withAttributionRatio(conversions.attributionRatio)
                })
                .withInfo(
                    GdExternalConversionActionInfo()
                        .withMatchingRatio(1f)
                        .withUpdatedSecAgo(0)
                )
        }
    }

    private fun computeAggregatedStat(
        actions: List<ConversionAction>,
        goalStat: Map<Long, ConversionsStat>,
    ): GdConversionSourceStats? {
        return actions.asSequence()
            .mapNotNull { it.goalId }
            .mapNotNull { goalStat[it] }
            .reduceOrNull { acc, stat ->
                val sumVisits = acc.visitsCount + stat.visitsCount
                val aproxNonAttributed =
                    (acc.visitsCount * acc.attributionRatio) + (stat.visitsCount * stat.attributionRatio)
                ConversionsStat(sumVisits, sumVisits / aproxNonAttributed)
            }
            ?.let {
                GdConversionSourceStats()
                    .withConversionVisitsCount(it.visitsCount)
                    .withAttributionRatio(min(it.attributionRatio, 1f))
            }
    }

    fun convertUpdateConversionSourceInput(
        clientId: ClientId, operatorUid: Long, input: GdUpdateConversionSource,
    ): ConversionSource {
        return ConversionSource(
            id = input.id,
            typeCode = input.type.toCore(),
            clientId = clientId,
            name = input.name,
            settings = input.settings.toCore(),
            counterId = input.counterId,
            actions = convertActionsToCore(input.conversionActions),
            updatePeriodHours = if (input.updateFileReminder) input.updateFileReminderDaysCount * 24 else 0,
            destination = Destination.CrmApi(counterId = input.counterId, accessUid = operatorUid),
            processingInfo = ProcessingInfo(ProcessingStatus.NEW)
        )
    }

    private fun GdConversionSourceSettingsUnion.toCore(): ConversionSourceSettings {
        return when {
            linkSettings != null -> linkSettings.toCore()
            ftpSettings != null -> ftpSettings.toCore()
            sftpSettings != null -> sftpSettings.toCore()
            googleSheetsSettings != null -> googleSheetsSettings.toCore()
            else -> throw IllegalStateException("Unknown GdConversionSourceSettingsUnion value")
        }
    }

    private fun ConversionSourceSettings.SFtp.toGrid(): GdSFtpConversionSourceSettings {
        return GdSFtpConversionSourceSettings()
            .withHost(host)
            .withLogin(login)
            .withPath(path)
            .withPassword(encrypter.decryptText(encryptedPassword))
            .withPort(port)
    }

    private fun ConversionSourceSettings.Ftp.toGrid(): GdFtpConversionSourceSettings {
        return GdFtpConversionSourceSettings()
            .withHost(host)
            .withLogin(login)
            .withPath(path)
            .withPort(port)
            .withPassword(encrypter.decryptText(encryptedPassword))
            .withTls(tls)
    }

    private fun GdSFtpConversionSourceSettings.toCore(): ConversionSourceSettings {
        return ConversionSourceSettings.SFtp(
            host = host,
            port = port,
            login = login,
            encryptedPassword = encrypter.encryptText(password),
            path = path
        )
    }

    private fun GdFtpConversionSourceSettings.toCore(): ConversionSourceSettings {
        return ConversionSourceSettings.Ftp(
            host = host,
            port = port,
            login = login,
            encryptedPassword = encrypter.encryptText(password),
            path = path,
            tls = tls ?: false
        )
    }
}

fun ConversionActionValue?.toGrid(): GdConversionActionValue {
    return when (this) {
        null -> GdConversionActionValueNotSet()
            .withType(GdConversionActionValueType.NOT_SET)
        is ConversionActionValue.Fixed -> GdConversionActionValueFixed()
            .withType(GdConversionActionValueType.FIXED)
            .withCost(cost.bigDecimalValue())
            .withCurrency(cost.currencyCode)
        is ConversionActionValue.Dynamic -> GdConversionActionValueDynamic()
            .withType(GdConversionActionValueType.DYNAMIC)
            .withCostFrom(costFrom?.bigDecimalValue())
            .withCostTo(costTo?.bigDecimalValue())
            .withCurrency(costFrom?.currencyCode)
    }
}

private fun convertActionsToCore(
    conversionActions: List<GdUpdateConversionSourceConversionAction>,
): List<ConversionAction> {
    return conversionActions.map { it.toCore() }
}

private fun GdUpdateConversionSourceConversionAction.toCore(): ConversionAction {
    return ConversionAction(
        name = this.name,
        goalId = null,
        value = this.value.toCore()
    )
}

private fun GdUpdateConversionSourceConversionActionValue.toCore(): ConversionActionValue? {
    @Suppress("WHEN_ENUM_CAN_BE_NULL_IN_JAVA")
    return when (valueType) {
        GdUpdateConversionSourceConversionActionValueType.NOT_SET -> null
        GdUpdateConversionSourceConversionActionValueType.FIXED ->
            ConversionActionValue.Fixed(Money.valueOf(fixedValue, currency))
        GdUpdateConversionSourceConversionActionValueType.FROM_FILE ->
            ConversionActionValue.Dynamic(null, null)
    }
}

private fun ConversionSourceSettings.Link.toGrid(): GdLinkConversionSourceSettings {
    return GdLinkConversionSourceSettings().withUrl(url)
}

private fun ConversionSourceSettings.GoogleSheets.toGrid(): GdGoogleSheetsConversionSourceSettings {
    return GdGoogleSheetsConversionSourceSettings().withUrl(url)
}

private fun GdLinkConversionSourceSettings.toCore(): ConversionSourceSettings {
    return ConversionSourceSettings.Link(url)
}

private fun GdGoogleSheetsConversionSourceSettings.toCore(): ConversionSourceSettings {
    return ConversionSourceSettings.GoogleSheets(url)
}

private fun GdConversionSourceTypeCode.toCore(): ConversionSourceTypeCode {
    return when (this) {
        GdConversionSourceTypeCode.APP -> ConversionSourceTypeCode.APP
        GdConversionSourceTypeCode.FILE -> ConversionSourceTypeCode.FILE
        GdConversionSourceTypeCode.FTP -> ConversionSourceTypeCode.FTP
        GdConversionSourceTypeCode.LINK -> ConversionSourceTypeCode.LINK
        GdConversionSourceTypeCode.GOOGLE_SHEETS -> ConversionSourceTypeCode.GOOGLE_SHEETS
        GdConversionSourceTypeCode.METRIKA -> ConversionSourceTypeCode.METRIKA
        GdConversionSourceTypeCode.API -> ConversionSourceTypeCode.API
        GdConversionSourceTypeCode.CRM -> ConversionSourceTypeCode.CRM
        GdConversionSourceTypeCode.OTHER -> ConversionSourceTypeCode.OTHER
        GdConversionSourceTypeCode.SFTP -> ConversionSourceTypeCode.SFTP
    }
}

fun ConversionSourceTypeCode.toGrid(): GdConversionSourceTypeCode {
    return when (this) {
        ConversionSourceTypeCode.APP -> GdConversionSourceTypeCode.APP
        ConversionSourceTypeCode.FILE -> GdConversionSourceTypeCode.FILE
        ConversionSourceTypeCode.FTP -> GdConversionSourceTypeCode.FTP
        ConversionSourceTypeCode.LINK -> GdConversionSourceTypeCode.LINK
        ConversionSourceTypeCode.GOOGLE_SHEETS -> GdConversionSourceTypeCode.GOOGLE_SHEETS
        ConversionSourceTypeCode.METRIKA -> GdConversionSourceTypeCode.METRIKA
        ConversionSourceTypeCode.API -> GdConversionSourceTypeCode.API
        ConversionSourceTypeCode.CRM -> GdConversionSourceTypeCode.CRM
        ConversionSourceTypeCode.OTHER -> GdConversionSourceTypeCode.OTHER
        ConversionSourceTypeCode.SFTP -> GdConversionSourceTypeCode.SFTP
    }
}
