package ru.yandex.direct.grid.processing.service.conversioncenter

import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.service.ConversionCenterMetrikaGoalsService
import ru.yandex.direct.core.entity.conversionsource.service.ConversionSourceService
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceType
import ru.yandex.direct.core.entity.conversionsourcetype.service.ConversionSourceTypeService
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.model.goal.GdConversionCenterInfo
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourceTypesInfo
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourcesInfo
import ru.yandex.direct.grid.processing.model.goal.GdConversionSourcesInfoParams
import ru.yandex.direct.grid.processing.model.goal.GdGoal
import ru.yandex.direct.grid.processing.model.goal.GdRemoveConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdRemoveConversionSourcePayload
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSource
import ru.yandex.direct.grid.processing.model.goal.GdUpdateConversionSourcePayload
import ru.yandex.direct.grid.processing.model.goal.GdUpdateMetrikaGoalsSelection
import ru.yandex.direct.grid.processing.model.goal.GdUpdateMetrikaGoalsSelectionPayload
import ru.yandex.direct.grid.processing.service.conversioncenter.validation.ConversionCenterValidationService
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService
import ru.yandex.direct.grid.processing.service.validation.GridValidationService
import ru.yandex.direct.grid.processing.util.StatHelper
import ru.yandex.direct.validation.result.PathHelper.field
import ru.yandex.direct.validation.result.PathHelper.path
import java.time.Instant
import java.util.concurrent.CompletableFuture

/**
 * Запросы и мутации связанные с Центром Конверсий
 */
@GridGraphQLService
class ConversionCenterGraphQlService(
    private val conversionSourceTypeService: ConversionSourceTypeService,
    private val conversionCenterConverter: ConversionCenterConverter,
    private val conversionSourceService: ConversionSourceService,
    private val conversionCenterMetrikaGoalsService: ConversionCenterMetrikaGoalsService,
    private val validationResultConverter: GridValidationResultConversionService,
    private val conversionCenterValidationService: ConversionCenterValidationService,
    private val goalConversionCenterInfoDataLoader: GoalConversionCenterInfoDataLoader,
    private val gridValidationService: GridValidationService,
    private val statisticService: ConversionCenterStatisticService,
) {
    /**
     * Получить возможные типы источников конверсий
     *
     * Возвращает базовое описание типа, для отображения выбора на фронтенде. Пользователь, выбрав тип источника
     * конверсии из этого списка, сможет создать уже конкретный источник конверсий этого типа.
     */
    @GraphQLQuery(name = "conversionSourceTypes")
    fun getConversionSourceTypes(
        @GraphQLRootContext context: GridGraphQLContext,
    ): GdConversionSourceTypesInfo {
        val operator = context.subjectUser
            ?: throw IllegalStateException("No subjectUser in graphql context")

        val conversionSourceTypes: List<ConversionSourceType> = if (operator.role.isInternal) {
            @Suppress("UsePropertyAccessSyntax")
            conversionSourceTypeService.getAll()
        } else {
            @Suppress("UsePropertyAccessSyntax")
            conversionSourceTypeService.getNotDrafts()
        }

        return GdConversionSourceTypesInfo().apply {
            types = conversionCenterConverter.toGdConversionSourceTypes(conversionSourceTypes)
        }
    }

    @GraphQLQuery(name = "conversionSourcesInfo")
    fun getConversionSourcesInfo(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLContext client: GdClient,
        @GraphQLArgument(name = "input") input: GdConversionSourcesInfoParams,
    ): GdConversionSourcesInfo {
        val clientId = ClientId.fromLong(client.info.id)

        val conversionSources = conversionSourceService.getByClientId(clientId)
        val (fromDate, toDate) = StatHelper.getDayPeriod(input.preset, Instant.now(), null)
        val goalStat = statisticService.calcGoalStatistic(clientId, conversionSources, fromDate, toDate)

        return GdConversionSourcesInfo().apply {
            sources = conversionCenterConverter.toGdConversionSources(clientId, conversionSources, goalStat)
        }
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "conversionCenterUpdateMetrikaGoalsSelection")
    fun updateMetrikaGoalsSelection(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdUpdateMetrikaGoalsSelection,
    ): GdUpdateMetrikaGoalsSelectionPayload {
        val clientId = context.subjectUser?.clientId ?: throw IllegalStateException("No subjectUser in graphql context")
        val operatorUid = context.operator.uid

        val result = conversionCenterMetrikaGoalsService.updateMetrikaGoalsSelection(
            clientId,
            operatorUid,
            conversionCenterConverter.fromGdMetrikaGoalSelection(input.selection)
        )

        return GdUpdateMetrikaGoalsSelectionPayload()
            .withIsMetrikaAvailable(true)
            .withValidationResult(gridValidationService.toGdValidationResult(result, path(field("selection"))))
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "conversionCenterUpdateConversionSource")
    fun updateConversionSource(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdUpdateConversionSource,
    ): GdUpdateConversionSourcePayload {
        val clientId = context.subjectUser?.clientId
            ?: throw IllegalStateException("No subjectUser in graphql context")

        val vr = conversionCenterValidationService.validateUpdateConversionSourceInput(input)
        if (vr.hasAnyErrors()) {
            return GdUpdateConversionSourcePayload().apply {
                isMetrikaAvailable = true
                validationResult = validationResultConverter.buildGridValidationResult(vr)
            }
        }
        val conversionSource: ConversionSource =
            conversionCenterConverter.convertUpdateConversionSourceInput(clientId, context.operator.uid, input)
        val res = if (conversionSource.id == null) {
            conversionSourceService.add(clientId, listOf(conversionSource))
        } else {
            conversionSourceService.update(clientId, listOf(conversionSource))
        }
        val gdVr = if (res.validationResult.hasAnyErrors()) {
            validationResultConverter.buildGridValidationResult(res.validationResult)
        } else {
            null
        }
        return GdUpdateConversionSourcePayload()
            .withIsMetrikaAvailable(true)
            .withValidationResult(gdVr)
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "conversionCenterRemoveConversionSource")
    fun removeConversionSource(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdRemoveConversionSource,
    ): GdRemoveConversionSourcePayload {
        val clientId = context.subjectUser?.clientId ?: throw IllegalStateException("No subjectUser in graphql context")

        val result = conversionSourceService.remove(clientId, input.ids)

        return GdRemoveConversionSourcePayload()
            .withValidationResult(
                gridValidationService.toGdValidationResult(
                    result.validationResult,
                    path(field("id"))
                )
            )
    }

    /**
     * Возвращает "выбранность" и ценность цели в Центре Конверсий
     */
    @GraphQLQuery(name = "conversionCenterInfo")
    fun goalConversionCenterInfo(@GraphQLContext goal: GdGoal): CompletableFuture<GdConversionCenterInfo> {
        return goalConversionCenterInfoDataLoader.get().load(goal.id)
    }
}
