package ru.yandex.direct.grid.processing.service.conversioncenter

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaignstatistic.repository.CampaignStatisticService
import ru.yandex.direct.core.entity.container.LocalDateRange
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.metrika.client.MetrikaClient
import ru.yandex.direct.utils.CommonUtils.min
import java.time.LocalDate

data class ConversionsStat(val visitsCount: Long, val attributionRatio: Float)

@Service
class ConversionCenterStatisticService(
    private val metrikaClient: MetrikaClient,
    private val campaignStatisticService: CampaignStatisticService,
) {

    fun calcGoalStatistic(
        clientId: ClientId,
        conversionSources: List<ConversionSource>,
        fromDate: LocalDate,
        toDate: LocalDate,
    ): Map<Long, ConversionsStat> {
        val counterIds = conversionSources.map { it.counterId.toInt() }.distinct()
        val goalIds = conversionSources.asSequence()
            .flatMap { it.actions }
            .mapNotNull { it.goalId }
            .toSet()

        val metrikaGoalStat = metrikaClient.getGoalsStatistics(counterIds, fromDate, toDate)
        val localDateRange = LocalDateRange().withFromInclusive(fromDate).withToInclusive(toDate)
        val directGoalStat = campaignStatisticService.getGoalsConversionsCount(clientId, goalIds, localDateRange)
        return goalIds.associateWith { computeStatEntry(directGoalStat[it], metrikaGoalStat[it]) }
    }

    fun computeStatEntry(clientDirectConversions: Long?, allMetrikaConversions: Long?): ConversionsStat =
        computeStatEntry(clientDirectConversions ?: 0L, allMetrikaConversions ?: 0L)

    private fun computeStatEntry(clientDirectConversions: Long, allMetrikaConversions: Long): ConversionsStat {
        return if (clientDirectConversions == 0L) {
            val ratio = if (allMetrikaConversions == 0L) 1f else 0f
            ConversionsStat(allMetrikaConversions, ratio)
        } else {
            val ratio = if (allMetrikaConversions == 0L) 1f else 1f * clientDirectConversions / allMetrikaConversions
            ConversionsStat(allMetrikaConversions, min(1f, ratio))
        }
    }
}

