package ru.yandex.direct.grid.processing.service.creative

import io.leangen.graphql.annotations.*
import org.apache.commons.lang3.ObjectUtils
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.BannerStorageClient.FLAG_TRUE_VALUE
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException
import ru.yandex.direct.bannerstorage.client.BsCreativeNotFoundException
import ru.yandex.direct.bannerstorage.client.BsCreativeValidationFailedException
import ru.yandex.direct.bannerstorage.client.model.*
import ru.yandex.direct.common.TranslationService
import ru.yandex.direct.common.util.HttpUtil
import ru.yandex.direct.core.entity.creative.model.CreativeType
import ru.yandex.direct.core.entity.creative.model.ModerationInfo
import ru.yandex.direct.core.entity.creative.model.ModerationInfoHtml
import ru.yandex.direct.core.entity.creative.model.StatusModerate
import ru.yandex.direct.core.entity.creative.service.CreativeService
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.creative.*
import java.util.*
import kotlin.collections.ArrayList
import kotlin.math.roundToInt
import kotlin.math.roundToLong
import ru.yandex.direct.core.entity.creative.model.Creative as CoreCreative

// https://tanker-beta.yandex-team.ru/project/bannerstorage/keyset/rest_errors?branch=master
@GridGraphQLService
open class AdDesignerGraphQlService @Autowired constructor(
    private val bannerStorageClient: BannerStorageClient,
    private val creativeService: CreativeService,
    private val translationService: TranslationService,
    private val featureService: FeatureService) {

    companion object {
        private val logger = LoggerFactory.getLogger(AdDesignerGraphQlService::class.java)

        // Параметр, который добавляем во все шаблоны для удобства фронтенда
        // В него записывается название креатива, которое в базе (и в АПИ BannerStorage)
        // хранится в том числе и отдельно как название креатива
        private val NAME_PARAMETER = "NAME"

        // TNS категории и бренды всегда одни и те же
        private val TNS_ARTICLES = listOf(TnsArticle(-3))
        private val TNS_BRANDS = listOf(TnsBrand(1))

        // Все номера шаблонов, которые можно использовать в Директе
        // (сюда включены и те, которые открываются старым конструктором)
        private val OLD_AND_NEW_DESIGNER_TEMPLATES = setOf(1013, 1042, 1058, 1062, 1063, 1069)
    }

    private fun convertTemplateParameterType(type: ParameterType): GdDesignerTemplateParameterType {
        return when (type) {
            ParameterType.FILE -> GdDesignerTemplateParameterType.FILE
            ParameterType.SELECT -> GdDesignerTemplateParameterType.SELECT
            ParameterType.FLAG -> GdDesignerTemplateParameterType.FLAG
            ParameterType.INT -> GdDesignerTemplateParameterType.INT
            ParameterType.STRING,
            ParameterType.CLICK,
            ParameterType.PIXEL,
            ParameterType.ALT,
            ParameterType.SINGLELINE,
            ParameterType.HTTPURL,
            ParameterType.SERVERPIXEL -> GdDesignerTemplateParameterType.STRING
            else -> {
                logger.warn("unknown type ${type}, using string")
                GdDesignerTemplateParameterType.STRING
            }
        }
    }

    private fun isUrlParameter(type: ParameterType): Boolean {
        return type == ParameterType.CLICK
            || type == ParameterType.PIXEL
            || type == ParameterType.HTTPURL
            || type == ParameterType.SERVERPIXEL;
    }

    private fun convertTemplateParameter(templateParameter: TemplateParameter): GdDesignerTemplateParameter {
        val parameter = GdDesignerTemplateParameter()
            .withName(templateParameter.name)
            .withType(convertTemplateParameterType(templateParameter.type))
            .withControl(convertTemplateParameterControl(templateParameter.control))
            .withLabel(templateParameter.label)
            .withDescription(templateParameter.description)
            .withMinValuesCount(templateParameter.validation.minListLength ?: 0)
            .withMaxValuesCount(templateParameter.validation.maxListLength ?: 0)
        when (parameter.type) {
            GdDesignerTemplateParameterType.FILE -> {
                parameter
                    .withFileParameter(
                        GdDesignerTemplateParameterFile()
                            .withAllowedFileTypes(
                                templateParameter.validation.fileTypes?.map {
                                    GdDesignerTemplateParameterFileTypeItem()
                                        .withMimeType(it.mimeType)
                                        .withExtensions(it.extensions.split(","))
                                }
                            )
                            .withMinSize(templateParameter.validation.min ?: 1)
                            .withMaxSize(templateParameter.validation.max ?: 30 * 1024)
                            .withMinWidth(templateParameter.validation.minWidth)
                            .withMaxWidth(templateParameter.validation.maxWidth)
                            .withMinHeight(templateParameter.validation.minHeight)
                            .withMaxHeight(templateParameter.validation.maxHeight)
                            .withMinDuration(templateParameter.validation.minDuration)
                            .withMaxDuration(templateParameter.validation.maxDuration)
                            .withMinFrameRate(templateParameter.validation.minFrameRate?.roundToInt())
                            .withMaxFrameRate(templateParameter.validation.maxFrameRate?.roundToInt()))
                    .withDefaultValue(
                        templateParameter.defaultValue?.let {
                            val file = bannerStorageClient.getFile(it.toInt())
                            GdDesignerCreativeParameterValue().withValueFile(convertFile(file))
                        })
            }
            GdDesignerTemplateParameterType.FLAG -> {
                parameter.withDefaultValue(
                    templateParameter.defaultValue?.let {
                        GdDesignerCreativeParameterValue()
                            .withValueFlag(it == FLAG_TRUE_VALUE)
                    })
            }
            GdDesignerTemplateParameterType.INT -> {
                parameter
                    .withIntParameter(
                        GdDesignerTemplateParameterInt()
                            .withMin(templateParameter.validation.min ?: 0)
                            .withMax(templateParameter.validation.max ?: Int.MAX_VALUE))
                    .withDefaultValue(
                        templateParameter.defaultValue?.let {
                            GdDesignerCreativeParameterValue()
                                .withValueInt(it)
                        })
            }
            GdDesignerTemplateParameterType.SELECT -> {
                parameter
                    .withSelectParameter(
                        GdDesignerTemplateParameterSelect()
                            .withItems(
                                templateParameter.validation.selectItems?.map {
                                    GdDesignerTemplateParameterSelectItem()
                                        .withLabel(it.name)
                                        .withValue(it.value.toInt())
                                }))
                    .withDefaultValue(
                        templateParameter.defaultValue?.let {
                            GdDesignerCreativeParameterValue()
                                .withValueSelect(it.toInt())
                        })
            }
            GdDesignerTemplateParameterType.STRING -> {
                parameter
                    .withStringParameter(
                        GdDesignerTemplateParameterString()
                            .withIsURL(isUrlParameter(templateParameter.type))
                            .withMinLength(templateParameter.validation.min ?: 0)
                            .withMaxLength(templateParameter.validation.max ?: 4000))
                    .withDefaultValue(
                        templateParameter.defaultValue?.let {
                            GdDesignerCreativeParameterValue()
                                .withValueString(it)
                        })
            }
            else -> {
                logger.warn("unknown type ${parameter.type}")
            }
        }
        return parameter
    }

    private fun convertTemplateParameterControl(control: ControlType): GdDesignerTemplateControlType {
        return when (control) {
            ControlType.CHECK -> GdDesignerTemplateControlType.CHECK
            ControlType.COLORPICKER -> GdDesignerTemplateControlType.COLORPICKER
            ControlType.FILE -> GdDesignerTemplateControlType.FILE
            ControlType.SELECT -> GdDesignerTemplateControlType.SELECT
            ControlType.TEXT -> GdDesignerTemplateControlType.TEXT
            ControlType.TEXTAREA -> GdDesignerTemplateControlType.TEXTAREA
            else -> {
                logger.warn("unknown control ${control}, using text")
                GdDesignerTemplateControlType.TEXT
            }
        }
    }

    private fun convertTemplate(template: Template, clientId: ClientId): GdDesignerTemplate {
        val convertedParameters = template.parameters!!.map { convertTemplateParameter(it) }
        return GdDesignerTemplate()
            .withId(template.id)
            .withName(template.name)
            .withParameters(convertedParameters)
            .withEditInDna(canOpenWithNewDesigner(template, clientId))
    }

    private fun isAdDesignerTemplate(template: Template): Boolean {
        return OLD_AND_NEW_DESIGNER_TEMPLATES.contains(template.id)
    }

    private fun isAvailableToClient(template: Template, clientId: ClientId): Boolean {
        val features = featureService.getEnabledForClientId(clientId)
        return when (template.id) {
            1013 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1013_ENABLED.getName())
            1042 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1042_ENABLED.getName())
            1058 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1058_ENABLED.getName())
            1062 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1062_ENABLED.getName())
            1063 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1063_ENABLED.getName())
            1069 -> features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1069_ENABLED.getName())
            else -> false
        }
    }

    private fun canOpenWithNewDesigner(template: Template, clientId: ClientId): Boolean {
        if (!template.parameters!!.any { it.name == NAME_PARAMETER }) {
            return false
        }
        val features = featureService.getEnabledForClientId(clientId)
        if (template.id == 1063) {
            return true
        }
        if (template.id == 1069) {
            return true
        }
        if (template.id == 1058 && features.contains(FeatureName.BANNERSTORAGE_TEMPLATE_1058_NEW_INTERFACE.getName())) {
            return true
        }
        return false
    }

    /**
     * Ручка возвращает все шаблоны, доступные текущему клиенту, в том числе и те, которые
     * пока поддерживаются только старым конструктором
     */
    @GraphQLQuery(name = "templates")
    open fun templates(@GraphQLRootContext context: GridGraphQLContext,
                       @GraphQLArgument(name = "input") templateRequest: @GraphQLNonNull GdDesignerTemplateContainer
    ): @GraphQLNonNull GdDesignerTemplateResponse {
        val clientId = context.subjectUser!!.clientId

        try {
            val templates =
                if (templateRequest.templateId != null) {
                    val template = bannerStorageClient.getTemplate(templateRequest.templateId, TemplateInclude.PARAMETERS)
                    listOf(template)
                } else {
                    bannerStorageClient.getTemplates(TemplateInclude.PARAMETERS)
                }

            return GdDesignerTemplateResponse()
                .withTemplates(
                    templates
                        .filter { isAdDesignerTemplate(it) }
                        .filter { isAvailableToClient(it, clientId) }
                        .map { convertTemplate(it, clientId) }
                )
        } catch (e: BannerStorageClientException) {
            logger.error("Request to BannerStorage failed, fallback to no templates", e);
            return GdDesignerTemplateResponse().withTemplates(emptyList());
        }
    }

    @GraphQLQuery(name = "creative")
    open fun creative(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") creativeRequest: @GraphQLNonNull GdDesignerCreativeContainer
    ): GdDesignerCreative? {
        val creatives: List<Creative>
        try {
            creatives = bannerStorageClient.getCreatives(listOf(creativeRequest.creativeId), CreativeInclude.PREVIEW)
        } catch (e: BsCreativeNotFoundException) {
            logger.error("Creative ${creativeRequest.creativeId} was not found in BannerStorage")
            return null
        }

        val creative = creatives[0]
        return GdDesignerCreative()
            .withId(creative.id)
            .withTemplateId(creative.templateId)
            .withParameters(convertParameters(creative))
            .withPreview(
                GdDesignerCreativePreview()
                    .withWidth(creative.preview.width)
                    .withHeight(creative.preview.height)
                    .withPreviewHtml(creative.preview.code)
                    .withPreviewUrl(creative.preview.url)
            )
    }

    private fun convertFile(file: File): GdDesignerCreativeParameterValueFile {
        return GdDesignerCreativeParameterValueFile()
            .withId(file.id)
            .withName(file.fileName)
            .withUrl(file.stillageFileUrl)
            .withWidth(file.width)
            .withHeight(file.height)
    }

    private fun convertParameter(parameter: Parameter): GdDesignerCreativeParameter {
        val convertedValues =
            when (parameter.paramType) {
                ParameterType.FILE -> {
                    // Можно сделать здесь массовую ручку для получения всех файлов одним запросом
                    parameter.values.map {
                        GdDesignerCreativeParameterValue()
                            .withValueFile(if (it != null) convertFile(bannerStorageClient.getFile(it.toInt())) else null)
                    }
                }
                ParameterType.SELECT -> {
                    parameter.values.map { GdDesignerCreativeParameterValue().withValueSelect(it?.toInt()) }
                }
                ParameterType.FLAG -> {
                    parameter.values.map {
                        GdDesignerCreativeParameterValue()
                            .withValueFlag(if (it != null) (it == FLAG_TRUE_VALUE) else null)
                    }
                }
                ParameterType.INT -> {
                    parameter.values.map { str -> GdDesignerCreativeParameterValue().withValueInt(str) }
                }
                ParameterType.STRING,
                ParameterType.CLICK,
                ParameterType.PIXEL,
                ParameterType.ALT,
                ParameterType.SINGLELINE,
                ParameterType.HTTPURL,
                ParameterType.SERVERPIXEL -> {
                    parameter.values.map { str -> GdDesignerCreativeParameterValue().withValueString(str) }
                }
                else -> throw IllegalStateException("unknown type")
            }
        return GdDesignerCreativeParameter()
            .withName(parameter.paramName)
            .withValues(convertedValues)
    }

    private fun convertParameters(creative: Creative): List<GdDesignerCreativeParameter> {
        return creative.parameters.map { convertParameter(it) }
    }

    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "saveCreative")
    open fun saveCreative(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") creativeInput: @GraphQLNonNull GdDesignerSaveCreative
    ): @GraphQLNonNull GdDesignerSaveCreativePayload {
        val clientId = context.subjectUser!!.clientId

        val locale: Locale = HttpUtil.getCurrentLocale().orElse(translationService.getLocale())

        val id = creativeInput.id
        if (id != null) {
            // Можно добавить параметр dontCheckConflicts и элиминировать один вызов
            val creatives = bannerStorageClient.getCreatives(listOf(id))
            val oldCreative = creatives[0]

            try {
                val updatedCreative = bannerStorageClient.updateCreative(
                    Creative()
                        .withId(id)
                        .withName(extractName(creativeInput))
                        .withVersion(oldCreative.version)
                        .withParameters(extractParameters(creativeInput))
                        .withTnsArticles(TNS_ARTICLES)
                        .withTnsBrands(TNS_BRANDS),
                    locale,
                    CreativeInclude.PREVIEW
                )

                val existingCreative = creativeService.get(clientId, listOf(id.toLong()), listOf(CreativeType.BANNERSTORAGE))[0]
                val creative = existingCreative
                    .withName(updatedCreative.name)
                    .withWidth(updatedCreative.width.toLong())
                    .withHeight(updatedCreative.height.toLong())
                    .withPreviewUrl(updatedCreative.screenshotUrl ?: updatedCreative.thumbnailUrl)
                    .withLivePreviewUrl(updatedCreative.preview.url)
                    .withVersion(updatedCreative.version.toLong())
                    .withIsBannerstoragePredeployed(updatedCreative.isPredeployed)
                    .withModerationInfo(
                        ModerationInfo()
                            .withHtml(ModerationInfoHtml().withUrl(updatedCreative.preview.url))
                    )
                    .withDuration(extractDuration(updatedCreative))
                    .withStatusModerate(StatusModerate.NEW)

                val saveResult = creativeService.createOrUpdate(listOf(creative), clientId)
                if (!saveResult.isSuccessful || saveResult.validationResult.hasAnyErrors()) {
                    throw RuntimeException("Creatives save failed")
                }

                return GdDesignerSaveCreativePayload()
                    .withCreativeId(updatedCreative.id)
                    .withErrors(emptyList())
            } catch (e: BsCreativeValidationFailedException) {
                if (e.errorResponse != null) {
                    return extractValidationErrors(e.errorResponse!!)
                } else {
                    logger.error("Error when updating creative", e)
                    throw e
                }
            }
        } else {
            try {
                val createdCreative = bannerStorageClient.createCreative(
                    Creative()
                        .withTemplateId(creativeInput.templateId)
                        .withName(extractName(creativeInput))
                        .withParameters(extractParameters(creativeInput))
                        .withTnsArticles(TNS_ARTICLES)
                        .withTnsBrands(TNS_BRANDS),
                    locale,
                    CreativeInclude.PREVIEW
                )

                val creative = CoreCreative()
                    .withId(createdCreative.id.toLong())
                    .withClientId(clientId.asLong())
                    .withName(createdCreative.name)
                    .withType(CreativeType.BANNERSTORAGE)
                    .withPreviewUrl(createdCreative.screenshotUrl ?: createdCreative.thumbnailUrl)
                    .withLivePreviewUrl(createdCreative.preview.url)
                    .withWidth(createdCreative.width.toLong())
                    .withHeight(createdCreative.height.toLong())
                    .withStockCreativeId(createdCreative.id.toLong())
                    .withTemplateId(createdCreative.templateId.toLong())
                    .withVersion(createdCreative.version.toLong())
                    .withLayoutId(null)
                    .withThemeId(null)
                    .withBusinessType(null)
                    .withGroupName(null)
                    .withCreativeGroupId(null)
                    .withIsBannerstoragePredeployed(createdCreative.isPredeployed)
                    .withIsAdaptive(false)
                    .withIsBrandLift(false)
                    .withHasPackshot(false)
                    .withModerateTryCount(0)
                    .withModerationInfo(
                        ModerationInfo()
                            .withHtml(ModerationInfoHtml().withUrl(createdCreative.preview.url))
                    )
                    .withDuration(extractDuration(createdCreative))
                    .withStatusModerate(StatusModerate.NEW)

                val saveResult = creativeService.createOrUpdate(listOf(creative), clientId)
                if (!saveResult.isSuccessful || saveResult.validationResult.hasAnyErrors()) {
                    throw RuntimeException("Creatives save failed")
                }

                return GdDesignerSaveCreativePayload()
                    .withCreativeId(createdCreative.id)
                    .withErrors(emptyList())
            } catch (e: BsCreativeValidationFailedException) {
                if (e.errorResponse != null) {
                    return extractValidationErrors(e.errorResponse!!)
                } else {
                    logger.error("Error when creating creative", e)
                    throw e
                }
            }
        }
    }

    private fun extractDuration(creative: Creative): Long? {
        if (!creative.parameters.any { it.paramName == "VIDEO" }) {
            return null
        }
        val videoParameter = creative.parameters.first { it.paramName == "VIDEO" }
        if (videoParameter.values.isEmpty()) {
            return null
        }
        val fileId = videoParameter.values.first().toInt()
        return bannerStorageClient.getFile(fileId).duration?.roundToLong()
    }

    private fun extractValidationErrors(errorResponse: ErrorResponse): GdDesignerSaveCreativePayload {
        val errors = ArrayList<GdDesignerSaveCreativeError>()
        if (!errorResponse.data.message.isEmpty()) {
            errors.add(
                GdDesignerSaveCreativeError()
                    .withParameterName(null)
                    .withParameterValueIndex(null)
                    .withErrorMessage(errorResponse.data.message)
            )
        }
        for (parameterError in errorResponse.data.parameterErrors) {
            if (parameterError.valueIndexes.isEmpty()) {
                errors.add(
                    GdDesignerSaveCreativeError()
                        .withParameterName(parameterError.parameter)
                        .withParameterValueIndex(null)
                        .withErrorMessage(parameterError.message)
                )
            } else {
                for (valueIndex in parameterError.valueIndexes) {
                    errors.add(
                        GdDesignerSaveCreativeError()
                            .withParameterName(parameterError.parameter)
                            .withParameterValueIndex(valueIndex)
                            .withErrorMessage(parameterError.message)
                    )
                }
            }
        }
        return GdDesignerSaveCreativePayload()
            .withErrors(errors)
    }

    private fun extractParameters(creativeInput: GdDesignerSaveCreative): List<Parameter> {
        return creativeInput.parameters
            .map {
                Parameter()
                    .withParamName(it.name)
                    .withValues(it.values.map { value ->
                        ObjectUtils.firstNonNull(
                            value.valueString,
                            value.valueInt,
                            value.valueFlag,
                            value.valueSelect,
                            value.valueFile?.id)?.toString()
                    })
            }
    }

    private fun extractName(creativeInput: GdDesignerSaveCreative): String {
        return creativeInput.parameters.find { it.name == NAME_PARAMETER }?.values?.firstOrNull()?.valueString ?: ""
    }
}
