package ru.yandex.direct.grid.processing.service.creative

import ru.yandex.direct.core.entity.creative.model.Creative as CoreCreative
import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLNonNull
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.BannerStorageClient.SMART_TGO_LAYOUT_ID
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException
import ru.yandex.direct.bannerstorage.client.Utils
import ru.yandex.direct.bannerstorage.client.Utils.getSmartCreativeDisplayHeight
import ru.yandex.direct.bannerstorage.client.Utils.getSmartCreativeDisplayWidth
import ru.yandex.direct.bannerstorage.client.model.*
import ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstants.FEEDS_CREATIVES_COMPATIBILITY
import ru.yandex.direct.core.entity.creative.model.CreativeBusinessType
import ru.yandex.direct.core.entity.creative.model.CreativeType
import ru.yandex.direct.core.entity.creative.model.StatusModerate
import ru.yandex.direct.core.entity.creative.repository.BannerStorageDictRepository
import ru.yandex.direct.core.entity.creative.service.CreativeService
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter
import ru.yandex.direct.core.entity.feed.model.BusinessType
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesBusinessType
import ru.yandex.direct.grid.model.feed.GdBusinessType
import ru.yandex.direct.grid.model.feed.GdFeed
import ru.yandex.direct.grid.model.feed.GdSource
import ru.yandex.direct.grid.model.feed.GdUpdateStatus
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.creative.GdOffersPreviewRequest
import ru.yandex.direct.grid.processing.model.creative.GdOffersPreviewResponse
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeGroup
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeGroupContainer
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeGroupPayload
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeGroupPreviewRequest
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeLogo
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativePreview
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativePreviewResponse
import ru.yandex.direct.grid.processing.model.creative.GdSmartTheme
import ru.yandex.direct.grid.processing.model.creative.GdSmartUpdateCreativeGroup
import ru.yandex.direct.grid.processing.service.feed.FeedConverter
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.StringUtils
import java.time.LocalDateTime

/**
 * Graphql-ручки для конструктора performance-креативов
 */
@GridGraphQLService
open class CreativeConstructorGraphQlService @Autowired constructor(
    private val bannerStorageClient: BannerStorageClient,
    private val feedService: FeedService,
    private val bannerStorageDictRepository: BannerStorageDictRepository,
    private val creativeService: CreativeService,
    private val offersPreviewService: OffersPreviewService) {
    companion object {
        private val logger = LoggerFactory.getLogger(CreativeConstructorGraphQlService::class.java)

        // Тематики по номерам шаблонов bannerstorage
        private val TEMPLATE_THEMES = mapOf(
            740 to GdSmartTheme.RETAIL,
            741 to GdSmartTheme.HOTELS,
            778 to GdSmartTheme.REALTY,
            779 to GdSmartTheme.AUTOMOBILES,
            783 to GdSmartTheme.AIRLINE_TICKETS,
            839 to GdSmartTheme.CLOTHES,
            910 to GdSmartTheme.OTHER,
            1051 to GdSmartTheme.PHARM,
        )

        // Обратная map
        private val THEME_TEMPLATES = TEMPLATE_THEMES.entries.associate { (k, v) -> v to k }

        // Логотип по умолчанию (такой же файл используется и в AdDesigner'e)
        private const val DEFAULT_LOGO_FILE_ID = 2449921

        // Номер макета для креатива, превью которого возвращается первым
        private const val FIRST_PREVIEW_LAYOUT_ID = 29

        // Номер макета для плитки 2х2
        private const val TILE_2x2_LAYOUT_ID = 49

        // TNS категории и бренды для смартов всегда одни и те же
        private val TNS_ARTICLES = listOf(TnsArticle(-3))
        private val TNS_BRANDS = listOf(TnsBrand(1))

        // Регулярное выражение для извлечения домена из пользовательского ввода
        // Используется регулярное выражение, а не стандартные механизмы разбора URL/URI,
        // чтобы входные данные парсились так же, как при предварительной валидации на фронте, и схема была опциональной
        private val DOMAIN_REGEX = Regex("\\s*(?:https?://)?((?:[а-яёa-z-\\d]+\\.)+[а-яёa-z]+).*", RegexOption.IGNORE_CASE)
    }

    /**
     * Получение набора превьюшек (html+размеры) для всех креативов группы
     * (без разницы, уже созданной или ещё нет)
     */
    @GraphQLQuery(name = "previews")
    open fun creativeGroupPreview(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") previewRequest: @GraphQLNonNull GdSmartCreativeGroupPreviewRequest
    ): @GraphQLNonNull GdSmartCreativePreviewResponse {
        val theme = previewRequest.theme
        val templateId = THEME_TEMPLATES[theme] ?: error("unknown theme $theme")
        val templateInfo = bannerStorageDictRepository.getTemplateById(templateId)
        val businessType = CreativeBusinessType.fromSource(PerfCreativesBusinessType.valueOf(templateInfo.businessType))!!

        val clientId = context.subjectUser!!.clientId
        val feeds = feedService.getFeeds(clientId, FeedQueryFilter.newBuilder().build()).asSequence()
            .filter { it.offerExamples != null && it.offerExamples.isNotEmpty() }
            .filter { isCompatible(it.businessType, businessType) }
            .sortedBy { it.id }
            .toList()

        val dcParams: String?
        dcParams = if (previewRequest.feedId != null) {
            feeds.find { it.id == previewRequest.feedId }?.offerExamples
        } else if (feeds.isNotEmpty()) {
            feeds[0].offerExamples
        } else null

        val parameters = extractParameters(previewRequest)

        val previews = if (previewRequest.id != null) {
            val smartCreativeGroup = bannerStorageClient.getSmartCreativeGroup(previewRequest.id)
            val creativeIds = smartCreativeGroup.creatives.map { it.id }
            bannerStorageClient.getSmartGroupPreviews(creativeIds, parameters, dcParams)
        } else {
            bannerStorageClient.getSmartGroupPreviews(templateId, parameters, dcParams)
        }

        val previewsList = sortForDesigner(previews).asSequence()
            .drop(previewRequest.offset ?: 0)
            .take(previewRequest.limit ?: 100)
            .map {
                GdSmartCreativePreview()
                    .withPreviewHtml(it.preview)
                    .withIsSmartTGO(it.layoutId == SMART_TGO_LAYOUT_ID)
                    .withWidth(it.width)
                    .withHeight(it.height)
                    .withDisplayWidth(getSmartCreativeDisplayWidth(it.width, it.layoutId))
                    .withDisplayHeight(getSmartCreativeDisplayHeight(it.height, it.layoutId))
            }.toList()

        return GdSmartCreativePreviewResponse()
            .withFeeds(addFakeFeedAndConvert(feeds))
            .withPreviews(previewsList)
            .withPreviewsTotal(previews.size)
    }

    /**
     * Сортирует превьюшки в правильном порядке (для фронта)
     */
    private fun sortForDesigner(previewsList: List<Preview>): List<Preview> {
        if (previewsList.size < 2) {
            // Нечего сортировать
            return previewsList
        }

        val sorted = ArrayList<Preview>()

        // Первым должна идти превьюшка "Несколько товаров с одним описанием и мозаикой" с размерами 240х400
        // Если такой нет, то смарт-плитка 2х2 (она наиболее близка по размерам к 240х400)
        // Если же и такой не окажется (хотя сейчас такого нет), то любая плитка
        val first240x400 = previewsList.firstOrNull {
            it.layoutId == FIRST_PREVIEW_LAYOUT_ID && it.width == 240 && it.height == 400
        }
        if (first240x400 != null) {
            sorted.add(first240x400)
        } else {
            val some240x400 = previewsList.firstOrNull { it.width == 240 && it.height == 400 }
            if (some240x400 != null) {
                sorted.add(some240x400)
            } else {
                val first2x2Tile = previewsList.firstOrNull { it.layoutId == TILE_2x2_LAYOUT_ID }
                if (first2x2Tile != null) {
                    sorted.add(first2x2Tile)
                }
            }
        }

        // Второй кладём Смарт-ТГО
        val smartTGO = previewsList.firstOrNull { it.layoutId == SMART_TGO_LAYOUT_ID && !sorted.contains(it) }
        if (smartTGO != null) {
            sorted.add(smartTGO)
        }

        // Дальше идёт вся плитка в детерминированном порядке: по layoutId
        val tilePreviews = previewsList
            .filter { Utils.isSmartTile(it.layoutId) && !sorted.contains(it) }
            .sortedBy { it.layoutId }
        sorted.addAll(tilePreviews)

        // И уже потом всё оставшееся, тоже упорядоченное по layoutId+codeId
        val remaining = previewsList
            .filter { !sorted.contains(it) }
            .sortedWith(compareBy(Preview::layoutId).thenBy(Preview::codeId))
        sorted.addAll(remaining)

        check(sorted.size == previewsList.size)
        return sorted
    }

    private fun extractParameters(saveRequest: GdSmartUpdateCreativeGroup): List<Parameter> {
        return listOf(
            Parameter()
                .withParamName("LOGO")
                .withValues(saveRequest.logoFileId?.let { listOf(it.toString()) } ?: emptyList()),
            Parameter()
                .withParamName("DOMAIN_LIST")
                .withValues(listOf(parseDomain(saveRequest.domain))),
            Parameter()
                .withParamName("BUY_BUTTON_TEXT")
                .withValues(listOf(saveRequest.buttonText ?: "Купить")),
            Parameter()
                .withParamName("BUTTON_TEXT_COLOR")
                .withValues(listOf(saveRequest.buttonTextColor ?: "#000000")),
            Parameter()
                .withParamName("BUTTON_COLOR")
                .withValues(listOf(saveRequest.buttonColor ?: "#FFDB4D")),
        )
    }

    private fun parseDomain(domain: String): String {
        return DOMAIN_REGEX.matchEntire(domain)?.groupValues?.getOrNull(1) ?: ""
    }

    private fun addFakeFeedAndConvert(feeds: List<Feed>): List<GdFeed> {
        val list = ArrayList<GdFeed>()
        //передаем operatorCanWrite false, так как тут проверка доступа не нужна
        list.addAll(
            feeds.map {
                FeedConverter.convertFeedToGd(it, null, emptyList(), false)
            }
        )
        list.add(
            GdFeed()
                .withId(0)
                .withName("default")
                // Эти параметры фида выставляем для соответствия схеме
                .withBusinessType(GdBusinessType.OTHER)
                .withUpdateStatus(GdUpdateStatus.DONE)
                .withSource(GdSource.FILE)
                .withHasPassword(false)
                .withLastChange(LocalDateTime.now())
                .withFetchErrorsCount(0)
                .withIsReadOnly(true)
                .withIsRemoveUtm(false)
                .withCampaigns(emptyList())
        )
        return list
    }

    private fun isCompatible(feedBusinessType: BusinessType,
                             creativeBusinessType: CreativeBusinessType): Boolean {
        return FEEDS_CREATIVES_COMPATIBILITY[feedBusinessType]?.contains(creativeBusinessType) ?: false
    }

    /**
     * Получение превью оферов по id фидов или по ecom доменам
     */
    @GraphQLQuery(name = "offersPreview")
    open fun offersPreview(
        @GraphQLRootContext context: GridGraphQLContext?,
        @GraphQLArgument(name = "input") previewRequest: @GraphQLNonNull GdOffersPreviewRequest
    ): GdOffersPreviewResponse {
        val clientId = context?.subjectUser!!.clientId
        val feedIds = previewRequest.feedIds ?: emptyList()
        val urls = previewRequest.urls ?: emptyList()

        if (feedIds.isEmpty() && urls.isEmpty()) {
            return GdOffersPreviewResponse()
                .withResults(emptyList())
        }

        val offersPreviewForFeedIds = offersPreviewService.getOffersPreviewByFeedIds(clientId, feedIds)
        val offersPreviewForUrls = offersPreviewService.getOffersPreviewByUrls(urls)

        return GdOffersPreviewResponse()
            .withResults(offersPreviewForFeedIds + offersPreviewForUrls)
    }

    /**
     * Получение группы креативов по id группы креативов или по id креатива/креативов внутри группы
     * (полезно для случаев, когда id группы заранее не известен)
     */
    @GraphQLQuery(name = "creativeGroup")
    open fun creativeGroup(
        @GraphQLRootContext context: GridGraphQLContext?,
        @GraphQLArgument(name = "input") creativeGroupInput: @GraphQLNonNull GdSmartCreativeGroupContainer
    ): GdSmartCreativeGroup? {
        val groupId: Int?
        if (creativeGroupInput.id != null) {
            groupId = creativeGroupInput.id
        } else {
            val creativeId = creativeGroupInput.creativeId
            if (creativeId == null) {
                logger.warn("Neither id nor creativeId is not specified")
                return null
            }
            val creatives = bannerStorageClient.getCreatives(listOf(creativeId))
            if (creatives.isEmpty()) {
                logger.warn("Creative group by creativeId $creativeId not found")
                return null
            }
            groupId = creatives[0].groupId
            if (groupId == null) {
                logger.warn("Creative $creativeId has no groupId")
                return null
            }
        }

        val creativeGroup: CreativeGroup
        try {
            creativeGroup = bannerStorageClient.getSmartCreativeGroup(groupId, CreativeInclude.LAYOUT_CODE)
        } catch (e: BannerStorageClientException) {
            logger.error("Error when trying to get creative group $groupId", e)
            return null
        }
        val creatives = creativeGroup.creatives
        val templateId = creatives[0].templateId

        val smartCreativeGroup = GdSmartCreativeGroup()
            .withId(groupId)
            .withName(creativeGroup.name)
            .withTheme(TEMPLATE_THEMES[templateId])
            .withDomain(getSingleParameterValue(creatives, "DOMAIN_LIST"))
            .withButtonText(getSingleParameterValue(creatives, "BUY_BUTTON_TEXT"))
            .withButtonTextColor(getSingleParameterValue(creatives, "BUTTON_TEXT_COLOR"))
            .withButtonColor(getSingleParameterValue(creatives, "BUTTON_COLOR"))

        val logoFileId = StringUtils.ifNotBlank(getSingleParameterValue(creatives, "LOGO"), Integer::parseInt)
        if (logoFileId != null) {
            val logoFile = bannerStorageClient.getFile(logoFileId)
            return smartCreativeGroup.withLogo(
                GdSmartCreativeLogo()
                    .withId(logoFile.id)
                    .withName(logoFile.fileName)
                    .withWidth(logoFile.width)
                    .withHeight(logoFile.height)
                    .withUrl(logoFile.stillageFileUrl)
            )
        }

        return smartCreativeGroup
    }

    private fun getSingleParameterValue(creatives: List<Creative>,
                                        parameterName: String,
                                        default: String = ""): String {
        val values = getParameterValue(creatives, parameterName)
        check(values.isEmpty() || values.size == 1)
        return if (values.isNotEmpty()) values[0] else default
    }

    /**
     * Извлекает из группы креативов значение параметра (список строк), если оно одинаковое у всех креативов
     * Если же параметр где-то отсутствует или имеет отличное от других значение
     * (если креатив был отредактирован отдельно), то функция возвращает то значение, которое встречается чаще других
     */
    private fun getParameterValue(creatives: List<Creative>, parameterName: String): List<String> {
        val allVariants: List<List<String>> = creatives.asSequence()
            .filter { !Utils.isPerformanceLayoutObsolete(it.layoutCode?.layoutId) }
            .map { it.parameters.find { it.paramName == parameterName } }
            .map { it?.values }
            .filterNotNull()
            .toList()
        if (allVariants.isEmpty()) {
            return emptyList()
        }
        return allVariants.groupBy { it }
            .mapValues { it.value.size }
            .entries.maxByOrNull { it.value }!!
            .key
    }

    /**
     * Сохранение (создание или редактирование) группы креативов
     */
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "saveCreativeGroup")
    open fun saveCreativeGroup(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") creativeGroupInput: @GraphQLNonNull GdSmartUpdateCreativeGroup
    ): GdSmartCreativeGroupPayload {
        val parameters = extractParameters(creativeGroupInput)
        val name = creativeGroupInput.name!!

        val clientId = context.subjectUser!!.clientId

        val result: CreativeGroup
        val massResult: MassResult<Long>
        if (creativeGroupInput.id != null) {
            val creativeGroup = bannerStorageClient.getSmartCreativeGroup(creativeGroupInput.id)
            result = bannerStorageClient.editSmartCreativeGroup(
                creativeGroupInput.id,
                CreativeGroup(
                    creativeGroupInput.id,
                    name,
                    creativeGroup.creatives
                        .map {
                            Creative()
                                .withId(it.id)
                                .withVersion(it.version)
                                .withParameters(parameters)
                                .withTnsArticles(TNS_ARTICLES)
                                .withTnsBrands(TNS_BRANDS)
                        }
                ))

            val creativesById = result.creatives.associateBy { it.id }

            val existingCreatives = creativeService.get(clientId, result.creatives.map { it.id.toLong() }, listOf(CreativeType.PERFORMANCE))

            for (existingCreative in existingCreatives) {
                val creative: Creative = creativesById[existingCreative.id.toInt()]!!
                existingCreative
                    .withName(creative.name)
                    .withGroupName(result.name)
                    .withWidth(creative.width.toLong())
                    .withHeight(creative.height.toLong())
                    .withPreviewUrl(creative.screenshotUrl ?: creative.thumbnailUrl)
                    .withLivePreviewUrl(creative.preview.url)
                    .withVersion(creative.version.toLong())
                    .withModerationInfo(null)
                    .withIsBannerstoragePredeployed(creative.isPredeployed)
                    .withStatusModerate(StatusModerate.NEW)
            }

            massResult = creativeService.createOrUpdate(existingCreatives, clientId)
        } else {
            val templateId = THEME_TEMPLATES[creativeGroupInput.theme] ?: error("unknown theme")
            val template = bannerStorageClient.getTemplate(templateId, TemplateInclude.LAYOUT_CODES)
            result = bannerStorageClient.createSmartCreativeGroup(
                CreativeGroup(
                    null,
                    name,
                    template.layoutCodes!!.map {
                        Creative()
                            .withTemplateId(templateId)
                            .withLayoutCodeId(it.id)
                            .withParameters(parameters)
                            .withTnsArticles(TNS_ARTICLES)
                            .withTnsBrands(TNS_BRANDS)
                    }
                )
            )

            val creativesToAdd = result.creatives.map { creative ->
                CoreCreative()
                    .withId(creative.id.toLong())
                    .withClientId(clientId.asLong())
                    .withName(creative.name)
                    .withType(CreativeType.PERFORMANCE)
                    .withStatusModerate(StatusModerate.NEW)
                    .withPreviewUrl(creative.screenshotUrl ?: creative.thumbnailUrl)
                    .withLivePreviewUrl(creative.preview.url)
                    .withWidth(creative.width.toLong())
                    .withHeight(creative.height.toLong())
                    .withStockCreativeId(creative.id.toLong())
                    .withLayoutId(creative.layoutCode.layoutId.toLong())
                    .withTemplateId(creative.templateId.toLong())
                    .withVersion(creative.version.toLong())
                    .withThemeId(creative.layoutCode.themeId.toLong())
                    .withBusinessType(CreativeBusinessType.fromSource(PerfCreativesBusinessType.valueOf(creative.businessType.directId)))
                    .withGroupName(result.name)
                    .withCreativeGroupId(result.id!!.toLong())
                    .withIsBannerstoragePredeployed(creative.isPredeployed)
                    .withIsAdaptive(false)
                    .withIsBrandLift(false)
                    .withHasPackshot(false)
                    .withModerateTryCount(0)
            }

            massResult = creativeService.createOrUpdate(creativesToAdd, clientId)
        }

        val errors = massResult.errors
        if (errors.isNotEmpty()) {
            logger.warn(errors.toString())
        }
        if (!massResult.isSuccessful) {
            throw RuntimeException("Creatives save failed")
        }

        return GdSmartCreativeGroupPayload().withId(result.id)
    }
}
