package ru.yandex.direct.grid.processing.service.creative;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bannerstorage.client.BannerStorageClient;
import ru.yandex.direct.bannerstorage.client.BannerStorageClientException;
import ru.yandex.direct.bannerstorage.client.model.File;
import ru.yandex.direct.bannerstorage.client.model.Parameter;
import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.canvas.client.model.video.GenerateConditions;
import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.creative.model.BannerStorageDictLayoutItem;
import ru.yandex.direct.core.entity.creative.model.BannerStorageDictThemeItem;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeConverter;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.BannerStorageDictRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.banner.service.GridBannerService;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.model.campaign.GdCampaignForLink;
import ru.yandex.direct.grid.model.campaign.GdSmartCampaign;
import ru.yandex.direct.grid.processing.model.cliententity.GdAvailableCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdAvailableCreativesContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdAvailableCreativesFilter;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeGroupsInfoContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeLayout;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeTheme;
import ru.yandex.direct.grid.processing.model.cliententity.GdGenerateVideoAddition;
import ru.yandex.direct.grid.processing.model.cliententity.GdSmartCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdTypedCreative;
import ru.yandex.direct.grid.processing.model.creative.GdSmartCreativeLogo;
import ru.yandex.direct.grid.processing.model.group.GdSmartAdGroup;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.bannerstorage.client.BannerStorageClient.MAX_CREATIVES_IN_BATCH;
import static ru.yandex.direct.bannerstorage.client.Utils.isPerformanceLayoutObsolete;
import static ru.yandex.direct.canvas.client.model.video.Creative.CreativeType.VIDEO_ADDITION;
import static ru.yandex.direct.core.entity.banner.type.creative.BannerWithCreativeConstraints.isConsistentCreativeGeoToAdGroupGeo;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toGdCampaignType;
import static ru.yandex.direct.grid.processing.service.client.converter.ClientEntityConverter.toGdCreativeImplementation;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CreativeDataService {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(CreativeDataService.class);

    private static final Comparator<GdSmartCreative> SMART_CREATIVE_COMPARATOR =
            Comparator.comparing(GdSmartCreative::getGroupId, Comparator.nullsFirst(Long::compareTo))
                    .thenComparing(GdSmartCreative::getCreativeId);

    private final BannerStorageClient bannerStorageClient;
    private final BannerStorageDictRepository bannerStorageDictRepository;
    private final CreativeService creativeService;
    private final FeedService feedService;
    private final ClientGeoService clientGeoService;
    private final ShardHelper shardHelper;
    private final BannerTypedRepository bannerTypedRepository;
    private final CampaignService campaignService;
    private final ClientService clientService;
    private final CanvasClient canvasClient;
    private final AdGroupService adGroupService;
    private final AdGroupRepository adGroupRepository;
    private final BannerService bannerService;
    private final GridBannerService gridBannerService;
    private final TranslationService translationService;

    public CreativeDataService(
            BannerStorageClient bannerStorageClient,
            BannerStorageDictRepository bannerStorageDictRepository,
            CreativeService creativeService,
            FeedService feedService,
            ClientGeoService clientGeoService,
            ShardHelper shardHelper,
            BannerTypedRepository bannerTypedRepository,
            CampaignService campaignService, ClientService clientService, CanvasClient canvasClient,
            AdGroupService adGroupService, AdGroupRepository adGroupRepository, BannerService bannerService,
            GridBannerService gridBannerService,
            TranslationService translationService) {
        this.bannerStorageClient = bannerStorageClient;
        this.bannerStorageDictRepository = bannerStorageDictRepository;
        this.creativeService = creativeService;
        this.feedService = feedService;
        this.clientGeoService = clientGeoService;
        this.shardHelper = shardHelper;
        this.bannerTypedRepository = bannerTypedRepository;
        this.campaignService = campaignService;
        this.clientService = clientService;
        this.canvasClient = canvasClient;
        this.adGroupService = adGroupService;
        this.adGroupRepository = adGroupRepository;
        this.bannerService = bannerService;
        this.gridBannerService = gridBannerService;
        this.translationService = translationService;
    }

    Map<Long, GdCreativeLayout> getCreativeLayouts(Collection<Long> layoutIds) {
        Map<Long, BannerStorageDictLayoutItem> layoutsByIds = bannerStorageDictRepository.getLayoutsByIds(layoutIds);

        return EntryStream.of(layoutsByIds)
                .mapValues(CreativeDataService::toGdCreativeLayout)
                .toMap();
    }

    GdAvailableCreativesContext getGdAvailableCreativeContext(ClientId clientId, GdSmartAdGroup gdSmartAdGroup,
                                                              LimitOffset range,
                                                              @Nullable GdAvailableCreativesContainer input) {
        String idOrNameLike = Optional.ofNullable(input)
                .map(GdAvailableCreativesContainer::getFilter)
                .map(GdAvailableCreativesFilter::getIdOrNameLike)
                .orElse(null);

        List<GdSmartCreative> availableCreatives =
                getAvailableCreatives(clientId, singletonList(gdSmartAdGroup), idOrNameLike).get(0);

        Order idSortOrder = ifNotNull(input, GdAvailableCreativesContainer::getIdSortOrder);
        Comparator<GdSmartCreative> comparator =
                (idSortOrder == Order.ASC) ? SMART_CREATIVE_COMPARATOR : SMART_CREATIVE_COMPARATOR.reversed();

        List<GdSmartCreative> rowset = StreamEx.of(availableCreatives)
                .sorted(comparator)
                .skip(range.offset())
                .limit(range.limit())
                .toList();
        return new GdAvailableCreativesContext().withRowset(rowset);
    }

    List<List<GdSmartCreative>> getAvailableCreatives(ClientId clientId, List<GdSmartAdGroup> gdSmartAdGroups,
                                                      @Nullable String idOrNameLike) {
        Map<Long, Feed> feedByIds = listToMap(
                feedService.getFeeds(clientId, mapList(gdSmartAdGroups, GdSmartAdGroup::getFeedId)), Feed::getId);

        Map<Long, List<Creative>> creativesByFeedIds = EntryStream.of(feedByIds)
                .mapValues(feed -> creativeService.getCreativesWithBusinessType(clientId, feed.getBusinessType(),
                        idOrNameLike))
                .toMap();
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        return StreamEx.of(gdSmartAdGroups)
                .mapToEntry(identity(),
                        smartAdGroup -> creativesByFeedIds.getOrDefault(smartAdGroup.getFeedId(), emptyList()))
                .mapKeyValue((smartAdGroup, creatives) -> filterAvailableCreatives(geoTree, creatives, smartAdGroup))
                .toList();
    }

    private List<GdSmartCreative> filterAvailableCreatives(GeoTree geoTree, List<Creative> creatives,
                                                           GdSmartAdGroup gdSmartAdGroup) {
        return StreamEx.of(creatives)
                .filter(creative -> isConsistentCreativeGeoToAdGroupGeo(singletonList(creative),
                        getAdGroupCountries(geoTree, gdSmartAdGroup)))
                .map(creative -> (GdSmartCreative) toGdCreativeImplementation(creative))
                .toList();
    }

    private Set<Long> getAdGroupCountries(GeoTree geoTree, GdSmartAdGroup gdSmartAdGroup) {
        List<Long> regionsIds = StreamEx.of(gdSmartAdGroup.getRegionsInfo().getRegionIds())
                .map(Integer::longValue)
                .toList();
        return geoTree.getModerationCountries(regionsIds);
    }

    GdCreativeGroupsInfoContext getCreativeGroupsInfoContext(ClientId clientId, GdSmartAdGroup gdSmartAdGroup) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        var creatives = creativeService.getCreativesByPerformanceAdGroups(clientId, List.of(gdSmartAdGroup.getId()))
                .getOrDefault(gdSmartAdGroup.getId(), List.of());
        var creativeGroupIds = StreamEx.of(creatives)
                .map(Creative::getCreativeGroupId)
                .nonNull()
                .toSet();

        var adGroupIds = adGroupRepository.getAdGroupIdsByCreativeGroupIds(shard, creativeGroupIds);
        var relatedAdGroupIds = StreamEx.of(adGroupIds)
                .remove(gdSmartAdGroup.getId()::equals)
                .toList();

        try {
            var creativeIds = StreamEx.of(creatives)
                    .removeBy(Creative::getCreativeGroupId, null)
                    .remove(creative -> isPerformanceLayoutObsolete(creative.getLayoutId()))
                    .map(Creative::getId)
                    .map(Long::intValue)
                    .toList();
            var batches = Lists.partition(creativeIds, MAX_CREATIVES_IN_BATCH);
            var bsCreatives = flatMap(batches, bannerStorageClient::getCreatives);

            var logos = StreamEx.of(bsCreatives)
                    .flatCollection(ru.yandex.direct.bannerstorage.client.model.Creative::getParameters)
                    .filterBy(Parameter::getParamName, "LOGO")
                    .map(Parameter::getValues)
                    .flatCollection(values -> CollectionUtils.isEmpty(values) ? Collections.singletonList(null) : values)
                    .map(logoFileId -> ifNotNull(logoFileId, Integer::parseInt))
                    .distinct()
                    .map(logoFileId -> ifNotNull(logoFileId, bannerStorageClient::getFile))
                    .map(CreativeDataService::toGdSmartCreativeLogo)
                    .toList();

            return new GdCreativeGroupsInfoContext()
                    .withRelatedAdGroupIds(relatedAdGroupIds)
                    .withLogos(logos)
                    .withIsBannerStorageAvailable(true);
        } catch (BannerStorageClientException e) {
            logger.error("Request to BannerStorage has failed", e);

            return new GdCreativeGroupsInfoContext()
                    .withRelatedAdGroupIds(relatedAdGroupIds)
                    .withLogos(null)
                    .withIsBannerStorageAvailable(false);
        }
    }

    @Nullable
    private static GdSmartCreativeLogo toGdSmartCreativeLogo(@Nullable File logoFile) {
        if (logoFile == null) {
            return null;
        }
        return new GdSmartCreativeLogo()
                .withId(logoFile.getId())
                .withName(logoFile.getFileName())
                .withWidth(logoFile.getWidth())
                .withHeight(logoFile.getHeight())
                .withUrl(logoFile.getStillageFileUrl());
    }

    @Nullable
    private static GdCreativeLayout toGdCreativeLayout(@Nullable BannerStorageDictLayoutItem layoutItem) {
        if (layoutItem == null) {
            return null;
        }
        return new GdCreativeLayout()
                .withId(layoutItem.getId())
                .withImgSrc(layoutItem.getImgSrc())
                .withName(layoutItem.getName());
    }

    Map<Long, GdCreativeTheme> getCreativeThemes(Collection<Long> themeIds) {
        Map<Long, BannerStorageDictThemeItem> themesByIds = bannerStorageDictRepository.getThemesByIds(themeIds);

        return EntryStream.of(themesByIds)
                .mapValues(CreativeDataService::toGdCreativeTheme)
                .toMap();
    }

    @Nullable
    private static GdCreativeTheme toGdCreativeTheme(@Nullable BannerStorageDictThemeItem layoutItem) {
        if (layoutItem == null) {
            return null;
        }
        return new GdCreativeTheme()
                .withId(layoutItem.getId())
                .withName(layoutItem.getName());
    }

    Map<Long, List<GdCampaignForLink>> getUsedInCampaigns(ClientId clientId, Collection<Long> creativeIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, List<BannerWithCreative>> bannersByCreativeIds =
                getBannersByCreativeIds(shard, clientId, creativeIds);

        Set<Long> campaignIds = StreamEx.ofValues(bannersByCreativeIds)
                .flatMap(Collection::stream)
                .map(BannerWithCreative::getCampaignId)
                .toSet();
        Map<Long, Campaign> campaignsById =
                listToMap(campaignService.getCampaigns(clientId, campaignIds), Campaign::getId);

        return EntryStream.of(bannersByCreativeIds)
                .mapValues(banners -> StreamEx.of(banners)
                        .map(BannerWithCreative::getCampaignId)
                        .distinct()
                        .map(campaignsById::get)
                        .map(CreativeDataService::toGdCampaignForLink)
                        .toList())
                .toMap();
    }

    /**
     * Получить мапу creativeId -> List<BannerWithCreative> по коллекции creativeIds
     */
    private Map<Long, List<BannerWithCreative>> getBannersByCreativeIds(int shard, ClientId clientId,
                                                                        Collection<Long> creativeIds) {
        List<BannerWithCreative> bannersWithCreative =
                bannerTypedRepository.getBannersByCreativeIds(shard, clientId, creativeIds,
                        BannerWithCreative.class);

        return StreamEx.of(bannersWithCreative)
                .mapToEntry(BannerWithCreative::getCreativeId, identity())
                .grouping();
    }

    private static GdCampaignForLink toGdCampaignForLink(Campaign campaign) {
        // Это наиболее подходящий неабстрактный класс кампании
        GdCampaignForLink gdCampaign = new GdSmartCampaign();
        gdCampaign.setId(campaign.getId());
        gdCampaign.setName(campaign.getName());
        gdCampaign.setType(toGdCampaignType(campaign.getType()));

        return gdCampaign;
    }

    List<GdTypedCreative> generateVideoAddition(ClientId clientId, GdGenerateVideoAddition input) {
        List<Long> adGroupIds = input.getAdGroupIds();

        Map<Long, Creative> videoAdditions = generateVideoAdditions(clientId, adGroupIds);
        return StreamEx.of(adGroupIds)
                .map(id -> toGdCreativeImplementation(videoAdditions.get(id)))
                .toList();
    }

    Map<Long, Creative> generateVideoAdditions(ClientId clientId, List<Long> adGroupIds) {

        //Проверяем, что у клиента включена опция по автодобавлению видеокреативов
        Client client = clientService.getClient(clientId);
        if (client == null || !client.getAutoVideo()) {
            return emptyMap();
        }

        int shard = shardHelper.getShardByClientId(clientId);

        List<Long> remainingAdGroupIds = new ArrayList<>(adGroupRepository.getClientExistingAdGroupIds(shard,
                clientId, adGroupIds));

        //Проверяем, что запрошено добавление в текстовую группу
        Map<Long, AdGroupType> adGroupTypes = adGroupService.getAdGroupTypes(clientId, Set.copyOf(remainingAdGroupIds));
        adGroupIds.forEach(id -> {
            if (adGroupTypes.get(id) != AdGroupType.BASE) {
                remainingAdGroupIds.remove(id);
            }
        });

        if (remainingAdGroupIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, Creative> creativeByAdGroupId = getVideoCreativeForAgGroups(clientId, remainingAdGroupIds);

        creativeByAdGroupId.forEach((id, creative) -> {
            creativeByAdGroupId.put(id, creative);
            remainingAdGroupIds.remove(id);
        });

        if (remainingAdGroupIds.isEmpty()) {
            return creativeByAdGroupId;
        }

        //Если баннер первый или в предыдущих автовидео дополнение не нашлось — делаем новое
        GenerateConditions.VideoAdditionLocale locale;
        try {
            locale = GenerateConditions.VideoAdditionLocale.valueOf(translationService.getLocale().toLanguageTag());
        } catch (IllegalArgumentException | NullPointerException e) {
            locale = GenerateConditions.VideoAdditionLocale.EN_US;
        }

        List<Creative> generatedVideoAdditions = generateVideoAdditionInCanvas(clientId.asLong(),
                new GenerateConditions().withLocale(locale.getLocale())
                        .withBannerType(BannersBannerType.text.name().toLowerCase())
                        .withCategories(emptyList())
                        .withCreativeType(VIDEO_ADDITION.getValue())
                        .withCount(remainingAdGroupIds.size()));
        List<Long> generatedCreativeIds = mapList(generatedVideoAdditions, Creative::getId);
        creativeService.synchronizeVideoAdditionCreatives(shard, clientId, Set.copyOf(generatedCreativeIds));

        List<Creative> creatives = creativeService.get(clientId, generatedCreativeIds,
                List.of(CreativeType.VIDEO_ADDITION_CREATIVE));

        checkState(remainingAdGroupIds.size() == creatives.size());

        for (int i = 0; i < creatives.size(); i++) {
            creativeByAdGroupId.put(remainingAdGroupIds.get(i), creatives.get(i));
        }
        return creativeByAdGroupId;
    }

    private Map<Long, Creative> getVideoCreativeForAgGroups(ClientId clientId, List<Long> adGroupIds) {
        Map<Long, List<BannerWithSystemFields>> bannersByAdGroups =
                bannerService.getBannersByAdGroupIds(adGroupIds);
        int shard = shardHelper.getShardByClientId(clientId);
        return StreamEx.of(adGroupIds)
                .mapToEntry(identity(), adGroupId ->
                        getAdGroupCreative(shard, adGroupId, bannersByAdGroups))
                .nonNullValues()
                .toMap();
    }

    @Nullable
    private Creative getAdGroupCreative(int shard, Long adGroupId,
                                        Map<Long, List<BannerWithSystemFields>> bannersByAdGroups) {
        List<BannerWithSystemFields> adGroupBanners = bannersByAdGroups.get(adGroupId);
        if (isEmpty(adGroupBanners)) {
            return null;
        }
        List<Long> bannerIds = mapList(adGroupBanners, BannerWithSystemFields::getId);
        Map<Long, Creative> creatives = gridBannerService.getCreatives(shard, bannerIds);
        return creatives.values()
                .stream()
                .filter(creative -> creative.getType() == CreativeType.VIDEO_ADDITION_CREATIVE)
                .findAny()
                .orElse(null);
    }


    private List<Creative> generateVideoAdditionInCanvas(Long clientId, GenerateConditions generateConditions) {
        List<List<ru.yandex.direct.canvas.client.model.video.Creative>> generatedAdditions =
                canvasClient.generateAdditions(clientId, List.of(generateConditions));

        checkState(generatedAdditions.size() == 1,
                "Expected exactly one list of autogenerated videoadditions");

        return mapList(generatedAdditions.get(0), CreativeConverter::fromCanvasCreative);
    }
}
