package ru.yandex.direct.grid.processing.service.creative;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.campaign.GdCampaignForLink;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.cliententity.GdAvailableCreativesContainer;
import ru.yandex.direct.grid.processing.model.cliententity.GdAvailableCreativesContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeGroupsInfoContext;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeLayout;
import ru.yandex.direct.grid.processing.model.cliententity.GdCreativeTheme;
import ru.yandex.direct.grid.processing.model.cliententity.GdGenerateVideoAddition;
import ru.yandex.direct.grid.processing.model.cliententity.GdSmartCreative;
import ru.yandex.direct.grid.processing.model.cliententity.GdTypedCreative;
import ru.yandex.direct.grid.processing.model.creative.GdUpdateSmartAdGroupCreativeGroups;
import ru.yandex.direct.grid.processing.model.creative.GdUpdateSmartAdGroupCreativeGroupsPayload;
import ru.yandex.direct.grid.processing.model.group.GdSmartAdGroup;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@GridGraphQLService
public class CreativeGraphQlService {
    private final BannerStorageDictThemeDataLoader themeDataLoader;
    private final BannerStorageDictLayoutDataLoader layoutDataLoader;
    private final GridValidationService gridValidationService;
    private final CreativeDataService creativeDataService;
    private final CreativeMutationService creativeMutationService;
    private final UsedInCampaignsLoader usedInCampaignsLoader;

    public CreativeGraphQlService(
            BannerStorageDictThemeDataLoader themeDataLoader,
            BannerStorageDictLayoutDataLoader layoutDataLoader,
            GridValidationService gridValidationService,
            CreativeDataService creativeDataService,
            CreativeMutationService creativeMutationService,
            UsedInCampaignsLoader usedInCampaignsLoader) {
        this.themeDataLoader = themeDataLoader;
        this.layoutDataLoader = layoutDataLoader;
        this.gridValidationService = gridValidationService;
        this.creativeDataService = creativeDataService;
        this.creativeMutationService = creativeMutationService;
        this.usedInCampaignsLoader = usedInCampaignsLoader;
    }

    @GraphQLQuery(name = "layout")
    public CompletableFuture<GdCreativeLayout> creativeLayout(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdSmartCreative smartCreative
    ) {
        Long layoutId = smartCreative.getLayoutId();
        if (layoutId == null) {
            return CompletableFuture.completedFuture(null);
        }
        return layoutDataLoader.get().load(layoutId);
    }

    @GraphQLQuery(name = "availableCreatives")
    public GdAvailableCreativesContext getAvailableCreatives(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdSmartAdGroup gdSmartAdGroup,
            @GraphQLArgument(name = "limitOffset") GdLimitOffset oldLimitOffset,
            @GraphQLArgument(name = "input") GdAvailableCreativesContainer input) {
        GdLimitOffset limitOffset = ifNotNull(input, GdAvailableCreativesContainer::getLimitOffset);
        gridValidationService.validateLimitOffset(limitOffset);
        LimitOffset range = normalizeLimitOffset(limitOffset);

        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        return creativeDataService.getGdAvailableCreativeContext(clientId, gdSmartAdGroup, range, input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "creativeGroupsInfo")
    public GdCreativeGroupsInfoContext getCreativeGroupsInfo(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdSmartAdGroup gdSmartAdGroup) {
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        return creativeDataService.getCreativeGroupsInfoContext(clientId, gdSmartAdGroup);
    }

    @GraphQLQuery(name = "theme")
    public CompletableFuture<GdCreativeTheme> creativeTheme(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdSmartCreative smartCreative
    ) {
        Long themeId = smartCreative.getThemeId();
        if (themeId == null) {
            return CompletableFuture.completedFuture(null);
        }
        return themeDataLoader.get().load(themeId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "usedInCampaigns")
    public CompletableFuture<List<@GraphQLNonNull GdCampaignForLink>> usedInCampIds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdSmartCreative smartCreative
    ) {
        return usedInCampaignsLoader.get().load(smartCreative.getCreativeId());
    }

    @PreAuthorizeWrite
    @GraphQLMutation(name = "generateVideoAddition")
    public List<GdTypedCreative> generateVideoAddition(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGenerateVideoAddition input) {
        return creativeDataService.generateVideoAddition(context.getSubjectUser().getClientId(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateSmartAdGroupCreativeGroups")
    public GdUpdateSmartAdGroupCreativeGroupsPayload updateSmartAdGroupCreativeGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateSmartAdGroupCreativeGroups input) {
        return creativeMutationService.updateSmartAdGroupCreativeGroups(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }
}
