package ru.yandex.direct.grid.processing.service.creative;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroups;
import ru.yandex.direct.core.entity.creative.model.UpdateSmartAdGroupCreativeGroupsItem;
import ru.yandex.direct.core.entity.creative.service.BannerStorageCreativeService;
import ru.yandex.direct.core.entity.creative.service.BannerStorageCreativeValidationService;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.creative.GdSmartTheme;
import ru.yandex.direct.grid.processing.model.creative.GdUpdateSmartAdGroupCreativeGroups;
import ru.yandex.direct.grid.processing.model.creative.GdUpdateSmartAdGroupCreativeGroupsPayload;
import ru.yandex.direct.grid.processing.model.creative.GdUpdateSmartAdGroupCreativeGroupsPayloadItem;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapAndFilterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CreativeMutationService {
    private final CreativeService creativeService;
    private final BannerStorageCreativeService bannerStorageCreativeService;
    private final GridValidationService gridValidationService;
    private final BannerStorageCreativeValidationService bannerStorageCreativeValidationService;

    @Autowired
    public CreativeMutationService(CreativeService creativeService,
            BannerStorageCreativeService bannerStorageCreativeService,
            GridValidationService gridValidationService,
            BannerStorageCreativeValidationService bannerStorageCreativeValidationService) {
        this.creativeService = creativeService;
        this.bannerStorageCreativeService = bannerStorageCreativeService;
        this.gridValidationService = gridValidationService;
        this.bannerStorageCreativeValidationService = bannerStorageCreativeValidationService;
    }

    /**
     * Обновление групп креативов, привязанных к перформанс-группам объявлений
     * (может затронуть другие группы объявлений - если они используют те же группы креативов, что и обновляемые).
     * В случае отсутствия привязанных групп креативов для группы, создает группу креативов с заданными параметрами,
     * а затем создает по баннеру для каждого креатива в группе.
     * Валидируется, что нет групп креативов, связанных со сразу несколькими обновляемыми группами объявлений.
     */
    GdUpdateSmartAdGroupCreativeGroupsPayload updateSmartAdGroupCreativeGroups(ClientId clientId, Long operatorUid,
                                                                               GdUpdateSmartAdGroupCreativeGroups input) {
        UpdateSmartAdGroupCreativeGroups request = new UpdateSmartAdGroupCreativeGroups()
                .withUpdateItems(mapList(input.getUpdateItems(),
                        item -> new UpdateSmartAdGroupCreativeGroupsItem()
                                .withAdGroupId(item.getAdGroupId())
                                .withDomain(item.getDomain())
                                .withLogoFileId(item.getLogoFileId())
                                .withTheme(GdSmartTheme.toSource(item.getTheme()))
                ))
                .withSaveDraft(input.getSaveDraft());

        if (isEmpty(request.getUpdateItems())) {
            return new GdUpdateSmartAdGroupCreativeGroupsPayload()
                    .withUpdatedAdGroupItems(emptyList());
        }

        var adGroupIds = mapList(request.getUpdateItems(), UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId);
        var creativesByAdGroupId = creativeService.getCreativesByPerformanceAdGroups(clientId, adGroupIds);

        gridValidationService.applyValidator(r ->
                        bannerStorageCreativeValidationService
                                .validateAdGroupCreativeGroupsUpdateRequest(request, clientId, creativesByAdGroupId),
                request, false);

        var creativeGroupIdsByAdGroupId = EntryStream.of(creativesByAdGroupId)
                .mapValues(creatives -> mapAndFilterToSet(creatives,
                        ru.yandex.direct.core.entity.creative.model.Creative::getCreativeGroupId,
                        Objects::nonNull))
                .toMap();

        boolean saveDraft = nvl(request.getSaveDraft(), false);

        var successfullyUpdatedAdGroups = StreamEx.of(request.getUpdateItems())
                .filter(item -> bannerStorageCreativeService.updateSmartAdGroupCreativeGroupsItem(clientId, operatorUid,
                        saveDraft, item, creativeGroupIdsByAdGroupId.get(item.getAdGroupId())))
                .map(UpdateSmartAdGroupCreativeGroupsItem::getAdGroupId)
                .map(adGroupId -> new GdUpdateSmartAdGroupCreativeGroupsPayloadItem()
                    .withAdGroupId(adGroupId))
                .toList();

        return new GdUpdateSmartAdGroupCreativeGroupsPayload()
                .withUpdatedAdGroupItems(successfullyUpdatedAdGroups);
    }

}
