package ru.yandex.direct.grid.processing.service.creative

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.feed.service.FeedService
import ru.yandex.direct.core.entity.feed.service.MbiService
import ru.yandex.direct.core.entity.feed.model.FeedOfferExamples
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService
import ru.yandex.direct.currency.CurrencyCode.isRealCurrencyCode
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreview
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewBannerPrice
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewBannerText
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewClickUrls
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewImage
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewImageInfo
import ru.yandex.direct.grid.processing.model.creative.GdOfferPreviewSmartCenter
import ru.yandex.direct.grid.processing.model.creative.GdOffersPreview
import ru.yandex.direct.utils.JsonUtils
import ru.yandex.direct.validation.constraint.StringConstraints.isValidUrl
import java.math.BigDecimal
import java.math.RoundingMode
import java.text.DecimalFormat

@Service
class OffersPreviewService @Autowired constructor(
    private val feedService: FeedService,
    private val ecomDomainsService: EcomDomainsService
) {
    companion object {
        private val logger = LoggerFactory.getLogger(OffersPreviewService::class.java)
        private val DECIMAL_FORMAT = DecimalFormat()
        init {
            DECIMAL_FORMAT.maximumFractionDigits = 2
            DECIMAL_FORMAT.minimumFractionDigits = 0
            DECIMAL_FORMAT.isGroupingUsed = false
        }
    }

    fun getOffersPreviewByFeedIds(clientId: ClientId, feedIds: List<Long>): List<GdOffersPreview> {
        if (feedIds.isEmpty()) {
            return emptyList()
        }

        val feeds = feedService.getFeeds(clientId, feedIds)
            .associateBy { it.id }

        return feeds
            .filterValues { it.offerExamples != null && it.offerExamples.isNotBlank() }
            .mapValues { processFeedOffersExamples(it.value.offerExamples) }
            .map {
                GdOffersPreview()
                    .withFeedId(it.key)
                    .withPreviews(it.value)
            }
    }

    fun getOffersPreviewByUrls(urls: List<String>): List<GdOffersPreview> {
        val validUrls = urls
            .filter { it.isNotBlank() && isValidUrl(it) }

        if (validUrls.isEmpty()) {
            return emptyList()
        }

        val minOffersForEcomAllowed = ecomDomainsService.getMinOffersForEcomAllowed()
        val maxOffersForEcomAllowed = ecomDomainsService.getMaxOffersForEcomAllowed()

        return ecomDomainsService.getEcomDomainsByUrls(validUrls)
            .filterValues {
                it.previewOffers != null && it.previewOffers.isNotBlank()
                    && it.offersCount in minOffersForEcomAllowed..maxOffersForEcomAllowed
            }
            .map {
                GdOffersPreview()
                    .withUrl(it.key)
                    .withPreviews(processFeedOffersExamples(it.value.previewOffers))
                    .withStrongOffersCount(it.value.offersCount)
            }
    }

    fun processFeedOffersExamples(offerExamples: String): List<GdOfferPreview> {
        val parsedOfferExamples = JsonUtils.fromJson(offerExamples, FeedOfferExamples::class.java)

        return parsedOfferExamples.offers?.asSequence()
            ?.sortedBy { it.key }
            ?.mapNotNull { convertOffer(it.value) }
            ?.take(MbiService.OFFERS_COUNT_FOR_PREVIEW)
            ?.toList() ?: emptyList()
    }

    private fun convertOffer(offer: FeedOfferExamples.Offer): GdOfferPreview? {
        val price = convertPrice(offer.price) ?: return null
        val offerText = convertOfferText(offer.text) ?: return null

        return GdOfferPreview()
            .withTargetUrl(offer.targetUrl)
            .withText(offerText)
            .withClickUrls(convertClickUrls(offer.clickUrls))
            .withPrice(price)
            .withImages(listOf(convertImageInfo(offer.image)))
    }

    private fun convertImageInfo(imageInfo: FeedOfferExamples.ImageInfo): GdOfferPreviewImageInfo {
        return GdOfferPreviewImageInfo()
            .withSmall(convertImage(imageInfo.small))
            .withOrig(convertImage(imageInfo.orig))
            .withBig(convertImage(imageInfo.big))
            .withHuge(convertImage(imageInfo.huge))
    }

    private fun convertImage(image: FeedOfferExamples.Image): GdOfferPreviewImage {
        return GdOfferPreviewImage()
            .withWidth(image.width)
            .withHeight(image.height)
            .withUrl(image.url)
            .withSmartCenters(convertSmartCenters(image.smartCenters))
    }

    private fun convertSmartCenters(smartCenters: Map<String, FeedOfferExamples.SmartCenter>?): List<GdOfferPreviewSmartCenter> {
        if (smartCenters == null)
            return emptyList()

        return smartCenters
            .map { it.value }
            .map {
                GdOfferPreviewSmartCenter()
                    .withX(it.x)
                    .withY(it.y)
                    .withW(it.w)
                    .withH(it.h)
            }
            .toList()
    }

    private fun convertPrice(price: FeedOfferExamples.Price): GdOfferPreviewBannerPrice? {
        if (price.current == "null") {
            return null
        }

        val current = DECIMAL_FORMAT.format(BigDecimal(price.current).setScale(2, RoundingMode.DOWN))
        val old = if (price.old == null || price.old == "") {
            null
        } else {
            DECIMAL_FORMAT.format(BigDecimal(price.old).setScale(2, RoundingMode.DOWN))
        }

        return GdOfferPreviewBannerPrice()
            .withCurrent(current)
            .withOld(old)
    }

    private fun convertClickUrls(clickUrls: FeedOfferExamples.ClickUrls): GdOfferPreviewClickUrls? {
        return GdOfferPreviewClickUrls()
            .withTextName(clickUrls.textName)
            .withTextBody(clickUrls.textBody ?: clickUrls.textName)
            .withPriceCurrent(clickUrls.priceCurrent ?: clickUrls.textName)
            .withPriceOld(clickUrls.priceOld ?: clickUrls.textName)
            .withImageSmall(clickUrls.imageSmall ?: clickUrls.textName)
            .withImageOrig(clickUrls.imageOrig ?: clickUrls.textName)
            .withImageBig(clickUrls.imageBig ?: clickUrls.textName)
            .withImageHuge(clickUrls.imageHuge ?: clickUrls.textName)
    }

    private fun convertOfferText(text: FeedOfferExamples.Text): GdOfferPreviewBannerText? {
        val currency = convertCurrencyCode(text.currencyIsoCode) ?: return null

        return GdOfferPreviewBannerText()
            .withDescriptionForDirect(text.descriptionForDirect)
            .withParamsForDirect(text.paramsForDirect)
            .withName(text.name)
            .withCurrencyIsoCode(currency)
    }

    private fun convertCurrencyCode(currencyCode: String): String? = when {
        "RUR" == currencyCode -> {
            logger.warn("Currency code \"RUR\" is not supported. Converted to \"RUB\"")
            "RUB"
        }
        "null" == currencyCode -> null
        !isRealCurrencyCode(currencyCode) -> {
            logger.warn("Currency code \"$currencyCode\" is not supported")
            null
        }
        else -> currencyCode
    }
}
