package ru.yandex.direct.grid.processing.service.crypta;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.crypta.service.CryptaSuggestService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaGoalsSuggest;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaGoalsSuggestContainer;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegment;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegmentFilter;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegmentsContainer;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegmentsContext;
import ru.yandex.direct.grid.processing.service.goal.CryptaGoalsConverter;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.web.core.entity.inventori.service.CryptaService;
import ru.yandex.direct.web.core.model.retargeting.CryptaGoalWeb;
import ru.yandex.direct.web.core.model.retargeting.CryptaInterestTypeWeb;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий справочник сегментов крипты
 **/
@GridGraphQLService
@ParametersAreNonnullByDefault
public class CryptaGoalGraphQlService {
    private final CryptaService cryptaService;
    private final GridShortenerService gridShortenerService;
    private final CryptaGoalsInputValidationService validationService;
    private final CryptaSuggestService cryptaSuggestService;

    @Autowired
    public CryptaGoalGraphQlService(CryptaService cryptaService,
                                    GridShortenerService gridShortenerService,
                                    CryptaGoalsInputValidationService validationService,
                                    CryptaSuggestService cryptaSuggestService) {
        this.cryptaService = cryptaService;
        this.gridShortenerService = gridShortenerService;
        this.validationService = validationService;
        this.cryptaSuggestService = cryptaSuggestService;
    }

    /**
     * GraphQL подзапрос. Получает справочник сегментов крипты
     *
     * @param context контекст
     * @param input   параметры фильтра и вывода
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "cryptaSegments")
    public GdCryptaSegmentsContext getGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCryptaSegmentsContainer input) {
        validationService.validateInput(input, context.getOperator());
        ClientId clientId = context.getSubjectUser().getClientId();

        if (input.getFilterKey() != null) {
            GdCryptaSegmentFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    clientId,
                    GdCryptaSegmentFilter.class,
                    () -> new GdCryptaSegmentFilter().withType(emptySet()));
            input.setFilter(savedFilter);
        }

        GdCryptaSegmentFilter filter = input.getFilter();
        List<CryptaGoalWeb> segments = cryptaService.getSegments(filter.getScope(), clientId);
        List<GdCryptaSegment> gdSegments = filterAndMapList(segments,
                s -> filter == null
                        || (filter.getType() == null || filter.getType().contains(s.getType()))
                        && (filter.getInterestType() == null
                        || filter.getInterestType() == CryptaInterestTypeWeb.all
                        || s.getInterestType() == CryptaInterestTypeWeb.all
                        || filter.getInterestType() == s.getInterestType()),
                s -> new GdCryptaSegment()
                        .withId(s.getId())
                        .withName(s.getName())
                        .withType(s.getType())
                        .withParentId(s.getParentId())
                        .withInterestType(s.getInterestType())
                        .withDescription(s.getDescription())
                        .withCanSelectAll(s.getCanSelectAll())
                        .withCanNotCheck(s.getCanNotCheck())
                        .withKeyword(s.getKeyword())
                        .withKeywordValue(s.getKeywordValue())
                        .withMutuallyExclusiveIds(s.getMutuallyExclusiveIds())
                        .withOrder(s.getOrder())
                        .withAvailableGroups(s.getAvailableGroups())
        );
        return new GdCryptaSegmentsContext()
                .withRowset(gdSegments)
                .withFilter(input.getFilter())
                .withTotalCount(gdSegments.size());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "cryptaGoalsSuggest")
    @Deprecated
    public GdCryptaGoalsSuggest getGoalsSuggestFromCrypta(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCryptaGoalsSuggestContainer input) {

        var suggestData = mapList(
                cryptaSuggestService.getRetargetingSuggests(input.getText()),
                CryptaGoalsConverter::toGdCryptaGoalsSuggestItem
        );

        return new GdCryptaGoalsSuggest().withRowset(suggestData);
    }
}
