package ru.yandex.direct.grid.processing.service.crypta;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalScope;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.validation.defects.RightsDefects;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegmentFilter;
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaSegmentsContainer;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.user.utils.UserUtil.isInternalAdRole;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuper;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isSuperReader;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

@Service
@ParametersAreNonnullByDefault
public class CryptaGoalsInputValidationService {
    private final GridValidationService gridValidationService;

    @Autowired
    public CryptaGoalsInputValidationService(GridValidationService gridValidationService) {
        this.gridValidationService = gridValidationService;
    }

    public void validateInput(GdCryptaSegmentsContainer input, User operator) {
        gridValidationService.applyValidator(i -> validateInputInternal(i, operator), input, false);
    }

    private ValidationResult<GdCryptaSegmentsContainer, Defect> validateInputInternal(GdCryptaSegmentsContainer input,
                                                                                      User operator) {
        ModelItemValidationBuilder<GdCryptaSegmentsContainer> vb = ModelItemValidationBuilder.of(input);

        vb.item(GdCryptaSegmentsContainer.FILTER)
                .check(notNull(), When.isTrue(input.getFilterKey() == null));
        vb.item(GdCryptaSegmentsContainer.FILTER_KEY)
                .check(notNull(), When.isTrue(input.getFilter() == null));

        vb.item(GdCryptaSegmentsContainer.FILTER)
                .checkBy(filter -> validateFilter(filter, operator), When.notNull());
        vb.item(GdCryptaSegmentsContainer.FILTER_KEY)
                .check(notBlank(), When.notNull());

        return vb.getResult();
    }

    private ValidationResult<GdCryptaSegmentFilter, Defect> validateFilter(GdCryptaSegmentFilter filter,
                                                                           User operator) {
        ItemValidationBuilder<GdCryptaSegmentFilter, Defect> vb = ItemValidationBuilder.of(filter);
        vb.item(filter.getScope(), "scope")
                .check(notNull())
                .check(Constraint.fromPredicate(scope -> allowedScope(scope, operator), RightsDefects.noRights()));
        return vb.getResult();
    }

    private boolean allowedScope(CryptaGoalScope scope, User operator) {
        // не хотим, чтобы обычные пользователи могли увидеть внутренние сегменты крипты
        return scope != CryptaGoalScope.INTERNAL_AD || (isInternalAdRole(operator) || isSuper(operator) || isSuperReader(operator));
    }

}
