package ru.yandex.direct.grid.processing.service.dataloader;

import java.util.List;

import org.dataloader.DataLoaderRegistry;
import org.springframework.aop.support.AopUtils;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

/**
 * Реестр {@link org.dataloader.DataLoader}'ов.
 * Используется в инструментировании GraphQL для вызова
 * batch'евой загрузки данных в нужный момент обработки запроса.
 * <p>
 * На каждый запрос используется свой registry. Поэтому {@code scope=REQUEST}.
 * Подробнее про схему с проксированием через proxyMode={@link org.springframework.context.annotation.ScopedProxyMode#TARGET_CLASS TARGET_CLASS}
 * <a href="https://docs.spring.io/spring/docs/4.3.14.RELEASE/spring-framework-reference/html/beans.html#beans-factory-scopes-other-injection-proxies">
 * в документации</a>
 *
 * @see DataLoaderProvider
 * @see GridBatchingDataLoader
 */
@Component
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
public class GridDataLoaderRegistry extends DataLoaderRegistry {

    public GridDataLoaderRegistry(List<DataLoaderProvider<?, ?>> dataLoaders) {
        dataLoaders.forEach(this::register);
    }

    @Override
    public void dispatchAll() {
        try (TraceProfile ignored = Trace.current().profile("gridDataLoader:dispatch")) {
            super.dispatchAll();
        }
    }

    private void register(DataLoaderProvider<?, ?> dataLoaderProxy) {
        register(AopUtils.getTargetClass(dataLoaderProxy).getSimpleName(), dataLoaderProxy.get());
    }
}
