package ru.yandex.direct.grid.processing.service.deal;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDeal;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDealFilter;
import ru.yandex.direct.grid.core.entity.deal.service.GridDealService;
import ru.yandex.direct.grid.processing.model.deal.GdDealOrderBy;
import ru.yandex.direct.utils.Counter;

import static ru.yandex.direct.grid.processing.service.deal.DealServiceHelpers.DEAL_FILTER_PROCESSOR;
import static ru.yandex.direct.grid.processing.service.deal.DealServiceHelpers.DEAL_ORDERING_PROCESSOR;

/**
 * Сервис, возвращающий данные о сделках клиента
 */
@Service
@ParametersAreNonnullByDefault
public class DealDataService {

    private final GridDealService gridDealService;

    @Autowired
    public DealDataService(GridDealService gridDealService) {
        this.gridDealService = gridDealService;
    }


    /**
     * Получить список всех сделок заданного клиента
     *
     * @param shard     шард
     * @param clientId  идентификатор клиента
     * @param needStats нужна ли статистика
     */
    public List<GdiDeal> getAllDeals(int shard, ClientId clientId, boolean needStats) {
        return gridDealService.getDeals(shard, clientId, needStats);
    }

    /**
     * Получить фильтрованный и сортированный список сделок клиента
     *
     * @param deals           список всех сделок клиента
     * @param selectedDealIds список идентификаторов сделок, выбранных в интерфейсе
     * @param filter          фильтр, по которому выбираем сделки
     * @param orderBy         список полей, по которым нужно производить сортировку получившегося списка сделок
     */
    public List<GdiDeal> getFilteredDeals(List<GdiDeal> deals,
                                          @Nullable Set<Long> selectedDealIds, @Nullable GdiDealFilter filter,
                                          @Nullable List<GdDealOrderBy> orderBy) {
        Counter counter = new Counter();
        return deals.stream()
                .filter(d -> DEAL_FILTER_PROCESSOR.test(filter, d))
                .filter(d -> selectedDealIds == null || selectedDealIds.contains(d.getId()))
                .sorted(DEAL_ORDERING_PROCESSOR.comparator(orderBy))
                .map(d -> d.withIndex(counter.next()))
                .collect(Collectors.toList());
    }
}
