package ru.yandex.direct.grid.processing.service.deal;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDeal;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDealFilter;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.deal.GdDealOrderBy;
import ru.yandex.direct.grid.processing.model.deal.GdDealsContext;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.deal.container.DealsCacheFilterData;
import ru.yandex.direct.grid.processing.service.deal.container.DealsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные о сделках клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class DealGraphQlService {

    private final GridCacheService gridCacheService;
    private final DealDataService dealDataService;
    private final GridValidationService gridValidationService;

    @Autowired
    public DealGraphQlService(GridCacheService gridCacheService, DealDataService dealDataService,
                              GridValidationService gridValidationService) {
        this.gridCacheService = gridCacheService;
        this.dealDataService = dealDataService;
        this.gridValidationService = gridValidationService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о сделках клиента, полученного из контекста выполнения запроса
     *
     * @param selectedDealIds выбранные в интерфейсе сделки, возвращать нужно только пересечение полученных
     *                        сделок с этим списком
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "deals")
    public GdDealsContext getDeals(
            @GraphQLContext GdClient client,
            @Nullable @GraphQLArgument(name = "filter") GdiDealFilter filter,
            @Nullable @GraphQLArgument(name = "orderBy") List<@GraphQLNonNull GdDealOrderBy> orderBy,
            @Nullable @GraphQLArgument(name = "selectedDealIds") Set<@GraphQLNonNull Long> selectedDealIds,
            @Nullable @GraphQLArgument(name = "limitOffset") GdLimitOffset limitOffset,
            @Nullable @GraphQLArgument(name = "cacheKey") String cacheKey,
            @Nullable @GraphQLArgument(name = "needStats") Boolean needStats) {
        gridValidationService.validateIdsCollection(selectedDealIds);
        gridValidationService.validateLimitOffset(limitOffset);
        gridValidationService.validateDealFilter(filter);

        LimitOffset range = normalizeLimitOffset(limitOffset);

        // пытаемся прочитать из кеша нужный диапазон строк
        DealsCacheRecordInfo recordInfo = new DealsCacheRecordInfo(client.getInfo().getId(), cacheKey,
                new DealsCacheFilterData()
                        .withFilter(filter)
                        .withSelectedDealIds(selectedDealIds)
                        .withOrderBy(orderBy)
                        .withNeedStats(needStats));
        Optional<GdDealsContext> res = gridCacheService.getFromCache(recordInfo, range);

        if (res.isPresent()) {
            return res.get();
        }

        // в кеше данные не нашлись, читаем из mysql/YT
        List<GdiDeal> deals = dealDataService
                .getAllDeals(client.getInfo().getShard(), ClientId.fromLong(client.getInfo().getId()),
                        nvl(needStats, false));

        List<GdiDeal> rowsetFull =
                dealDataService.getFilteredDeals(deals, selectedDealIds, filter, orderBy);

        GdDealsContext data = new GdDealsContext()
                .withTotalCount(rowsetFull.size())
                .withDealIds(mapList(rowsetFull, GdiDeal::getId));

        return gridCacheService.getResultAndSaveToCache(recordInfo, data, rowsetFull, range);
    }
}
