package ru.yandex.direct.grid.processing.service.deal;

import java.util.Comparator;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.grid.core.entity.deal.model.GdiDeal;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDealFilter;
import ru.yandex.direct.grid.core.entity.deal.model.GdiDealOrderByField;
import ru.yandex.direct.grid.core.util.filters.FilterProcessor;
import ru.yandex.direct.grid.core.util.ordering.OrderingProcessor;

import static ru.yandex.direct.grid.core.util.filters.FilterProvider.contains;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.containsMatching;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.greaterOrEqual;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.isSubStringCU;
import static ru.yandex.direct.grid.core.util.filters.FilterProvider.lessOrEqual;
import static ru.yandex.direct.validation.BiPredicates.not;

/**
 * Класс, содержащий обработчики, используемые для работы {@link DealDataService}
 * <p>
 * Обработчики вынесены отдельно, чтобы не засорять основной код
 */
@ParametersAreNonnullByDefault
public class DealServiceHelpers {

    private DealServiceHelpers() {
    }

    /**
     * Обработчик фильтров для сделок
     */
    static final FilterProcessor<GdiDealFilter, GdiDeal> DEAL_FILTER_PROCESSOR =
            new FilterProcessor.Builder<GdiDealFilter, GdiDeal>()
                    .withFilter(GdiDealFilter::getDealIdIncluded, contains(GdiDeal::getId))
                    .withFilter(GdiDealFilter::getDealIdExcluded, not(contains(GdiDeal::getId)))
                    .withFilter(GdiDealFilter::getTypeIn, contains(GdiDeal::getDealType))
                    .withFilter(GdiDealFilter::getNameContains, isSubStringCU(GdiDeal::getName))
                    .withFilter(GdiDealFilter::getNameNotContains, not(isSubStringCU(GdiDeal::getName)))
                    .withFilter(GdiDealFilter::getNameIn,
                            containsMatching(GdiDeal::getName, String::equalsIgnoreCase))
                    .withFilter(GdiDealFilter::getNameNotIn,
                            not(containsMatching(GdiDeal::getName, String::equalsIgnoreCase)))
                    .withFilter(GdiDealFilter::getStatusContains, contains(GdiDeal::getStatus))
                    .withFilter(GdiDealFilter::getCompleteReasonContains,
                            (cr, d) -> d.getCompleteReason() == null || cr.contains(d.getCompleteReason()))
                    .withFilter(GdiDealFilter::getMinShows, lessOrEqual(GdiDeal::getShows))
                    .withFilter(GdiDealFilter::getMaxShows, greaterOrEqual(GdiDeal::getShows))
                    .withFilter(GdiDealFilter::getMinClicks, lessOrEqual(GdiDeal::getClicks))
                    .withFilter(GdiDealFilter::getMaxClicks, greaterOrEqual(GdiDeal::getClicks))
                    .withFilter(GdiDealFilter::getMinSpent, lessOrEqual(GdiDeal::getSpent))
                    .withFilter(GdiDealFilter::getMaxSpent, greaterOrEqual(GdiDeal::getSpent))
                    .withFilter(GdiDealFilter::getMinMinPrice, lessOrEqual(GdiDeal::getMinPrice))
                    .withFilter(GdiDealFilter::getMaxMinPrice, greaterOrEqual(GdiDeal::getMinPrice))
                    .withFilter(GdiDealFilter::getMinCpm, lessOrEqual(GdiDeal::getCpm))
                    .withFilter(GdiDealFilter::getMaxCpm, greaterOrEqual(GdiDeal::getCpm))
                    .withFilter(GdiDealFilter::getMinCpc, lessOrEqual(GdiDeal::getCpc))
                    .withFilter(GdiDealFilter::getMaxCpc, greaterOrEqual(GdiDeal::getCpc))
                    .withFilter(GdiDealFilter::getMinCtr, lessOrEqual(GdiDeal::getCtr))
                    .withFilter(GdiDealFilter::getMaxCtr, greaterOrEqual(GdiDeal::getCtr))
                    .withFilter(GdiDealFilter::getMinNumberOfLinkedCampaigns,
                            lessOrEqual(GdiDeal::getNumberOfLinkedCampaigns))
                    .withFilter(GdiDealFilter::getMaxNumberOfLinkedCampaigns,
                            greaterOrEqual(GdiDeal::getNumberOfLinkedCampaigns))
                    .build();

    /**
     * Обработчик запросов на сортировку результата
     */
    static final OrderingProcessor<GdiDealOrderByField, GdiDeal> DEAL_ORDERING_PROCESSOR =
            OrderingProcessor.<GdiDealOrderByField, GdiDeal>builder(false)
                    .withFieldComparator(GdiDealOrderByField.ID, GdiDeal::getId)
                    .withComparator(GdiDealOrderByField.STATUS, statusComparator())
                    .withFieldComparator(GdiDealOrderByField.COMPLETE_REASON, GdiDeal::getCompleteReason)
                    .withFieldComparator(GdiDealOrderByField.DEAL_TYPE, GdiDeal::getDealType)
                    .withFieldComparator(GdiDealOrderByField.NAME, GdiDeal::getName)
                    .withFieldComparator(GdiDealOrderByField.SHOWS, GdiDeal::getShows)
                    .withFieldComparator(GdiDealOrderByField.CLICKS, GdiDeal::getClicks)
                    .withFieldComparator(GdiDealOrderByField.SPENT, GdiDeal::getSpent)
                    .withFieldComparator(GdiDealOrderByField.MIN_PRICE, GdiDeal::getMinPrice)
                    .withFieldComparator(GdiDealOrderByField.CPM, GdiDeal::getCpm)
                    .withFieldComparator(GdiDealOrderByField.CPC, GdiDeal::getCpc)
                    .withFieldComparator(GdiDealOrderByField.CTR, GdiDeal::getCtr)
                    .build();

    private static Comparator<GdiDeal> statusComparator() {
        Comparator<GdiDeal> comparator = defaultComparator(GdiDeal::getStatus);
        return comparator.thenComparing(defaultComparator(GdiDeal::getCompleteReason));
    }

    private static <GdiDeal, T extends Comparable<? super T>> Comparator<GdiDeal> defaultComparator(
            Function<GdiDeal, T> extractor) {
        Comparator<T> comparator = Comparator.nullsLast(Comparator.naturalOrder());
        return (a, b) -> comparator.compare(extractor.apply(a), extractor.apply(b));
    }
}
