package ru.yandex.direct.grid.processing.service.dialogs;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.campaign.model.Dialog;
import ru.yandex.direct.core.entity.dialogs.service.DialogsService;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.dialogs.GdAddDialogPayload;
import ru.yandex.direct.grid.processing.model.dialogs.GdAddDialogRequest;
import ru.yandex.direct.grid.processing.model.dialogs.GdDialog;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.grid.processing.service.dialogs.DialogConverter.toGdDialog;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class DialogGraphQLService {
    private final DialogsService dialogsService;
    private final GridValidationResultConversionService validationResultConverter;

    @Autowired
    public DialogGraphQLService(DialogsService dialogsService,
                                GridValidationResultConversionService validationResultConverter) {
        this.dialogsService = dialogsService;
        this.validationResultConverter = validationResultConverter;
    }

    /**
     * GraphQL ручка. Получает информацию о чатах клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "dialogs")
    public List<GdDialog> getDialogs(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        List<Dialog> existedDialogs = dialogsService.getDialogsByClientId(clientId);
        Long operatorUid = context.getOperator().getUid();
        List<Dialog> usersDialogs = dialogsService.getDialogsByUserId(clientId, operatorUid);
        return StreamEx.of(existedDialogs)
                .append(usersDialogs)
                .distinct(Dialog::getSkillId)
                .map(DialogConverter::toGdDialog)
                .sortedBy(GdDialog::getName)
                .toList();
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "addDialog")
    @PreAuthorizeWrite
    public GdAddDialogPayload addDialog(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddDialogRequest input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        String skillId = input.getSkillId();
        Map<String, Dialog> clientDialogs = listToMap(dialogsService.getDialogsByClientId(clientId),
                Dialog::getSkillId);
        if (clientDialogs.containsKey(skillId)) {
            return new GdAddDialogPayload()
                    .withDialog(toGdDialog(clientDialogs.get(skillId)));
        }
        Result<Dialog> result = dialogsService.addDialog(clientId, skillId);
        GdValidationResult validationResult =
                validationResultConverter.buildGridValidationResult(result.getValidationResult());
        return new GdAddDialogPayload()
                .withValidationResult(validationResult)
                .withDialog(toGdDialog(result.getResult()));
    }
}
