package ru.yandex.direct.grid.processing.service.dynamiccondition;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetsQueryFilter;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRuleKind;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRuleType;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.grid.core.entity.dynamiccondition.model.GdiDynamicCondition;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.dynamiccondition.DynamicAdTargetsCacheFilterData;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTarget;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetBaseStatus;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetFilter;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetPrimaryStatus;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetStatus;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetTab;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetsContainer;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicFeedAdTarget;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicFeedCondition;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicFeedConditionOperator;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageAdTarget;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageCondition;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageConditionOperand;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageConditionOperator;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.service.dynamiccondition.container.DynamicAdTargetsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter;
import ru.yandex.direct.utils.converter.Converters;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetMapping.isAllPageCondition;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.toGdAutobudgetPriority;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class DynamicAdTargetConverter {

    private static final ImmutableMap<WebpageRuleType, GdDynamicWebpageConditionOperand> CORE_TO_GD_WEBPAGE_OPERAND =
            ImmutableMap.<WebpageRuleType, GdDynamicWebpageConditionOperand>builder()
                    .put(WebpageRuleType.CONTENT, GdDynamicWebpageConditionOperand.PAGE_CONTENT)
                    .put(WebpageRuleType.DOMAIN, GdDynamicWebpageConditionOperand.DOMAIN)
                    .put(WebpageRuleType.TITLE, GdDynamicWebpageConditionOperand.PAGE_TITLE)
                    .put(WebpageRuleType.URL, GdDynamicWebpageConditionOperand.URL)
                    .put(WebpageRuleType.URL_PRODLIST, GdDynamicWebpageConditionOperand.OFFERS_LIST_URL)
                    .build();

    private static final ImmutableMap<WebpageRuleKind, GdDynamicWebpageConditionOperator> CORE_TO_GD_WEBPAGE_OPERATOR =
            ImmutableMap.<WebpageRuleKind, GdDynamicWebpageConditionOperator>builder()
                    .put(WebpageRuleKind.EQUALS, GdDynamicWebpageConditionOperator.EQUALS_ANY)
                    .put(WebpageRuleKind.NOT_EQUALS, GdDynamicWebpageConditionOperator.NOT_EQUALS_ALL)
                    .put(WebpageRuleKind.EXACT, GdDynamicWebpageConditionOperator.CONTAINS_ANY)
                    .put(WebpageRuleKind.NOT_EXACT, GdDynamicWebpageConditionOperator.NOT_CONTAINS_ALL)
                    .build();

    private static final ImmutableMap<Operator, GdDynamicFeedConditionOperator> CORE_TO_GD_FEED_OPERATOR =
            ImmutableMap.<Operator, GdDynamicFeedConditionOperator>builder()
                    .put(Operator.GREATER, GdDynamicFeedConditionOperator.GREATER_THAN)
                    .put(Operator.LESS, GdDynamicFeedConditionOperator.LESS_THAN)
                    .put(Operator.EQUALS, GdDynamicFeedConditionOperator.EQUALS_ANY)
                    .put(Operator.RANGE, GdDynamicFeedConditionOperator.RANGE)
                    .put(Operator.CONTAINS, GdDynamicFeedConditionOperator.CONTAINS_ANY)
                    .put(Operator.NOT_CONTAINS, GdDynamicFeedConditionOperator.NOT_CONTAINS_ALL)
                    .put(Operator.EXISTS, GdDynamicFeedConditionOperator.EXISTS)
                    .build();

    private DynamicAdTargetConverter() {
        // no instantiation
    }

    public static GdDynamicAdTarget toGdDynamicAdTarget(DynamicAdTarget coreDynamicAdTarget,
                                                        GdAdGroupTruncated adGroup,
                                                        PerformanceFilterStorage performanceFilterStorage) {
        GdDynamicAdTarget gdDynamicAdTarget;
        if (coreDynamicAdTarget instanceof DynamicTextAdTarget) {
            DynamicTextAdTarget dynamicTextAdTarget = (DynamicTextAdTarget) coreDynamicAdTarget;

            gdDynamicAdTarget = new GdDynamicWebpageAdTarget()
                    .withWebpageConditions(toGdWebpageConditions(dynamicTextAdTarget.getCondition()));

        } else if (coreDynamicAdTarget instanceof DynamicFeedAdTarget) {
            DynamicFeedAdTarget dynamicFeedAdTarget = (DynamicFeedAdTarget) coreDynamicAdTarget;

            gdDynamicAdTarget = new GdDynamicFeedAdTarget()
                    .withFeedConditions(toGdFeedConditions(dynamicFeedAdTarget, performanceFilterStorage));

        } else {
            throw new IllegalArgumentException("Unknown class: " + coreDynamicAdTarget.getClass().getName());
        }

        GdCampaignTruncated campaign = adGroup.getCampaign();
        GdDynamicAdTargetStatus status = calcStatus(coreDynamicAdTarget, campaign, adGroup);

        return gdDynamicAdTarget
                .withId(coreDynamicAdTarget.getId())
                .withDynamicConditionId(coreDynamicAdTarget.getDynamicConditionId())
                .withAdGroupId(coreDynamicAdTarget.getAdGroupId())
                .withCampaignId(coreDynamicAdTarget.getCampaignId())
                .withName(coreDynamicAdTarget.getConditionName())
                .withPrice(coreDynamicAdTarget.getPrice())
                .withPriceContext(coreDynamicAdTarget.getPriceContext())
                .withAutobudgetPriority(toGdAutobudgetPriority(coreDynamicAdTarget.getAutobudgetPriority()))
                .withIsSuspended(coreDynamicAdTarget.getIsSuspended())
                .withStatus(status)
                .withTab(GdDynamicAdTargetTab.fromSource(coreDynamicAdTarget.getTab()));
    }

    private static GdDynamicAdTargetStatus calcStatus(DynamicAdTarget dynamicAdTarget,
                                                      GdCampaignTruncated campaign,
                                                      GdAdGroupTruncated adGroup) {
        return new GdDynamicAdTargetStatus()
                .withReadOnly(campaign.getStatus().getReadOnly() || adGroup.getStatus().getArchived())
                .withSuspended(dynamicAdTarget.getIsSuspended())
                .withPrimaryStatus(calcPrimaryStatus(dynamicAdTarget));
    }

    private static GdDynamicAdTargetPrimaryStatus calcPrimaryStatus(DynamicAdTarget dynamicAdTarget) {
        if (dynamicAdTarget.getIsSuspended()) {
            return GdDynamicAdTargetPrimaryStatus.STOPPED;
        }
        return GdDynamicAdTargetPrimaryStatus.ACTIVE;
    }

    private static List<GdDynamicWebpageCondition> toGdWebpageConditions(List<WebpageRule> conditions) {
        if (isAllPageCondition(conditions)) {
            return emptyList();
        }
        return mapList(conditions, DynamicAdTargetConverter::toGdWebpageCondition);
    }

    private static GdDynamicWebpageCondition toGdWebpageCondition(WebpageRule rule) {
        return new GdDynamicWebpageCondition()
                .withOperand(toGdWebpageConditionOperand(rule.getType()))
                .withOperator(toGdWebpageConditionOperator(rule.getKind()))
                .withArguments(rule.getValue());
    }

    private static List<GdDynamicFeedCondition> toGdFeedConditions(DynamicFeedAdTarget dynamicFeedAdTarget,
                                                                   PerformanceFilterStorage performanceFilterStorage) {
        BusinessType businessType = dynamicFeedAdTarget.getBusinessType();
        FeedType feedType = dynamicFeedAdTarget.getFeedType();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType);

        return mapList(dynamicFeedAdTarget.getCondition(),
                rule -> toGdFeedCondition(rule, filterSchema));
    }

    private static GdDynamicFeedCondition toGdFeedCondition(DynamicFeedRule rule, FilterSchema filterSchema) {
        return new GdDynamicFeedCondition()
                .withField(filterSchema.translateFieldNameForGd(rule.getFieldName()))
                .withOperator(toGdFeedConditionOperator(rule.getOperator()))
                .withStringValue(rule.getStringValue());
    }

    public static GdDynamicWebpageConditionOperand toGdWebpageConditionOperand(WebpageRuleType type) {
        return Converters.mappingValueConverter(CORE_TO_GD_WEBPAGE_OPERAND).convert(type);
    }

    public static GdDynamicWebpageConditionOperator toGdWebpageConditionOperator(WebpageRuleKind kind) {
        return Converters.mappingValueConverter(CORE_TO_GD_WEBPAGE_OPERATOR).convert(kind);
    }

    public static GdDynamicFeedConditionOperator toGdFeedConditionOperator(Operator operator) {
        return Converters.mappingValueConverter(CORE_TO_GD_FEED_OPERATOR).convert(operator);
    }

    static GdiDynamicCondition toGdiDynamicCondition(GdDynamicAdTarget dynamicTarget) {
        return new GdiDynamicCondition()
                .withDynamicConditionId(dynamicTarget.getDynamicConditionId())
                .withAdGroupId(dynamicTarget.getAdGroupId())
                .withCampaignId(dynamicTarget.getCampaignId());
    }

    public static DynamicAdTargetsQueryFilter toDynamicAdTargetsQueryFilter(GdDynamicAdTargetFilter filter) {
        return new DynamicAdTargetsQueryFilter()
                .withCampaignIds(filter.getCampaignIdIn())
                .withAdGroupIds(filter.getAdGroupIdIn())
                .withIds(filter.getIdIn())
                .withIncludeDeleted(false)
                .withBaseStatuses(mapSet(filter.getStatusIn(), GdDynamicAdTargetBaseStatus::toSource))
                .withNameContains(filter.getNameContains())
                .withNameNotContains(filter.getNameNotContains())
                .withMinPrice(filter.getMinPrice())
                .withMaxPrice(filter.getMaxPrice())
                .withMinPriceContext(filter.getMinPriceContext())
                .withMaxPriceContext(filter.getMaxPriceContext())
                .withAutobudgetPriorities(mapSet(filter.getAutobudgetPriorityIn(),
                        SmartFilterConverter::fromGdAutobudgetPriority));
    }

    @NotNull
    public static DynamicAdTargetsCacheRecordInfo toDynamicAdTargetsCacheRecordInfo(GdDynamicAdTargetsContainer input,
                                                                                    GdClientInfo client) {
        return new DynamicAdTargetsCacheRecordInfo(
                client.getId(),
                input.getCacheKey(),
                new DynamicAdTargetsCacheFilterData()
                        .withFilter(input.getFilter())
                        .withOrderBy(input.getOrderBy())
                        .withStatRequirements(input.getStatRequirements()));
    }
}
