package ru.yandex.direct.grid.processing.service.dynamiccondition;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.util.comparator.Comparators;

import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTarget;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetOrderBy;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetOrderByField;

import static com.google.common.base.Preconditions.checkArgument;
import static java.math.BigDecimal.ZERO;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
class DynamicAdTargetUtils {

    private static final Map<GdDynamicAdTargetOrderByField, Comparator<GdDynamicAdTarget>> COMPARATOR_BY_FIELD
            = ImmutableMap.<GdDynamicAdTargetOrderByField, Comparator<GdDynamicAdTarget>>builder()
            .put(GdDynamicAdTargetOrderByField.ID, comparing(GdDynamicAdTarget::getId))
            .put(GdDynamicAdTargetOrderByField.GROUP_ID, comparing(GdDynamicAdTarget::getAdGroupId))
            .put(GdDynamicAdTargetOrderByField.CAMPAIGN_ID, comparing(GdDynamicAdTarget::getCampaignId))
            .put(GdDynamicAdTargetOrderByField.NAME, comparing(GdDynamicAdTarget::getName))
            .put(GdDynamicAdTargetOrderByField.PRICE, comparing(GdDynamicAdTarget::getPrice))
            .put(GdDynamicAdTargetOrderByField.PRICE_CONTEXT, comparing(GdDynamicAdTarget::getPriceContext))
            .put(GdDynamicAdTargetOrderByField.AUTOBUDGET_PRIORITY, comparing(GdDynamicAdTarget::getAutobudgetPriority))
            .put(GdDynamicAdTargetOrderByField.STAT_COST, comparingStat(GdEntityStats::getCost))
            .put(GdDynamicAdTargetOrderByField.STAT_COST_WITH_TAX, comparingStat(GdEntityStats::getCostWithTax))
            .put(GdDynamicAdTargetOrderByField.STAT_SHOWS, comparingStat(GdEntityStats::getShows))
            .put(GdDynamicAdTargetOrderByField.STAT_CLICKS, comparingStat(GdEntityStats::getClicks))
            .put(GdDynamicAdTargetOrderByField.STAT_CTR, comparingStat(GdEntityStats::getCtr))
            .put(GdDynamicAdTargetOrderByField.STAT_REVENUE, comparingStat(GdEntityStats::getRevenue))
            .put(GdDynamicAdTargetOrderByField.STAT_GOALS, comparingStat(GdEntityStats::getGoals))
            .put(GdDynamicAdTargetOrderByField.STAT_CPM_PRICE, comparingStat(GdEntityStats::getCpmPrice))
            .put(GdDynamicAdTargetOrderByField.STAT_PROFITABILITY, comparingStat(GdEntityStats::getProfitability))
            .put(GdDynamicAdTargetOrderByField.STAT_AVG_DEPTH, comparingStat(GdEntityStats::getAvgDepth))
            .put(GdDynamicAdTargetOrderByField.STAT_AVG_GOAL_COST, comparingStat(GdEntityStats::getAvgGoalCost))
            .put(GdDynamicAdTargetOrderByField.STAT_AVG_CLICK_COST, comparingStat(GdEntityStats::getAvgClickCost))
            .put(GdDynamicAdTargetOrderByField.STAT_AVG_SHOW_POSITION, comparingStat(GdEntityStats::getAvgShowPosition))
            .put(GdDynamicAdTargetOrderByField.STAT_AVG_CLICK_POSITION,
                    comparingStat(GdEntityStats::getAvgClickPosition))
            .put(GdDynamicAdTargetOrderByField.STAT_BOUNCE_RATE, comparingStat(GdEntityStats::getBounceRate))
            .put(GdDynamicAdTargetOrderByField.STAT_CONVERSION_RATE, comparingStat(GdEntityStats::getConversionRate))
            .build();

    private static final Map<GdDynamicAdTargetOrderByField, GoalComparatorFactory> GOAL_COMPARATOR_BY_FIELD
            = ImmutableMap.<GdDynamicAdTargetOrderByField, GoalComparatorFactory>builder()
            .put(GdDynamicAdTargetOrderByField.STAT_GOALS, comparingStatByGoalId(GdGoalStats::getGoals))
            .put(GdDynamicAdTargetOrderByField.STAT_CONVERSION_RATE,
                    comparingStatByGoalId(GdGoalStats::getConversionRate))
            .put(GdDynamicAdTargetOrderByField.STAT_COST_PER_ACTION,
                    comparingStatByGoalId(GdGoalStats::getCostPerAction))
            .build();

    private static final GdGoalStats EMPTY_GOAL_STAT =
            new GdGoalStats().withGoals(0L).withConversionRate(ZERO).withCostPerAction(ZERO);

    /**
     * @return компаратор, сравнивающий поля условий, полученные с помощью {@code extractor}
     */
    private static <T extends Comparable<T>> Comparator<GdDynamicAdTarget> comparing(Function<GdDynamicAdTarget, T> extractor) {
        return Comparator.comparing(extractor, Comparators.nullsHigh());
    }

    /**
     * @return компаратор, сравнивающий поля <b>статистики</b> условий, полученные с помощью {@code extractor}
     */
    private static <T extends Comparable<T>> Comparator<GdDynamicAdTarget> comparingStat(Function<GdEntityStats, T> extractor) {
        return comparing(gdDynamicAdTarget -> ifNotNull(gdDynamicAdTarget.getStats(), extractor));
    }

    /**
     * @param extractor поле статистики, которое будем сравнивать
     * @return фабрика компараторов, сравнивающих поля статистики по конкретной цели
     */
    private static <T extends Comparable<T>> GoalComparatorFactory comparingStatByGoalId(Function<GdGoalStats, T> extractor) {
        return targetGoalId -> comparing(gdDynamicAdTarget -> {
            // Выбираем статистику, связанную с целью targetGoalId
            for (GdGoalStats goalStats : gdDynamicAdTarget.getGoalStats()) {
                if (Objects.equals(goalStats.getGoalId(), targetGoalId)) {
                    return extractor.apply(goalStats);
                }
            }

            // Иначе считаем, что все значения статистики равны нулю
            return extractor.apply(EMPTY_GOAL_STAT);
        });
    }

    /**
     * Компаратор для динамических условий, построенный по полям {@link GdDynamicAdTargetOrderBy}
     * Аналог {@code SmartFilterUtils}
     */
    @Nonnull
    public static Comparator<GdDynamicAdTarget> getComparator(@Nullable List<GdDynamicAdTargetOrderBy> orderByItems) {
        // Компаратор, для которого любые два элемента равны
        Comparator<GdDynamicAdTarget> comparator = (a, b) -> 0;

        if (orderByItems == null) {
            return comparator;
        }

        for (GdDynamicAdTargetOrderBy item : orderByItems) {
            Comparator<GdDynamicAdTarget> current;
            if (item.getParams() == null) {
                current = COMPARATOR_BY_FIELD.get(item.getField());
                checkArgument(current != null, "Invalid order by field: " + item.getField());
            } else {
                var comparatorFactory = GOAL_COMPARATOR_BY_FIELD.get(item.getField());
                checkArgument(comparatorFactory != null, "Invalid order by field with goal id: " + item.getField());

                current = comparatorFactory.getComparatorByGoalId(item.getParams().getGoalId());
            }

            if (item.getOrder() == Order.DESC) {
                current = current.reversed();
            }

            comparator = comparator.thenComparing(current);
        }

        return comparator;
    }

    private interface GoalComparatorFactory {
        /**
         * @return компаратор, сравнивающий статистику по цели {@code goalId}
         */
        Comparator<GdDynamicAdTarget> getComparatorByGoalId(Long goalId);
    }
}
