package ru.yandex.direct.grid.processing.service.dynamiccondition;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetsQueryFilter;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.dynamiccondition.GridDynamicTargetYtRepository;
import ru.yandex.direct.grid.core.entity.dynamiccondition.model.GdiDynamicCondition;
import ru.yandex.direct.grid.core.entity.dynamiccondition.model.GdiDynamicTargetStats;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.DynamicTargetFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.model.GdEntityStats;
import ru.yandex.direct.grid.model.GdEntityStatsFilter;
import ru.yandex.direct.grid.model.GdGoalStats;
import ru.yandex.direct.grid.model.GdGoalStatsFilter;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTarget;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetFilter;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetsContainer;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetsContext;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicAdTargetsPayloadItem;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicFeedAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicWebpageAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdDeleteDynamicAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdDeleteDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicAdTargetsPayloadItem;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicFeedAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicWebpageAdTargets;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.grid.processing.service.dynamiccondition.container.DynamicAdTargetsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.group.GroupDataService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.grid.processing.util.GoalHelper;
import ru.yandex.direct.grid.processing.util.StatHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.BooleanUtils.isTrue;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicAdTargetConverter.toDynamicAdTargetsCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicAdTargetConverter.toDynamicAdTargetsQueryFilter;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicAdTargetConverter.toGdDynamicAdTarget;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicConditionsPathConverters.toGdAddFeedAdTargetsValidationResult;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicConditionsPathConverters.toGdAddWebpageAdTargetsValidationResult;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicConditionsPathConverters.toGdUpdateFeedAdTargetsValidationResult;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.DynamicConditionsPathConverters.toGdUpdateWebpageAdTargetsValidationResult;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.GdUpdateDynamicConditionsConverter.toDynamicTextAdTargetModelChangesList;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getSuccessfullyResults;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getSuccessfullyUpdatedIds;
import static ru.yandex.direct.grid.processing.util.StatHelper.applyEntityStatsFilter;
import static ru.yandex.direct.grid.processing.util.StatHelper.applyGoalsStatFilters;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalGoalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.calcTotalStats;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;
import static ru.yandex.direct.grid.processing.util.StatHelper.recalcTotalStatsForUnitedGrid;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
@ParametersAreNonnullByDefault
public class DynamicConditionsDataService {

    private final DynamicTextAdTargetService dynamicTextAdTargetService;
    private final GdAddDynamicConditionsConverter gdAddDynamicConditionsConverter;
    private final GdUpdateDynamicConditionsConverter gdUpdateDynamicConditionsConverter;
    private final GridValidationService gridValidationService;
    private final CampaignInfoService campaignInfoService;
    private final GroupDataService groupDataService;
    private final GridDynamicTargetYtRepository gridDynamicTargetYtRepository;
    private final PerformanceFilterStorage performanceFilterStorage;
    private final GridShortenerService gridShortenerService;
    private final GridCacheService gridCacheService;
    private final FeatureService featureService;
    private final MetrikaGoalsService metrikaGoalsService;

    @Autowired
    public DynamicConditionsDataService(
            DynamicTextAdTargetService dynamicTextAdTargetService,
            GdAddDynamicConditionsConverter gdAddDynamicConditionsConverter,
            GdUpdateDynamicConditionsConverter gdUpdateDynamicConditionsConverter,
            GridValidationService gridValidationService,
            CampaignInfoService campaignInfoService,
            GroupDataService groupDataService,
            GridDynamicTargetYtRepository gridDynamicTargetYtRepository,
            PerformanceFilterStorage performanceFilterStorage,
            FeatureService featureService,
            GridShortenerService gridShortenerService, GridCacheService gridCacheService,
            MetrikaGoalsService metrikaGoalsService) {
        this.dynamicTextAdTargetService = dynamicTextAdTargetService;
        this.gdAddDynamicConditionsConverter = gdAddDynamicConditionsConverter;
        this.gdUpdateDynamicConditionsConverter = gdUpdateDynamicConditionsConverter;
        this.gridValidationService = gridValidationService;
        this.campaignInfoService = campaignInfoService;
        this.groupDataService = groupDataService;
        this.gridDynamicTargetYtRepository = gridDynamicTargetYtRepository;
        this.performanceFilterStorage = performanceFilterStorage;
        this.gridShortenerService = gridShortenerService;
        this.featureService = featureService;
        this.gridCacheService = gridCacheService;
        this.metrikaGoalsService = metrikaGoalsService;
    }

    /**
     * Добавление динамических условий нацеливания по сайту
     */
    public GdAddDynamicAdTargetsPayload addDynamicWebpageAdTargets(ClientId clientId, Long operatorUid,
                                                                   GdAddDynamicWebpageAdTargets request) {
        List<DynamicTextAdTarget> dynamicTextAdTargets =
                mapList(request.getAddItems(), GdAddDynamicConditionsConverter::fromGdWebpageAdTarget);

        MassResult<Long> result = dynamicTextAdTargetService.addDynamicTextAdTargets(clientId, operatorUid,
                dynamicTextAdTargets);

        List<GdAddDynamicAdTargetsPayloadItem> addedItems = toGdAddedItems(clientId, result);
        GdValidationResult validationResult = toGdAddWebpageAdTargetsValidationResult(result.getValidationResult());

        return new GdAddDynamicAdTargetsPayload()
                .withAddedItems(addedItems)
                .withValidationResult(validationResult);
    }

    /**
     * Добавление динамических условий нацеливания по фиду
     */
    public GdAddDynamicAdTargetsPayload addDynamicFeedAdTargets(ClientId clientId, Long operatorUid,
                                                                GdAddDynamicFeedAdTargets request) {
        List<DynamicFeedAdTarget> dynamicFeedAdTargets =
                gdAddDynamicConditionsConverter.fromGdFeedAdTargets(request.getAddItems(), clientId);

        MassResult<Long> result = dynamicTextAdTargetService.addDynamicFeedAdTargets(clientId, operatorUid,
                dynamicFeedAdTargets);

        List<GdAddDynamicAdTargetsPayloadItem> addedItems = toGdAddedItems(clientId, result);
        GdValidationResult validationResult = toGdAddFeedAdTargetsValidationResult(result.getValidationResult());

        return new GdAddDynamicAdTargetsPayload()
                .withAddedItems(addedItems)
                .withValidationResult(validationResult);
    }

    private List<GdAddDynamicAdTargetsPayloadItem> toGdAddedItems(ClientId clientId, MassResult<Long> result) {

        List<Long> dynamicConditionIds = getSuccessfullyResults(result, identity());
        Map<Long, Long> idByDynamicConditionId =
                dynamicTextAdTargetService.getIdByDynamicConditionId(clientId, dynamicConditionIds);

        return getResults(result, dynamicConditionId -> new GdAddDynamicAdTargetsPayloadItem()
                .withDynamicConditionId(dynamicConditionId)
                .withId(idByDynamicConditionId.get(dynamicConditionId)));
    }

    /**
     * Обновление динамических условий нацеливания по сайту
     */
    public GdUpdateDynamicAdTargetsPayload updateDynamicWebpageAdTargets(ClientId clientId, Long operatorUid,
                                                                         GdUpdateDynamicWebpageAdTargets request) {
        List<ModelChanges<DynamicTextAdTarget>> modelChanges =
                toDynamicTextAdTargetModelChangesList(request.getUpdateItems());

        MassResult<Long> result = dynamicTextAdTargetService.updateDynamicTextAdTargets(clientId, operatorUid,
                modelChanges);

        List<GdUpdateDynamicAdTargetsPayloadItem> updatedItems = toGdUpdatedItems(clientId, result);
        GdValidationResult validationResult = toGdUpdateWebpageAdTargetsValidationResult(result.getValidationResult());

        return new GdUpdateDynamicAdTargetsPayload()
                .withUpdatedItems(updatedItems)
                .withValidationResult(validationResult);
    }

    /**
     * Обновление динамических условий нацеливания по фиду
     */
    public GdUpdateDynamicAdTargetsPayload updateDynamicFeedAdTargets(ClientId clientId, Long operatorUid,
                                                                      GdUpdateDynamicFeedAdTargets request) {
        List<ModelChanges<DynamicFeedAdTarget>> modelChanges = gdUpdateDynamicConditionsConverter
                .toDynamicFeedAdTargetModelChangesList(request.getUpdateItems(), clientId);

        MassResult<Long> result = dynamicTextAdTargetService.updateDynamicFeedAdTargets(clientId, operatorUid,
                modelChanges);

        List<GdUpdateDynamicAdTargetsPayloadItem> updatedItems = toGdUpdatedItems(clientId, result);
        GdValidationResult validationResult = toGdUpdateFeedAdTargetsValidationResult(result.getValidationResult());

        return new GdUpdateDynamicAdTargetsPayload()
                .withUpdatedItems(updatedItems)
                .withValidationResult(validationResult);
    }

    private List<GdUpdateDynamicAdTargetsPayloadItem> toGdUpdatedItems(ClientId clientId, MassResult<Long> result) {
        List<Long> ids = getSuccessfullyResults(result, identity());
        Map<Long, Long> dynamicConditionIdById = dynamicTextAdTargetService.getDynamicConditionIdById(clientId, ids);

        return getResults(result, id -> new GdUpdateDynamicAdTargetsPayloadItem()
                .withId(id)
                .withDynamicConditionId(dynamicConditionIdById.get(id)));
    }

    /**
     * Удаление динамических условий нацеливания
     */
    public GdDeleteDynamicAdTargetsPayload deleteDynamicAdTargets(
            ClientId clientId, Long operatorUid, GdDeleteDynamicAdTargets request) {
        List<Long> dynamicConditionIds = request.getDynamicConditionIds();

        MassResult<Long> result = dynamicTextAdTargetService.deleteDynamicAdTargets(operatorUid, clientId,
                dynamicConditionIds);

        GdValidationResult gdValidationResult = gridValidationService.toGdValidationResult(
                result.getValidationResult(), path(field(GdDeleteDynamicAdTargets.DYNAMIC_CONDITION_IDS)));

        return new GdDeleteDynamicAdTargetsPayload()
                .withDeletedDynamicConditionIds(getSuccessfullyUpdatedIds(result, identity()))
                .withValidationResult(gdValidationResult);
    }

    /**
     * Получение динамических условий нацеливания с кешированием
     */
    public GdDynamicAdTargetsContext getDynamicAdTargetsCaching(GdDynamicAdTargetsContainer input,
                                                                GridGraphQLContext context) {
        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());
        GdClientInfo client = context.getQueriedClient();

        if (input.getFilterKey() != null) {
            GdDynamicAdTargetFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(client.getId()),
                    GdDynamicAdTargetFilter.class,
                    () -> new GdDynamicAdTargetFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdStatRequirements statRequirements =
                normalizeStatRequirements(input.getStatRequirements(), context.getInstant());
        input.setStatRequirements(statRequirements);

        // Пытаемся достать из кеша
        DynamicAdTargetsCacheRecordInfo recordInfo = toDynamicAdTargetsCacheRecordInfo(input, client);
        Optional<GdDynamicAdTargetsContext> cached = gridCacheService.getFromCache(recordInfo, range);
        if (cached.isPresent()) {
            return cached.get();
        }

        boolean cpcAndCpmOnOneGridEnabled =
                featureService.isEnabledForClientId(ClientId.fromLong(client.getId()), CPC_AND_CPM_ON_ONE_GRID_ENABLED);

        // В кеше не нашлось, достаем из mysql/yt
        List<GdDynamicAdTarget> rowsetFull = getDynamicAdTargets(input, context, cpcAndCpmOnOneGridEnabled);
        var totalStats = calcTotalStats(mapList(rowsetFull, GdDynamicAdTarget::getStats));
        if (cpcAndCpmOnOneGridEnabled) {
            var stats = StreamEx.of(rowsetFull)
                    .map(GdDynamicAdTarget::getStats)
                    .toList();
            recalcTotalStatsForUnitedGrid(totalStats, Map.of(GdCampaignType.DYNAMIC, stats));
        }
        var totalGoalStats = calcTotalGoalStats(totalStats, mapList(rowsetFull, GdDynamicAdTarget::getGoalStats));

        GdDynamicAdTargetsContext adTargetsContext = new GdDynamicAdTargetsContext()
                .withTotalCount(rowsetFull.size())
                .withIds(listToSet(rowsetFull, GdDynamicAdTarget::getId))
                .withFilter(input.getFilter())
                .withTotalStats(totalStats)
                .withTotalGoalStats(totalGoalStats);

        // Сохраняем в кеш
        return gridCacheService.getResultAndSaveToCache(recordInfo, adTargetsContext, rowsetFull, range);
    }

    private List<GdDynamicAdTarget> getDynamicAdTargets(GdDynamicAdTargetsContainer input,
                                                        GridGraphQLContext context,
                                                        boolean cpcAndCpmOnOneGridEnabled) {
        GdClientInfo client = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(client.getId());
        User operator = context.getOperator();

        GdDynamicAdTargetFilter filter = input.getFilter();
        DynamicAdTargetsQueryFilter queryFilter = toDynamicAdTargetsQueryFilter(filter);

        List<DynamicAdTarget> dynamicAdTargets = dynamicTextAdTargetService.getDynamicAdTargets(clientId, queryFilter);
        List<GdDynamicAdTarget> gdDynamicAdTargets = convertToGdDynamicAdTargets(client, operator, dynamicAdTargets);

        DynamicTargetFetchedFieldsResolver resolver = context.getFetchedFieldsReslover().getDynamicTarget();

        //получение статистики из YT, при необходимости
        if (isTrue(resolver.getStats()) || filter.getStats() != null
                || isTrue(resolver.getGoalStats()) || filter.getGoalStats() != null) {
            gdDynamicAdTargets = enrichAndFilterWithStat(operator.getUid(), clientId,
                    input.getStatRequirements(), filter, gdDynamicAdTargets, cpcAndCpmOnOneGridEnabled);
        }

        return gdDynamicAdTargets.stream()
                .sorted(DynamicAdTargetUtils.getComparator(input.getOrderBy()))
                .collect(toList());
    }

    @NotNull
    private List<GdDynamicAdTarget> enrichAndFilterWithStat(Long operatorUid,
                                                            ClientId clientId,
                                                            GdStatRequirements statRequirements,
                                                            GdDynamicAdTargetFilter filter,
                                                            List<GdDynamicAdTarget> gdDynamicAdTargets,
                                                            boolean cpcAndCpmOnOneGridEnabled) {
        Set<Long> goalIds = GoalHelper.combineGoalIds(statRequirements.getGoalIds(), filter.getGoalStats());
        boolean getRevenueOnlyByAvailableGoals = featureService
                .isEnabledForClientId(clientId, FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        Set<Long> availableGoalIds = null;
        if (CollectionUtils.isNotEmpty(goalIds) && getRevenueOnlyByAvailableGoals) {
            availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                    operatorUid, clientId);
        }
        List<GdiDynamicCondition> gdiDynamicConditions =
                mapList(gdDynamicAdTargets, DynamicAdTargetConverter::toGdiDynamicCondition);
        Map<Long, GdiDynamicTargetStats> statsByConditionId = gridDynamicTargetYtRepository.getStatistic(
                gdiDynamicConditions, statRequirements.getFrom(), statRequirements.getTo(), goalIds, availableGoalIds);

        GdEntityStatsFilter entityStatsFilter = filter.getStats();
        List<@GraphQLNonNull GdGoalStatsFilter> goalStatsFilters = filter.getGoalStats();
        List<GdDynamicAdTarget> resultDynamicAdTargets = new ArrayList<>();
        for (GdDynamicAdTarget dt : gdDynamicAdTargets) {
            Long conditionId = dt.getDynamicConditionId();
            GdiDynamicTargetStats gdiStats = statsByConditionId.get(conditionId);

            Optional<GdEntityStats> gdEntityStats = Optional.ofNullable(gdiStats)
                    .map(GdiDynamicTargetStats::getStat)
                    .or(() -> Optional.of(GridStatNew.addZeros(new GdiEntityStats())))
                    .map(es -> StatHelper.internalStatsToOuter(es, GdCampaignType.DYNAMIC, cpcAndCpmOnOneGridEnabled))
                    .filter(es -> applyEntityStatsFilter(entityStatsFilter, es));

            Optional<List<GdGoalStats>> gdGoalStats = Optional.ofNullable(gdiStats)
                    .map(GdiDynamicTargetStats::getGoalStats)
                    .or(() -> Optional.of(emptyList()))
                    .map(gs -> mapList(gs, StatHelper::internalGoalStatToOuter))
                    .filter(gs -> applyGoalsStatFilters(goalStatsFilters, gs));

            if (gdEntityStats.isPresent() && gdGoalStats.isPresent()) {
                dt.setStats(gdEntityStats.get());
                dt.setGoalStats(gdGoalStats.get());
                resultDynamicAdTargets.add(dt);
            }
        }
        return resultDynamicAdTargets;
    }

    private List<GdDynamicAdTarget> convertToGdDynamicAdTargets(GdClientInfo client, User operator,
                                                                List<DynamicAdTarget> dynamicAdTargets) {
        ClientId clientId = ClientId.fromLong(client.getId());
        int shard = client.getShard();

        Set<Long> adGroupIds = listToSet(dynamicAdTargets, DynamicAdTarget::getAdGroupId);
        Set<Long> campaignIds = listToSet(dynamicAdTargets, DynamicAdTarget::getCampaignId);
        Map<Long, GdCampaignTruncated> campaignsById = campaignInfoService.getTruncatedCampaigns(clientId, campaignIds);

        List<GdAdGroupTruncated> adGroups = groupDataService.getTruncatedAdGroups(
                shard, client.getCountryRegionId(), clientId, operator, adGroupIds, campaignsById);
        Map<Long, GdAdGroupTruncated> adGroupsById = listToMap(adGroups, GdAdGroupTruncated::getId);

        return StreamEx.of(dynamicAdTargets)
                .filter(d -> adGroupsById.containsKey(d.getAdGroupId()))
                .map(d -> toGdDynamicAdTarget(d, adGroupsById.get(d.getAdGroupId()), performanceFilterStorage))
                .toList();
    }
}
