package ru.yandex.direct.grid.processing.service.dynamiccondition;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetsContainer;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetsContext;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicFeedAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicWebpageAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdDeleteDynamicAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdDeleteDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicAdTargetsPayload;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicFeedAdTargets;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicWebpageAdTargets;

import static com.google.common.base.Preconditions.checkNotNull;

@GridGraphQLService
@ParametersAreNonnullByDefault
public class DynamicConditionsGraphQlService {

    public static final String DYNAMIC_AD_TARGETS_RESOLVER_NAME = "dynamicAdTargets";
    private final DynamicConditionsDataService dynamicConditionsDataService;

    @Autowired
    public DynamicConditionsGraphQlService(
            DynamicConditionsDataService dynamicConditionsDataService) {
        this.dynamicConditionsDataService = dynamicConditionsDataService;
    }

    /**
     * Мутация для массового добавления динамических условий нацеливания по сайту
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addDynamicWebpageAdTargets")
    public GdAddDynamicAdTargetsPayload addDynamicWebpageAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddDynamicWebpageAdTargets input) {
        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return dynamicConditionsDataService.addDynamicWebpageAdTargets(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового добавления динамических условий нацеливания по фиду
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addDynamicFeedAdTargets")
    public GdAddDynamicAdTargetsPayload addDynamicFeedAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddDynamicFeedAdTargets input) {
        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return dynamicConditionsDataService.addDynamicFeedAdTargets(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового обновления динамических условий нацеливания по сайту
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateDynamicWebpageAdTargets")
    public GdUpdateDynamicAdTargetsPayload updateDynamicWebpageAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateDynamicWebpageAdTargets input) {
        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return dynamicConditionsDataService.updateDynamicWebpageAdTargets(clientId,
                context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового обновления динамических условий нацеливания по фиду
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateDynamicFeedAdTargets")
    public GdUpdateDynamicAdTargetsPayload updateDynamicFeedAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateDynamicFeedAdTargets input) {
        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return dynamicConditionsDataService.updateDynamicFeedAdTargets(clientId, context.getOperator().getUid(), input);
    }

    /**
     * Мутация для массового удаления динамических условий нацеливания
     */
    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "deleteDynamicAdTargets")
    public GdDeleteDynamicAdTargetsPayload deleteDynamicAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDeleteDynamicAdTargets input) {
        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        return dynamicConditionsDataService.deleteDynamicAdTargets(clientId, context.getOperator().getUid(), input);
    }

    /**
     * GraphQL подзапрос. Получает информацию о динамических условиях нацеливания клиента,
     * полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = DYNAMIC_AD_TARGETS_RESOLVER_NAME)
    public GdDynamicAdTargetsContext getDynamicAdTargets(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdDynamicAdTargetsContainer input) {
        // TODO: validateContainer(input)
        return dynamicConditionsDataService.getDynamicAdTargetsCaching(input, context);
    }
}
