package ru.yandex.direct.grid.processing.service.dynamiccondition;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRuleKind;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRuleType;
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetTab;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicFeedConditionOperator;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageConditionOperand;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicWebpageConditionOperator;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicFeedAdTargetsItem;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicFeedCondition;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicWebpageAdTargetsItem;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicWebpageCondition;
import ru.yandex.direct.utils.converter.Converters;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdAutobudgetPriority;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class GdAddDynamicConditionsConverter {

    private static final ImmutableMap<GdDynamicWebpageConditionOperand, WebpageRuleType> GD_TO_CORE_WEBPAGE_OPERAND =
            ImmutableMap.<GdDynamicWebpageConditionOperand, WebpageRuleType>builder()
                    .put(GdDynamicWebpageConditionOperand.PAGE_CONTENT, WebpageRuleType.CONTENT)
                    .put(GdDynamicWebpageConditionOperand.DOMAIN, WebpageRuleType.DOMAIN)
                    .put(GdDynamicWebpageConditionOperand.PAGE_TITLE, WebpageRuleType.TITLE)
                    .put(GdDynamicWebpageConditionOperand.URL, WebpageRuleType.URL)
                    .put(GdDynamicWebpageConditionOperand.OFFERS_LIST_URL, WebpageRuleType.URL_PRODLIST)
                    .build();

    private static final ImmutableMap<GdDynamicWebpageConditionOperator, WebpageRuleKind> GD_TO_CORE_WEBPAGE_OPERATOR =
            ImmutableMap.<GdDynamicWebpageConditionOperator, WebpageRuleKind>builder()
                    .put(GdDynamicWebpageConditionOperator.EQUALS_ANY, WebpageRuleKind.EQUALS)
                    .put(GdDynamicWebpageConditionOperator.NOT_EQUALS_ALL, WebpageRuleKind.NOT_EQUALS)
                    .put(GdDynamicWebpageConditionOperator.CONTAINS_ANY, WebpageRuleKind.EXACT)
                    .put(GdDynamicWebpageConditionOperator.NOT_CONTAINS_ALL, WebpageRuleKind.NOT_EXACT)
                    .build();

    private static final ImmutableMap<GdDynamicFeedConditionOperator, Operator> GD_TO_CORE_FEED_OPERATOR =
            ImmutableMap.<GdDynamicFeedConditionOperator, Operator>builder()
                    .put(GdDynamicFeedConditionOperator.GREATER_THAN, Operator.GREATER)
                    .put(GdDynamicFeedConditionOperator.LESS_THAN, Operator.LESS)
                    .put(GdDynamicFeedConditionOperator.EQUALS_ANY, Operator.EQUALS)
                    .put(GdDynamicFeedConditionOperator.RANGE, Operator.RANGE)
                    .put(GdDynamicFeedConditionOperator.CONTAINS_ANY, Operator.CONTAINS)
                    .put(GdDynamicFeedConditionOperator.NOT_CONTAINS_ALL, Operator.NOT_CONTAINS)
                    .put(GdDynamicFeedConditionOperator.EXISTS, Operator.EXISTS)
                    .build();

    private final FeedService feedService;
    private final PerformanceFilterStorage performanceFilterStorage;

    @Autowired
    public GdAddDynamicConditionsConverter(
            FeedService feedService,
            PerformanceFilterStorage performanceFilterStorage) {
        this.feedService = feedService;
        this.performanceFilterStorage = performanceFilterStorage;
    }

    /**
     * вместе с конвертацией для консистентости заполняем hash'и
     */
    public static DynamicTextAdTarget fromGdWebpageAdTarget(GdAddDynamicWebpageAdTargetsItem gdAddItem) {
        List<WebpageRule> conditions = mapList(gdAddItem.getConditions(),
                GdAddDynamicConditionsConverter::fromGdWebpageCondition);

        DynamicTextAdTarget dynamicTextAdTarget = new DynamicTextAdTarget()
                .withAdGroupId(gdAddItem.getAdGroupId())
                .withConditionName(gdAddItem.getName())
                .withCondition(conditions)
                .withPrice(gdAddItem.getPrice())
                .withPriceContext(gdAddItem.getPriceContext())
                .withAutobudgetPriority(fromGdAutobudgetPriority(gdAddItem.getAutobudgetPriority()))
                .withIsSuspended(false);

        dynamicTextAdTarget
                .withConditionHash(DynamicTextAdTargetHashUtils.getHash(dynamicTextAdTarget.getCondition()))
                .withConditionUniqHash(DynamicTextAdTargetHashUtils.getUniqHash(dynamicTextAdTarget.getCondition()));

        return dynamicTextAdTarget;
    }

    static WebpageRule fromGdWebpageCondition(GdAddDynamicWebpageCondition condition) {
        return new WebpageRule()
                .withType(fromGdWebpageConditionOperand(condition.getOperand()))
                .withKind(fromGdWebpageConditionOperator(condition.getOperator()))
                .withValue(condition.getArguments());
    }

    public List<DynamicFeedAdTarget> fromGdFeedAdTargets(List<GdAddDynamicFeedAdTargetsItem> addItems,
                                                         ClientId clientId) {
        Set<Long> adGroupIds = listToSet(addItems, GdAddDynamicFeedAdTargetsItem::getAdGroupId);
        Map<Long, Feed> feedByAdGroupId = feedService.getFeedByDynamicAdGroupId(clientId, adGroupIds);

        return mapList(addItems, gdAddItem -> fromGdFeedAdTarget(gdAddItem, feedByAdGroupId));
    }

    private DynamicFeedAdTarget fromGdFeedAdTarget(GdAddDynamicFeedAdTargetsItem gdAddItem,
                                                   Map<Long, Feed> feedByAdGroupId) {
        Feed feed = feedByAdGroupId.get(gdAddItem.getAdGroupId());
        checkNotNull(feed, "feed is required");

        BusinessType businessType = feed.getBusinessType();
        FeedType feedType = feed.getFeedType();
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType);
        List<DynamicFeedRule> conditions = fromGdFeedConditions(gdAddItem.getConditions(), filterSchema);

        return new DynamicFeedAdTarget()
                .withAdGroupId(gdAddItem.getAdGroupId())
                .withConditionName(gdAddItem.getName())
                .withPrice(gdAddItem.getPrice())
                .withPriceContext(gdAddItem.getPriceContext())
                .withAutobudgetPriority(fromGdAutobudgetPriority(gdAddItem.getAutobudgetPriority()))
                .withIsSuspended(false)
                .withBusinessType(businessType)
                .withFeedType(feedType)
                .withCondition(conditions)
                .withConditionHash(getHashForDynamicFeedRules(conditions))
                .withTab(GdDynamicAdTargetTab.toSource(gdAddItem.getTab()));
    }

    static List<DynamicFeedRule> fromGdFeedConditions(List<GdAddDynamicFeedCondition> gdConditions,
                                                      FilterSchema filterSchema) {
        List<DynamicFeedRule> conditions = mapList(gdConditions,
                gdCondition -> new DynamicFeedRule<>(
                        filterSchema.translateFieldNameFromGd(gdCondition.getField()),
                        fromGdFeedConditionOperator(gdCondition.getOperator()),
                        gdCondition.getStringValue())
        );
        PerformanceFilterConditionDBFormatParser.setParsedValue(filterSchema, conditions);

        return conditions;
    }

    private static WebpageRuleType fromGdWebpageConditionOperand(GdDynamicWebpageConditionOperand type) {
        return Converters.mappingValueConverter(GD_TO_CORE_WEBPAGE_OPERAND).convert(type);
    }

    private static WebpageRuleKind fromGdWebpageConditionOperator(GdDynamicWebpageConditionOperator kind) {
        return Converters.mappingValueConverter(GD_TO_CORE_WEBPAGE_OPERATOR).convert(kind);
    }

    private static Operator fromGdFeedConditionOperator(GdDynamicFeedConditionOperator operator) {
        return Converters.mappingValueConverter(GD_TO_CORE_FEED_OPERATOR).convert(operator);
    }
}
