package ru.yandex.direct.grid.processing.service.dynamiccondition;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicTextAdTargetService;
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.dynamiccondition.GdDynamicAdTargetTab;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdAddDynamicWebpageCondition;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicFeedAdTargetsItem;
import ru.yandex.direct.grid.processing.model.dynamiccondition.mutation.GdUpdateDynamicWebpageAdTargetsItem;
import ru.yandex.direct.model.ModelChanges;

import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.ALL_PAGE_CONDITION;
import static ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils.getHashForDynamicFeedRules;
import static ru.yandex.direct.grid.processing.service.dynamiccondition.GdAddDynamicConditionsConverter.fromGdFeedConditions;
import static ru.yandex.direct.grid.processing.service.smartfilter.SmartFilterConverter.fromGdAutobudgetPriority;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class GdUpdateDynamicConditionsConverter {

    private final PerformanceFilterStorage performanceFilterStorage;
    private final DynamicTextAdTargetService dynamicTextAdTargetService;

    @Autowired
    public GdUpdateDynamicConditionsConverter(
            PerformanceFilterStorage performanceFilterStorage,
            DynamicTextAdTargetService dynamicTextAdTargetService) {
        this.performanceFilterStorage = performanceFilterStorage;
        this.dynamicTextAdTargetService = dynamicTextAdTargetService;
    }

    public static List<ModelChanges<DynamicTextAdTarget>> toDynamicTextAdTargetModelChangesList(
            List<GdUpdateDynamicWebpageAdTargetsItem> gdUpdateItems) {
        List<DynamicTextAdTarget> dynamicTextAdTargets =
                mapList(gdUpdateItems, GdUpdateDynamicConditionsConverter::fromGdWebpageAdTarget);
        return mapList(dynamicTextAdTargets, GdUpdateDynamicConditionsConverter::toDynamicTextAdTargetModelChanges);
    }

    private static DynamicTextAdTarget fromGdWebpageAdTarget(GdUpdateDynamicWebpageAdTargetsItem gdUpdateItem) {

        DynamicTextAdTarget dynamicTextAdTarget = new DynamicTextAdTarget()
                .withId(gdUpdateItem.getId())
                .withConditionName(gdUpdateItem.getName())
                .withPrice(gdUpdateItem.getPrice())
                .withPriceContext(gdUpdateItem.getPriceContext())
                .withAutobudgetPriority(fromGdAutobudgetPriority(gdUpdateItem.getAutobudgetPriority()))
                .withIsSuspended(gdUpdateItem.getIsSuspended());

        if (gdUpdateItem.getConditions() != null) {
            List<WebpageRule> conditions = fromGdWebpageConditions(gdUpdateItem.getConditions());
            dynamicTextAdTarget
                    .withCondition(conditions)
                    .withConditionHash(DynamicTextAdTargetHashUtils.getHash(conditions))
                    .withConditionUniqHash(DynamicTextAdTargetHashUtils.getUniqHash(conditions));
        }
        return dynamicTextAdTarget;
    }

    private static List<WebpageRule> fromGdWebpageConditions(List<GdAddDynamicWebpageCondition> gdWebpageConditions) {
        if (gdWebpageConditions.isEmpty()) {
            return ALL_PAGE_CONDITION;
        }
        return mapList(gdWebpageConditions, GdAddDynamicConditionsConverter::fromGdWebpageCondition);
    }

    public List<ModelChanges<DynamicFeedAdTarget>> toDynamicFeedAdTargetModelChangesList(
            List<GdUpdateDynamicFeedAdTargetsItem> gdUpdateItems, ClientId clientId) {
        Set<Long> ids = listToSet(gdUpdateItems, GdUpdateDynamicFeedAdTargetsItem::getId);

        List<DynamicFeedAdTarget> oldDynamicAdTargets =
                dynamicTextAdTargetService.getDynamicFeedAdTargetsByIds(clientId, ids);
        Map<Long, DynamicFeedAdTarget> oldDynamicAdTargetById = listToMap(oldDynamicAdTargets, DynamicAdTarget::getId);

        List<DynamicFeedAdTarget> newDynamicAdTargets = mapList(gdUpdateItems,
                gdUpdateItem -> fromGdFeedAdTarget(gdUpdateItem, oldDynamicAdTargetById));

        return mapList(newDynamicAdTargets, GdUpdateDynamicConditionsConverter::toDynamicFeedAdTargetModelChanges);
    }

    private DynamicFeedAdTarget fromGdFeedAdTarget(GdUpdateDynamicFeedAdTargetsItem gdUpdateItem,
                                                   Map<Long, DynamicFeedAdTarget> oldDynamicAdTargetById) {
        DynamicFeedAdTarget newDynamicAdTarget = new DynamicFeedAdTarget()
                .withId(gdUpdateItem.getId())
                .withConditionName(gdUpdateItem.getName())
                .withPrice(gdUpdateItem.getPrice())
                .withPriceContext(gdUpdateItem.getPriceContext())
                .withAutobudgetPriority(fromGdAutobudgetPriority(gdUpdateItem.getAutobudgetPriority()))
                .withIsSuspended(gdUpdateItem.getIsSuspended())
                .withTab(GdDynamicAdTargetTab.toSource(gdUpdateItem.getTab()));

        DynamicFeedAdTarget oldDynamicAdTarget = oldDynamicAdTargetById.get(gdUpdateItem.getId());

        if (gdUpdateItem.getConditions() != null && oldDynamicAdTarget != null) {
            BusinessType businessType = oldDynamicAdTarget.getBusinessType();
            FeedType feedType = oldDynamicAdTarget.getFeedType();
            FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(businessType, feedType);
            List<DynamicFeedRule> conditions = fromGdFeedConditions(gdUpdateItem.getConditions(), filterSchema);

            newDynamicAdTarget
                    .withCondition(conditions)
                    .withConditionHash(getHashForDynamicFeedRules(conditions));
        }
        return newDynamicAdTarget;
    }

    private static ModelChanges<DynamicTextAdTarget> toDynamicTextAdTargetModelChanges(
            DynamicTextAdTarget dynamicAdTarget) {
        ModelChanges<DynamicTextAdTarget> modelChanges =
                new ModelChanges<>(dynamicAdTarget.getId(), DynamicTextAdTarget.class);

        modelChanges.processNotNull(dynamicAdTarget.getCondition(), DynamicTextAdTarget.CONDITION);
        modelChanges.processNotNull(dynamicAdTarget.getConditionHash(), DynamicAdTarget.CONDITION_HASH);
        modelChanges.processNotNull(dynamicAdTarget.getConditionUniqHash(), DynamicTextAdTarget.CONDITION_UNIQ_HASH);

        modelChanges.processNotNull(dynamicAdTarget.getConditionName(), DynamicAdTarget.CONDITION_NAME);
        modelChanges.processNotNull(dynamicAdTarget.getPrice(), DynamicAdTarget.PRICE);
        modelChanges.processNotNull(dynamicAdTarget.getPriceContext(), DynamicAdTarget.PRICE_CONTEXT);
        modelChanges.processNotNull(dynamicAdTarget.getAutobudgetPriority(), DynamicAdTarget.AUTOBUDGET_PRIORITY);
        modelChanges.processNotNull(dynamicAdTarget.getIsSuspended(), DynamicAdTarget.IS_SUSPENDED);

        return modelChanges;
    }

    private static ModelChanges<DynamicFeedAdTarget> toDynamicFeedAdTargetModelChanges(
            DynamicFeedAdTarget dynamicAdTarget) {
        ModelChanges<DynamicFeedAdTarget> modelChanges =
                new ModelChanges<>(dynamicAdTarget.getId(), DynamicFeedAdTarget.class);

        modelChanges.processNotNull(dynamicAdTarget.getCondition(), DynamicFeedAdTarget.CONDITION);
        modelChanges.processNotNull(dynamicAdTarget.getConditionHash(), DynamicAdTarget.CONDITION_HASH);
        modelChanges.processNotNull(dynamicAdTarget.getTab(), DynamicAdTarget.TAB);

        modelChanges.processNotNull(dynamicAdTarget.getConditionName(), DynamicAdTarget.CONDITION_NAME);
        modelChanges.processNotNull(dynamicAdTarget.getPrice(), DynamicAdTarget.PRICE);
        modelChanges.processNotNull(dynamicAdTarget.getPriceContext(), DynamicAdTarget.PRICE_CONTEXT);
        modelChanges.processNotNull(dynamicAdTarget.getAutobudgetPriority(), DynamicAdTarget.AUTOBUDGET_PRIORITY);
        modelChanges.processNotNull(dynamicAdTarget.getIsSuspended(), DynamicAdTarget.IS_SUSPENDED);

        return modelChanges;
    }
}
