package ru.yandex.direct.grid.processing.service.feed;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.SortOrder;
import ru.yandex.direct.core.entity.feed.FeedUtilsKt;
import ru.yandex.direct.core.entity.feed.container.FeedsOrderBy;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedCategory;
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItem;
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResults;
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResultsDefect;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.processing.FeedOrderByField;
import ru.yandex.direct.core.entity.uac.model.ShopInShopBusinessInfo;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.feed.GdBusinessType;
import ru.yandex.direct.grid.model.feed.GdFeed;
import ru.yandex.direct.grid.model.feed.GdFeedAccess;
import ru.yandex.direct.grid.model.feed.GdFeedCategory;
import ru.yandex.direct.grid.model.feed.GdFeedDefect;
import ru.yandex.direct.grid.model.feed.GdFeedType;
import ru.yandex.direct.grid.model.feed.GdSource;
import ru.yandex.direct.grid.model.feed.GdUpdateStatus;
import ru.yandex.direct.grid.processing.model.feed.GdFeedOrderByField;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsOrderBy;

import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.core.entity.feed.service.FeedServiceKt.REFRESH_HOURS_PERIOD;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class FeedConverter {
    public static final Set<UpdateStatus> STATUSES_ALLOWED_TO_DELETE = Set.of(UpdateStatus.DONE, UpdateStatus.ERROR);

    private static final String FEED_NAME_BY_SHOP_IN_SHOP_PATTERN = "Фид по бизнесу #%d из маркетплейса %s";

    static UpdateStatus convertUpdateStatusFromGd(GdUpdateStatus gdUpdateStatus) {
        return gdUpdateStatus != null ? UpdateStatus.valueOf(gdUpdateStatus.name()) : null;
    }

    static Source convertSourceFromGd(GdSource gdSource) {
        return gdSource != null ? Source.valueOf(gdSource.name()) : null;
    }

    static BusinessType convertBusinessTypeFromGd(GdBusinessType gdBusinessType) {
        return gdBusinessType != null ? BusinessType.valueOf(gdBusinessType.name()) : null;
    }

    static FeedsOrderBy convertGdFeedsOrderByFromGd(GdFeedsOrderBy gdFeedsOrderBy) {
        FeedOrderByField feedOrderByField = GdFeedOrderByField.toSource((gdFeedsOrderBy.getField()));
        SortOrder sortOrder = gdFeedsOrderBy.getOrder() == Order.DESC ? SortOrder.DESC : SortOrder.ASC;
        return new FeedsOrderBy()
                .withField(feedOrderByField)
                .withOrder(sortOrder);
    }

    public static GdUpdateStatus convertUpdateStatusToGd(UpdateStatus updateStatus) {
        return updateStatus != null ? GdUpdateStatus.valueOf(updateStatus.name()) : null;
    }

    private static GdFeedType convertFeedTypeToGd(FeedType feedType) {
        return feedType != null ? GdFeedType.valueOf(feedType.name()) : null;
    }

    public static GdBusinessType convertBusinessTypeToGd(BusinessType businessType) {
        return businessType != null ? GdBusinessType.valueOf(businessType.name()) : null;
    }

    private static GdSource convertSourceToGd(Source source, MasterSystem masterSystem) {
        if (masterSystem == MasterSystem.MANUAL) {
            return GdSource.MANUAL;
        }

        return source != null ? GdSource.valueOf(source.name()) : null;
    }

    private static GdFeedCategory convertFeedCategoryToGd(@Nonnull FeedCategory category) {
        return new GdFeedCategory()
                .withCategoryId(category.getCategoryId())
                .withName(category.getName())
                .withParentCategoryId(category.getParentCategoryId())
                .withIsDeleted(category.getIsDeleted())
                .withOfferCount(category.getOfferCount());
    }

    static List<GdFeedCategory> convertFeedCategoryListToGd(List<FeedCategory> categories) {
        if (categories == null) {
            return emptyList();
        }
        return StreamEx.of(categories)
                .map(FeedConverter::convertFeedCategoryToGd)
                .toList();
    }

    private static GdFeedDefect convertFeedDefectToGd(FeedHistoryItemParseResultsDefect error) {
        return new GdFeedDefect()
                .withCode(error.getCode())
                .withMessageEn(error.getMessageEn())
                .withMessageRu(error.getMessageRu());
    }

    @Nullable
    static List<GdFeedDefect> convertFeedDefectListToGd(List<FeedHistoryItemParseResultsDefect> errors) {
        if (isEmpty(errors)) {
            return null;
        }
        return StreamEx.of(errors)
                .map(FeedConverter::convertFeedDefectToGd)
                .toList();
    }

    public static GdFeed convertFeedToGd(Feed feed,
                                         @Nullable FeedHistoryItem history,
                                         List<GdCampaignTruncated> gdCampaigns,
                                         boolean operatorCanWrite) {
        GdFeed gdFeed = new GdFeed()
                .withId(feed.getId())
                .withName(feed.getName())
                .withFeedType(convertFeedTypeToGd(feed.getFeedType()))
                .withBusinessType(convertBusinessTypeToGd(feed.getBusinessType()))
                .withUpdateStatus(convertUpdateStatusToGd(feed.getUpdateStatus()))
                .withErrors(convertFeedDefectListToGd(Optional.ofNullable(history)
                        .map(FeedHistoryItem::getParseResults)
                        .map(FeedHistoryItemParseResults::getErrors)
                        .orElse(null)))
                .withWarnings(convertFeedDefectListToGd(Optional.ofNullable(history)
                        .map(FeedHistoryItem::getParseResults)
                        .map(FeedHistoryItemParseResults::getWarnings)
                        .orElse(null)))
                .withLogin(feed.getLogin())
                .withHasPassword(isNotBlank(feed.getPlainPassword()))
                .withFileName(feed.getFilename())
                .withLastChange(feed.getLastChange())
                .withSource(convertSourceToGd(feed.getSource(), feed.getMasterSystem()))
                .withOffersCount(feed.getOffersCount())
                .withFetchErrorsCount(feed.getFetchErrorsCount())
                .withIsRemoveUtm(feed.getIsRemoveUtm())
                .withIsReadOnly(feed.getMasterSystem() != MasterSystem.DIRECT)
                .withIsRefreshingAllowed(feed.getSource() == Source.URL && feed.getUpdateStatus() != UpdateStatus.NEW
                        && (feed.getLastChange() == null || ChronoUnit.HOURS.between(feed.getLastChange(), LocalDateTime.now()) > REFRESH_HOURS_PERIOD))
                .withIsShopInShop(feed.getMasterSystem() == MasterSystem.SHOP_IN_SHOP)
                .withTopVendors(feed.getTopVendors())
                .withCampaigns(gdCampaigns)
                .withAccess(convertToGdFeedAccess(feed.getUpdateStatus(), isEmpty(gdCampaigns), operatorCanWrite));

        if (feed.getMasterSystem() != MasterSystem.SHOP_IN_SHOP) {
            gdFeed.withUrl(convertUrlToGd(feed.getUrl()));
        }

        return gdFeed;
    }

    public static GdFeedAccess convertToGdFeedAccess(UpdateStatus updateStatus, boolean notUsedInCampaigns,
                                                     boolean operatorCanWrite) {
        return new GdFeedAccess()
                .withCanEdit(operatorCanWrite)
                .withCanDelete(operatorCanWrite && notUsedInCampaigns
                        && STATUSES_ALLOWED_TO_DELETE.contains(nvl(updateStatus, UpdateStatus.NEW)));
    }

    private static String convertUrlToGd(String url) {
        if (url == null) {
            return null;
        }
        String srcUrl = FeedUtilsKt.tryGetSrcUrl(url);
        return (srcUrl != null) ? srcUrl : url;
    }

    static List<GdFeed> convertFeedsToGd(List<Feed> feeds,
                                         Map<Long, FeedHistoryItem> historyItemsByFeedId,
                                         Map<Long, List<GdCampaignTruncated>> campaignsByFeedId,
                                         boolean operatorCanWrite) {
        return mapList(feeds, feed -> {
            List<GdCampaignTruncated> gdCampaigns = campaignsByFeedId.getOrDefault(feed.getId(), emptyList());
            return convertFeedToGd(feed, historyItemsByFeedId.get(feed.getId()), gdCampaigns, operatorCanWrite);
        });
    }

    public static Feed convertShopInShopBusinessInfoToFeed(ClientId clientId, ShopInShopBusinessInfo businessInfo) {
        String feedName = String.format(FEED_NAME_BY_SHOP_IN_SHOP_PATTERN,
                businessInfo.getBusinessId(), businessInfo.getTargetDomain());

        return new Feed()
                .withClientId(clientId.asLong())
                .withBusinessType(BusinessType.RETAIL)
                .withName(feedName)
                .withSource(Source.URL)
                .withUrl(businessInfo.getFeedUrl())
                .withUpdateStatus(UpdateStatus.NEW)
                .withMasterSystem(MasterSystem.SHOP_IN_SHOP)
                .withTargetDomain(businessInfo.getTargetDomain());
    }
}
