package ru.yandex.direct.grid.processing.service.feed;

import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.SortOrder;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.container.FeedsOrderBy;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedCategory;
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItem;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.processing.FeedOrderByField;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.uac.model.ShopInShopBusinessInfo;
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService;
import ru.yandex.direct.core.entity.uac.service.shopinshop.ShopInShopBusinessesService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.model.feed.GdFeed;
import ru.yandex.direct.grid.model.feed.GdFeedCategory;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.feed.GdFeedFeatures;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsContext;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsFilter;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsOrderBy;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.feed.FeedConverter.convertFeedsToGd;
import static ru.yandex.direct.grid.processing.service.feed.FeedConverter.convertShopInShopBusinessInfoToFeed;
import static ru.yandex.direct.grid.processing.util.ResultConverterHelper.getCountOfTrueBooleanValues;
import static ru.yandex.direct.utils.CollectionUtils.flatToSet;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class FeedDataService {

    private static final Logger LOGGER = LoggerFactory.getLogger(FeedDataService.class);

    private static final List<FeedsOrderBy> DEFAULT_FEEDS_ORDER_BY = singletonList(
            new FeedsOrderBy()
                    .withField(FeedOrderByField.LAST_CHANGE)
                    .withOrder(SortOrder.DESC));

    private static final Map<Source, Integer> FEEDS_PRIORITY_BY_SOURCE = new EnumMap<>(Map.of(
            Source.URL, 1,
            Source.FILE, 1,
            Source.SITE, 0
    ));

    private final FeedService feedService;
    private final CampaignService campaignService;
    private final CampaignInfoService campaignInfoService;
    private final EcomDomainsService ecomDomainsService;
    private final RbacService rbacService;
    private final ShopInShopBusinessesService shopInShopBusinessesService;
    private final FeatureService featureService;

    @Autowired
    public FeedDataService(FeedService feedService, CampaignService campaignService,
                           CampaignInfoService campaignInfoService, EcomDomainsService ecomDomainsService,
                           RbacService rbacService, ShopInShopBusinessesService shopInShopBusinessesService,
                           FeatureService featureService) {
        this.feedService = feedService;
        this.campaignService = campaignService;
        this.campaignInfoService = campaignInfoService;
        this.ecomDomainsService = ecomDomainsService;
        this.rbacService = rbacService;
        this.shopInShopBusinessesService = shopInShopBusinessesService;
        this.featureService = featureService;
    }

    public GdFeedsContext getGdFeeds(GdClient client,
                                     User operator, @Nullable GdFeedsFilter filter,
                                     @Nullable List<GdFeedsOrderBy> ordersBy,
                                     @Nullable GdLimitOffset limitOffset,
                                     @Nullable String url) {

        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        Long chiefUid = client.getInfo().getChiefUserId();

        boolean isEnableFeedSuggestionByShopInShopBusiness = featureService.isEnabledForClientId(clientId,
                FeatureName.ENABLE_FEED_AND_COUNTER_SUGGESTION_BY_SHOP_IN_SHOP);

        ShopInShopBusinessInfo businessInfo = null;
        if (isEnableFeedSuggestionByShopInShopBusiness && url != null) {
            businessInfo = shopInShopBusinessesService.getBusinessInfoByUrl(url);
        }

        // limitOffset не передаём, т.к. для поля TotalCount нужны все фиды соответствующие запросу
        List<Feed> feeds = getFeedsByClient(clientId, filter, ordersBy);
        Long defaultFeedId = null;
        if (businessInfo != null) {
            // Если переданный урл - ссылка на бизнес маркетплейса - саджестится фид по маркетплейсу
            try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:getFeedByShopInShopUrl")) {
                Feed feed = getFeedByShopInShopUrl(clientId, chiefUid, operator, businessInfo);
                if (feed != null) {
                    defaultFeedId = feed.getId();
                    feeds.add(feed);
                }
            }
        }

        if (defaultFeedId == null) {
            defaultFeedId = getDefaultFeedId(feeds, url);
        }

        LimitOffset range = normalizeLimitOffset(limitOffset);
        List<Feed> pageFeeds = feeds.stream()
                .skip(range.offset()).limit(range.limit())
                .collect(toList());
        Set<Long> pageFeedIds = listToSet(pageFeeds, Feed::getId);

        Map<Long, FeedHistoryItem> historyItemsByFeedId;
        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:getLatestFeedHistoryItems")) {
            historyItemsByFeedId = feedService.getLatestFeedHistoryItems(clientId, pageFeedIds);
        }

        Map<Long, List<GdCampaignTruncated>> campaignsByFeedId;
        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:getCampaignsByFeedId")) {
            campaignsByFeedId = getCampaignsByFeedId(clientId, pageFeedIds);
        }

        boolean operatorCanWrite;
        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:canWrite")) {
            operatorCanWrite = rbacService.canWrite(operator.getUid(), chiefUid);
        }

        List<GdFeed> gdFeeds;
        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:convert")) {
            gdFeeds = convertFeedsToGd(pageFeeds, historyItemsByFeedId, campaignsByFeedId, operatorCanWrite);
        }

        GdFeedFeatures feedFeatures = new GdFeedFeatures()
                .withCanBeDeletedCount(getCountOfTrueBooleanValues(gdFeeds,
                        gdFeed -> gdFeed.getAccess().getCanDelete()))
                .withCanBeEditedCount(getCountOfTrueBooleanValues(gdFeeds,
                        gdFeed -> gdFeed.getAccess().getCanEdit()));

        return new GdFeedsContext()
                .withTotalCount(feeds.size())
                .withRowset(gdFeeds)
                .withDefaultFeedId(defaultFeedId)
                .withFeatures(feedFeatures);
    }

    private List<Feed> getFeedsByClient(ClientId clientId,
                                        @Nullable GdFeedsFilter filter,
                                        @Nullable List<GdFeedsOrderBy> ordersBy) {
        FeedQueryFilter feedQueryFilter;
        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:filter")) {
            feedQueryFilter = getFeedQueryFilter(filter, ordersBy);
        }

        try (TraceProfile ignored = Trace.current().profile("grid.api.feeds:getFeeds")) {
            return feedService.getFeedsWithVendors(clientId, feedQueryFilter);
        }
    }

    private Long getDefaultFeedId(List<Feed> feeds,
                                  @Nullable String url) {
        String domain = ifNotNull(url, u -> {
            Map<String, String> mainMirror = ecomDomainsService.toMainMirrors(List.of(u));

            return ecomDomainsService.normalizeDomain(mainMirror.getOrDefault(u, u));
        });

        return feeds.stream()
                .filter(feed -> feed.getUpdateStatus() == UpdateStatus.DONE
                        || feed.getUpdateStatus() == UpdateStatus.UPDATING)
                .filter(feed -> domain == null || Objects.equals(feed.getTargetDomain(), domain))
                .filter(feed -> feed.getSource() != Source.SITE)
                .filter(feed -> feed.getMasterSystem() != MasterSystem.MANUAL)
                .max(Comparator
                        .comparing(Feed::getSource, Comparator
                                .comparing(FEEDS_PRIORITY_BY_SOURCE::get))
                        .thenComparing(Feed::getLastChange))
                .map(Feed::getId)
                .orElse(null);
    }

    @Nullable
    private Feed getFeedByShopInShopUrl(ClientId clientId, Long chiefUid, User operator,
                                        ShopInShopBusinessInfo businessInfo) {
        Feed feed = feedService.getShopInShopFeedByUrl(clientId, businessInfo.getFeedUrl());
        if (feed != null) {
            return feed;
        }

        feed = convertShopInShopBusinessInfoToFeed(clientId, businessInfo);
        MassResult<Long> result = feedService.addFeeds(clientId, chiefUid, operator.getUid(), List.of(feed));
        if (!result.isSuccessful()) {
            result.getErrors().forEach(e -> LOGGER.error(
                    "Error when add a new shop_in_shop feed by business_id = {}, for clientId = {}. Defect: {}",
                    businessInfo.getBusinessId(), clientId.asLong(), e)
            );
            return null;
        }

        return feedService.getShopInShopFeedByUrl(clientId, businessInfo.getFeedUrl());
    }

    private FeedQueryFilter getFeedQueryFilter(@Nullable GdFeedsFilter filter,
                                               @Nullable List<GdFeedsOrderBy> ordersBy) {
        FeedQueryFilter.Builder feedFilterBuilder = FeedQueryFilter.newBuilder();
        if (filter != null) {
            if (filter.getStatuses() != null) {
                List<UpdateStatus> updateStatuses = mapList(filter.getStatuses(),
                        FeedConverter::convertUpdateStatusFromGd);
                feedFilterBuilder.withUpdateStatuses(updateStatuses);
            }
            if (filter.getSources() != null) {
                List<Source> sources = mapList(filter.getSources(), FeedConverter::convertSourceFromGd);
                feedFilterBuilder.withSources(sources);
            }
            if (filter.getSearchBy() != null) {
                feedFilterBuilder.withSearchBy(filter.getSearchBy());
            }
            if (filter.getBusinessTypes() != null) {
                List<BusinessType> businessTypes = mapList(filter.getBusinessTypes(),
                        FeedConverter::convertBusinessTypeFromGd);
                feedFilterBuilder.withTypes(businessTypes);
            }
        }
        if (ordersBy != null) {
            List<FeedsOrderBy> orders = mapList(ordersBy, FeedConverter::convertGdFeedsOrderByFromGd);
            feedFilterBuilder.withOrders(orders);
        } else {
            feedFilterBuilder.withOrders(DEFAULT_FEEDS_ORDER_BY);
        }
        return feedFilterBuilder.build();
    }

    private Map<Long, List<GdCampaignTruncated>> getCampaignsByFeedId(ClientId clientId, Set<Long> feedIds) {
        Map<Long, Set<Long>> campaignIdsByFeedId = campaignService.getCampaignIdsByFeedId(clientId, feedIds);
        Set<Long> allCampaignIds = flatToSet(campaignIdsByFeedId.values());
        Map<Long, Long> masterIdBySubCampaignId = campaignService.getMasterIdBySubCampaignId(clientId, allCampaignIds);
        Set<Long> effectiveCampaignIds = StreamEx.of(allCampaignIds)
                .mapToEntry(Function.identity())
                .mapKeyValue(masterIdBySubCampaignId::getOrDefault)
                .toSet();
        Map<Long, GdCampaignTruncated> campaignById = campaignInfoService.getTruncatedCampaigns(clientId,
                effectiveCampaignIds);

        return EntryStream.of(campaignIdsByFeedId)
                .mapValues(campaignIds -> StreamEx.of(campaignIds)
                        .mapToEntry(Function.identity())
                        .mapKeyValue(masterIdBySubCampaignId::getOrDefault)
                        .distinct()
                        .map(campaignById::get)
                        .nonNull()
                        .toList())
                .toMap();
    }

    public List<GdFeedCategory> getGdFeedCategories(GdFeed feed, ClientId clientId) {
        List<FeedCategory> feedCategories;
        try (TraceProfile profile = Trace.current().profile("grid.api.feeds:getFeedCategories")) {
            feedCategories = feedService.getFeedCategories(clientId, singletonList(feed.getId()));
        }

        List<GdFeedCategory> result;
        try (TraceProfile profile = Trace.current().profile("grid.api.feeds:convertFeedCategoryListToGd", "",
                feedCategories.size())) {
            result = FeedConverter.convertFeedCategoryListToGd(feedCategories);
        }

        return result;
    }

}
