package ru.yandex.direct.grid.processing.service.feed;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.model.feed.GdFeed;
import ru.yandex.direct.grid.model.feed.GdFeedCategory;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.GdLimitOffset;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsContext;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsFilter;
import ru.yandex.direct.grid.processing.model.feed.GdFeedsOrderBy;

/**
 * Сервис, возвращающий данные о фидах клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class FeedsGraphQlService {
    private static final String FEEDS_RESOLVER_NAME = "feeds";

    private final FeedDataService feedDataService;

    @Autowired
    public FeedsGraphQlService(FeedDataService feedDataService) {
        this.feedDataService = feedDataService;
    }

    /**
     * GraphQL подзапрос. Получает информацию о фидах клиента, полученного из контекста выполнения запроса.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = FEEDS_RESOLVER_NAME)
    public GdFeedsContext getFeeds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @Nullable @GraphQLArgument(name = "filter") GdFeedsFilter filter,
            @Nullable @GraphQLArgument(name = "ordersBy") List<GdFeedsOrderBy> ordersBy,
            @Nullable @GraphQLArgument(name = "limitOffset") GdLimitOffset limitOffset,
            @Nullable @GraphQLArgument(name = "url") String url) {
        User operator = context.getOperator();
        return feedDataService.getGdFeeds(client, operator, filter, ordersBy, limitOffset, url);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "categories")
    public List<@GraphQLNonNull GdFeedCategory> getFeedCategories(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFeed feed) {
        ClientId clientId = ClientId.fromLong(context.getQueriedClient().getId());
        // На текущих сценариях не планируется запрашивать категории более чем для одного фида за раз
        return feedDataService.getGdFeedCategories(feed, clientId);
    }

}
