package ru.yandex.direct.grid.processing.service.feedfilter

import ru.yandex.direct.core.entity.feedfilter.model.FeedFilter
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterCondition
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterTab
import ru.yandex.direct.core.entity.performancefilter.model.Operator
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema
import ru.yandex.direct.core.entity.performancefilter.schema.parser.AbstractPerformanceConditionValueParser
import ru.yandex.direct.grid.processing.model.feedfilter.GdFeedFilter
import ru.yandex.direct.grid.processing.model.feedfilter.GdFeedFilterTab
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterCondition
import ru.yandex.direct.grid.processing.model.smartfilter.GdSmartFilterConditionOperator
import java.util.*

class FeedFilterGdConverter private constructor() {
    companion object {
        private val gdOperatorByCoreOperator: Map<Operator, GdSmartFilterConditionOperator> =
            listOf(
                Operator.GREATER to GdSmartFilterConditionOperator.GREATER_THAN,
                Operator.LESS to GdSmartFilterConditionOperator.LESS_THAN,
                Operator.EQUALS to GdSmartFilterConditionOperator.EQUALS_ANY,
                Operator.RANGE to GdSmartFilterConditionOperator.RANGE,
                Operator.CONTAINS to GdSmartFilterConditionOperator.CONTAINS_ANY,
                Operator.NOT_CONTAINS to GdSmartFilterConditionOperator.NOT_CONTAINS_ALL,
                Operator.EXISTS to GdSmartFilterConditionOperator.EXISTS
            ).toMap(EnumMap(Operator::class.java))

        private val coreOperatorByGdOperator = gdOperatorByCoreOperator
            .map { it.value to it.key }
            .toMap(EnumMap(GdSmartFilterConditionOperator::class.java))

        @JvmStatic
        fun fromGdFeedFilter(filter: GdFeedFilter, filterSchema: FilterSchema): FeedFilter = FeedFilter().apply {
            this.tab = fromGdFeedFilterTab(filter.tab)
            this.conditions = filter.conditions.map { fromGdCondition(it, filterSchema) }
        }

        @JvmStatic
        fun toGdFeedFilter(filter: FeedFilter, filterSchema: FilterSchema): GdFeedFilter = GdFeedFilter().apply {
            this.tab = toGdFeedFilterTab(filter.tab)
            this.conditions = filter.conditions.map { toGdCondition(it, filterSchema) }
        }

        @JvmStatic
        fun fromGdCondition(condition: GdSmartFilterCondition, filterSchema: FilterSchema): FeedFilterCondition<Any> =
            FeedFilterCondition<Any>(
                filterSchema.translateFieldNameFromGd(condition.field),
                fromGdConditionOperator(condition.operator),
                condition.stringValue
            ).also { setParsedValueForCondition(it, filterSchema) }

        @JvmStatic
        @Suppress("UNCHECKED_CAST")
        private fun setParsedValueForCondition(condition: FeedFilterCondition<Any>, filterSchema: FilterSchema) {
            val parser = filterSchema.getParser(condition.fieldName, condition.operator)
                    as AbstractPerformanceConditionValueParser<Any>
            condition.parsedValue = parser.parse(condition)
        }

        @JvmStatic
        private fun toGdCondition(
            condition: FeedFilterCondition<*>, filterSchema: FilterSchema
        ): GdSmartFilterCondition =
            GdSmartFilterCondition().apply {
                this.field = filterSchema.translateFieldNameForGd(condition.fieldName)
                this.operator = toGdConditionOperator(condition.operator)
                this.stringValue = condition.stringValue
            }

        @JvmStatic
        fun fromGdConditionOperator(operator: GdSmartFilterConditionOperator): Operator =
            coreOperatorByGdOperator[operator] ?: throw IllegalArgumentException("Unknown grid operator: $operator")

        @JvmStatic
        private fun toGdConditionOperator(operator: Operator): GdSmartFilterConditionOperator =
            gdOperatorByCoreOperator[operator] ?: throw IllegalArgumentException("Unknown core operator: $operator")

        @JvmStatic
        fun fromGdFeedFilterTab(tab: GdFeedFilterTab?): FeedFilterTab? =
            GdFeedFilterTab.toSource(tab)

        @JvmStatic
        private fun toGdFeedFilterTab(tab: FeedFilterTab?): GdFeedFilterTab? =
            GdFeedFilterTab.fromSource(tab)
    }
}
