package ru.yandex.direct.grid.processing.service.feedoffer;

import java.util.List;
import java.util.function.Function;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer;
import ru.yandex.direct.core.entity.feedoffer.service.FeedOfferService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.feedoffer.GdAddOrUpdateFeedOffer;
import ru.yandex.direct.grid.processing.model.feedoffer.GdAddOrUpdateFeedOfferPayload;
import ru.yandex.direct.grid.processing.model.feedoffer.GdFeedOffersContext;
import ru.yandex.direct.grid.processing.service.feedoffer.converter.GdFeedOfferConverter;
import ru.yandex.direct.result.MassResult;

import static ru.yandex.direct.grid.processing.service.feedoffer.converter.GdFeedOfferConverter.fromGdFeedOfferAddItem;
import static ru.yandex.direct.grid.processing.service.feedoffer.converter.GdFeedOfferConverter.getValidationResult;
import static ru.yandex.direct.grid.processing.util.ResponseConverter.getResults;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

@Service
public class FeedOfferDataService {

    private final FeedOfferService feedOfferService;
    private final FeedService feedService;

    @Autowired
    public FeedOfferDataService(FeedOfferService feedOfferService, FeedService feedService) {
        this.feedOfferService = feedOfferService;
        this.feedService = feedService;
    }

    public GdFeedOffersContext getFeedOffers(ClientId clientId) {
        Long feedId = ifNotNull(feedService.getManuallyAddedFeed(clientId), FeedSimple::getId);
        List<FeedOffer> offers = feedOfferService.getOffersByClientId(clientId);

        return new GdFeedOffersContext()
                .withFeedId(feedId)
                .withRowset(mapList(offers, GdFeedOfferConverter::toGdFeedOffer));
    }

    public GdAddOrUpdateFeedOfferPayload addOrUpdateFeedOffers(ClientId clientId, Long subjectUserUid,
                                                               Long operatorUid, GdAddOrUpdateFeedOffer input) {
        List<FeedOffer> offers = mapList(input.getManualFeedOffers(), offer -> fromGdFeedOfferAddItem(offer, clientId));
        MassResult<Long> result = feedOfferService.addOrUpdateFeedOffers(clientId, subjectUserUid, operatorUid, offers);
        GdValidationResult validationResult = getValidationResult(result.getValidationResult(),
                path(field(GdAddOrUpdateFeedOffer.MANUAL_FEED_OFFERS)));

        List<FeedOffer> feedOffers = feedOfferService.getOffersByClientId(clientId);

        Long feedId;
        if (validationResult == null) {
            MassResult<Long> feedResult = feedService
                    .addOrUpdateManualFeed(clientId, subjectUserUid, operatorUid, feedOffers);
            feedId = getResults(feedResult, Function.identity()).get(0);
        } else {
            feedId = ifNotNull(feedService.getManuallyAddedFeed(clientId), FeedSimple::getId);
        }

        return new GdAddOrUpdateFeedOfferPayload()
                .withFeedId(feedId)
                .withRowset(mapList(feedOffers, GdFeedOfferConverter::toGdFeedOffer))
                .withValidationResult(validationResult);
    }
}
