package ru.yandex.direct.grid.processing.service.forecast.clicks;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.advq.SearchKeywordResult;
import ru.yandex.direct.core.entity.keyword.model.ForecastCtr;
import ru.yandex.direct.grid.processing.model.showcondition.GdAuctionData;
import ru.yandex.direct.grid.processing.model.showcondition.GdAuctionDataItem;

import static com.google.common.primitives.Longs.min;
import static java.lang.Math.round;
import static java.util.Comparator.comparing;
import static ru.yandex.direct.core.entity.auction.service.BsAuctionService.DEFAULT_GUARANTEE_CTR;
import static ru.yandex.direct.core.entity.auction.service.BsAuctionService.DEFAULT_PREMIUM_CTR;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
public class ClicksForecaster {

    private static final Long MINIMUM_TRAFFIC_VOLUME_IN_PREMIUM = 60L;

    @Autowired
    public ClicksForecaster() {
    }

    public Long forecastClicks(Long budget,
                               @Nullable Double cpa,
                               List<String> phrases,
                               Map<String, GdAuctionData> auctionDataByPhrase,
                               Map<String, SearchKeywordResult> keywordShowStatByPhrase,
                               Map<String, ForecastCtr> forecastCtrByPhrase) {
        List<TrafficItem> trafficData = convertToTrafficData(phrases, auctionDataByPhrase,
                keywordShowStatByPhrase, forecastCtrByPhrase);

        // Выкупать можно лишь тот трафик, для которого выполяется cpc <= cpa
        if (cpa != null) {
            trafficData = filterList(trafficData, trafficItem -> trafficItem.getCpc() <= cpa);
        }

        return calcClicks(budget, trafficData);
    }

    private List<TrafficItem> convertToTrafficData(List<String> phrases,
                                                   Map<String, GdAuctionData> auctionDataByPhrase,
                                                   Map<String, SearchKeywordResult> keywordShowStatByPhrase,
                                                   Map<String, ForecastCtr> forecastCtrByPhrase) {
        return StreamEx.of(phrases)
                .mapToEntry(auctionDataByPhrase::get)
                .flatMapValues(auctionData -> auctionData.getAuctionDataItems().stream())
                .mapKeyValue((phrase, auctionDataItem) ->
                        convertToTrafficDataItem(phrase, keywordShowStatByPhrase.get(phrase), auctionDataItem,
                                forecastCtrByPhrase.get(phrase)))
                .toList();
    }

    private TrafficItem convertToTrafficDataItem(String phrase,
                                                 SearchKeywordResult searchKeywordResult,
                                                 GdAuctionDataItem auctionDataItem,
                                                 ForecastCtr keywordCtr) {
        Long maxShows = searchKeywordResult.getResult().getTotalCount();
        Double expectedShowsInPosition = maxShows * auctionDataItem.getTrafficVolume() / 100.;
        Double ctr = calcCtr(keywordCtr, auctionDataItem.getTrafficVolume());
        Long expectedClicks = round(expectedShowsInPosition * ctr);

        return new TrafficItem(phrase, expectedClicks, auctionDataItem.getAmnestyPrice().doubleValue());
    }

    private Double calcCtr(ForecastCtr keywordCtr, Long trafficVolume) {
        if (trafficVolume >= MINIMUM_TRAFFIC_VOLUME_IN_PREMIUM) {
            return Optional.ofNullable(keywordCtr)
                    .map(ForecastCtr::getPremiumCtr)
                    .filter(ctr -> ctr > 0)
                    .orElse(DEFAULT_PREMIUM_CTR);
        } else {
            return Optional.ofNullable(keywordCtr)
                    .map(ForecastCtr::getGuaranteeCtr)
                    .filter(ctr -> ctr > 0)
                    .orElse(DEFAULT_GUARANTEE_CTR);
        }
    }

    private Long calcClicks(final Long budget, List<TrafficItem> traffic) {
        traffic.sort(comparing(TrafficItem::getCpc));
        Map<String, BoughtTrafficItem> trafficByPhrase = new HashMap<>();
        double budgetLeft = (double) budget;
        Long boughtClicks = 0L;

        for (TrafficItem trafficItem : traffic) {
            String phrase = trafficItem.getPhrase();
            BoughtTrafficItem boughtTrafficItem = trafficByPhrase.get(phrase);

            if (!shouldBuyTrafficItem(trafficItem, boughtTrafficItem, budgetLeft)) {
                continue;
            }

            if (boughtTrafficItem != null) {
                budgetLeft += boughtTrafficItem.getCost();
                boughtClicks -= boughtTrafficItem.getBoughtClicks();
            }

            Long clicksToBuy = maxClicksCanBuy(trafficItem, budgetLeft);
            budgetLeft -= clicksToBuy * trafficItem.getCpc();
            boughtClicks += clicksToBuy;
            trafficByPhrase.put(phrase, new BoughtTrafficItem(trafficItem, clicksToBuy));
        }

        return boughtClicks;
    }

    private boolean shouldBuyTrafficItem(TrafficItem trafficItem,
                                         BoughtTrafficItem boughtTrafficItem,
                                         double budgetLeft) {
        if (boughtTrafficItem == null) {
            return true;
        }
        if (!boughtTrafficItem.isBoughtMaxClicks()) {
            return false;
        }

        Long boughtClicks = boughtTrafficItem.getBoughtClicks();
        long clicksCanBuy = maxClicksCanBuy(trafficItem, budgetLeft + boughtTrafficItem.getCost());

        return clicksCanBuy > boughtClicks;
    }

    private Long maxClicksCanBuy(TrafficItem trafficItem, double budget) {
        return min((long) (budget / trafficItem.getCpc()), trafficItem.getMaxClicks());
    }
}
