package ru.yandex.direct.grid.processing.service.freelancer;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerContacts;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedbackComment;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOfferDuration;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerStatus;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.constants.GdFreelancerSkillType;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerAdvQuality;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerCard;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerCertificate;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerCertificateType;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerContacts;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFeedback;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFeedbackComment;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerSkill;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerSkillOfferDuration;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerStatus;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerUgcModerationStatus;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancersCardStatusModerate;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackForFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerCardItem;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerContactsItem;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerSkillItem;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертер из core-модели {@link Freelancer} в модель интерфейса {@link GdFreelancer}
 *
 * @implNote Во внутренней модели время храним по московской таймзоне, отдавать в Gd должны в ней же.
 */
@ParametersAreNonnullByDefault
@Component
public class FreelancerConverter {
    private static final Logger logger = LoggerFactory.getLogger(FreelancerConverter.class);

    private static final String CHECK_CERT_LINK_TEMPLATE =
            "https://yandex.ru/adv/expert/certificates?certId=%s&lastname=%s";


    public FreelancerConverter() {
    }

    @Nonnull
    <T extends GdFreelancer> T convertToGd(Freelancer freelancer, Supplier<T> modelSupplier,
                                           @Nullable String avatarUrl) {
        T model = modelSupplier.get();
        model.withFreelancerId(freelancer.getFreelancerId())
                .withFio(freelancer.getFirstName() + " " + freelancer.getSecondName())
                .withRegionId(freelancer.getRegionId())
                .withRating(freelancer.getRating())
                .withFeedbackCount(freelancer.getFeedbackCount())
                .withCertificates(convertCertificatesToGd(freelancer.getCertificates(), freelancer.getSecondName()))
                .withStatus(convertToGd(freelancer.getStatus()))
                .withCard(convertToGd(freelancer.getCard(), avatarUrl, freelancer.getRegionId()));
        return model;
    }

    @Nonnull
    <T extends GdFreelancer> T convertToGd(Freelancer freelancer, User user, CurrencyCode currencyCode,
                                           Supplier<T> modelSupplier,
                                           @Nullable String avatarUrl) {
        T model = convertToGd(freelancer, modelSupplier, avatarUrl);
        model.withLogin(user.getLogin())
                .withWorkCurrency(currencyCode)
                .withCreateTime(user.getCreateTime());
        return model;
    }

    @Nonnull
    FreelancerBase convertFromGd(Long freelancerId, GdUpdateFreelancer gdFreelancer) {
        return new Freelancer()
                .withFreelancerId(freelancerId)
                .withRegionId(gdFreelancer.getRegionId());
    }

    @Nonnull
    GdFreelancerStatus convertToGd(FreelancerStatus status) {
        switch (status) {
            case FREE:
                return GdFreelancerStatus.FREE;
            case BUSY:
                return GdFreelancerStatus.BUSY;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + status);
        }
    }

    @Nonnull
    FreelancerStatus convertFromGd(GdFreelancerStatus status) {
        switch (status) {
            case FREE:
                return FreelancerStatus.FREE;
            case BUSY:
                return FreelancerStatus.BUSY;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + status);
        }
    }

    @Nonnull
    List<GdFreelancerCertificate> convertCertificatesToGd(@Nullable List<FreelancerCertificate> certificates,
                                                          String freelancerSecondName) {
        if (certificates == null) {
            return emptyList();
        }
        return mapList(certificates, t -> convertToGd(t, freelancerSecondName));
    }

    @Nonnull
    private GdFreelancerCertificate convertToGd(FreelancerCertificate certificate, String freelancerSecondName) {
        return new GdFreelancerCertificate()
                .withCertId(certificate.getCertId())
                .withLink(String.format(CHECK_CERT_LINK_TEMPLATE, certificate.getCertId(), freelancerSecondName))
                .withType(GdFreelancerCertificateType.valueOf(certificate.getType().name()));
    }

    @Nonnull
    List<FreelancerCertificate> convertCertificatesFromGd(
            @Nullable List<GdFreelancerCertificate> certificates) {
        if (certificates == null) {
            return emptyList();
        }
        return mapList(certificates, this::convertFromGd);
    }

    @Nonnull
    private FreelancerCertificate convertFromGd(GdFreelancerCertificate certificate) {
        return new FreelancerCertificate()
                .withCertId(certificate.getCertId())
                .withType(FreelancerCertificateType.valueOf(certificate.getType().name()));
    }

    GdFreelancerCard convertToGd(@Nullable FreelancerCard freelancerCard, @Nullable String avatarUrl,
                                 @Nullable Long regionId) {
        if (freelancerCard == null) {
            return null;
        }
        return new GdFreelancerCard()
                .withAvatarUrl(avatarUrl)
                .withBriefInfo(freelancerCard.getBriefInfo())
                .withContacts(convertContactsToGd(freelancerCard.getContacts(), regionId))
                .withStatusModerate(GdFreelancersCardStatusModerate.fromSource(freelancerCard.getStatusModerate()));
    }

    @Nonnull
    FreelancerCard convertFromGd(Long freelancerId, GdFreelancerCard freelancerCard) {
        return new FreelancerCard()
                .withId(freelancerId)
                .withBriefInfo(freelancerCard.getBriefInfo())
                .withContacts(convertFromGd(freelancerCard.getContacts()))
                .withStatusModerate(GdFreelancersCardStatusModerate.toSource(freelancerCard.getStatusModerate()));
    }

    @Nonnull
    FreelancerCard convertFromGd(Long freelancerId, GdUpdateFreelancerCardItem card) {
        return new FreelancerCard()
                .withId(freelancerId)
                .withAvatarId(card.getAvatarId())
                .withBriefInfo(card.getBriefInfo())
                .withContacts(convertFromGd(card.getContacts()));
    }

    @Nonnull
    GdFreelancerContacts convertContactsToGd(@Nullable FreelancerContacts contacts, @Nullable Long regionId) {
        if (contacts == null) {
            // ситуация не должна возникать, если данные в БД корректны
            logger.warn("Missed contacts for freelancer");
            return new GdFreelancerContacts()
                    .withPhone("")
                    .withEmail("");
        }
        String siteUrl = contacts.getSiteUrl();
        // town в Contacts сейчас всегда возвращаем null
        return new GdFreelancerContacts()
                .withEmail(contacts.getEmail())
                .withIcq(contacts.getIcq())
                .withPhone(contacts.getPhone())
                .withSiteUrl(siteUrl)
                .withTelegram(contacts.getTelegram())
                .withSkype(contacts.getSkype())
                .withViber(contacts.getViber())
                .withWhatsApp(contacts.getWhatsApp());
    }

    @Nullable
    FreelancerContacts convertFromGd(@Nullable GdFreelancerContacts contacts) {
        if (contacts == null) {
            return null;
        }
        return new FreelancerContacts()
                .withEmail(contacts.getEmail())
                .withIcq(contacts.getIcq())
                .withPhone(contacts.getPhone())
                .withSiteUrl(contacts.getSiteUrl())
                .withTelegram(contacts.getTelegram())
                .withSkype(contacts.getSkype())
                .withViber(contacts.getViber())
                .withWhatsApp(contacts.getWhatsApp());
    }

    @Nullable
    FreelancerContacts convertFromGd(@Nullable GdUpdateFreelancerContactsItem contacts) {
        if (contacts == null) {
            return null;
        }
        return new FreelancerContacts()
                .withEmail(contacts.getEmail())
                .withIcq(contacts.getIcq())
                .withPhone(contacts.getPhone())
                .withSiteUrl(contacts.getSiteUrl())
                .withTelegram(contacts.getTelegram())
                .withSkype(contacts.getSkype())
                .withViber(contacts.getViber())
                .withWhatsApp(contacts.getWhatsApp());
    }

    @Nonnull
    GdFreelancerSkill convertToGd(@Nonnull FreelancerSkillOffer offer) {
        FreelancerSkill freelancerSkill = FreelancerSkill.getById(offer.getSkillId());
        return new GdFreelancerSkill()
                .withFreelancerId(offer.getFreelancerId())
                .withSkillId(freelancerSkill.getSkillId())
                .withSkillCode(freelancerSkill.getSkillCode())
                .withName(freelancerSkill.getDescription())
                .withPrice(offer.getPrice())
                .withDays(durationToDays(offer.getDuration()))
                .withDuration(GdFreelancerSkillOfferDuration.fromSource(offer.getDuration()));
    }

    @Nonnull
    FreelancerSkillOffer convertFromGd(ClientId freelancerId, GdUpdateFreelancerSkillItem skillItem) {
        FreelancerSkillOfferDuration duration;
        if (skillItem.getDuration() != null) {
            duration = GdFreelancerSkillOfferDuration.toSource(skillItem.getDuration());
        } else {
            duration = durationFromDays(skillItem.getDays());
        }
        return new FreelancerSkillOffer()
                .withFreelancerId(freelancerId.asLong())
                .withSkillId(skillItem.getSkillId())
                .withDuration(duration)
                .withPrice(BigDecimal.valueOf(skillItem.getPrice()));
    }

    private String durationToDays(@Nullable FreelancerSkillOfferDuration duration) {
        //todo mariachernova: Временный маппер, чтобы не сломался фронт
        if (duration == null) {
            return "";
        }
        switch (duration) {
            case FROM_1_TO_3_DAYS:
                return "1-3";
            case FROM_3_TO_7_DAYS:
                return "3-7";
            case FROM_7_TO_14_DAYS:
                return "7-14";
            case FROM_14_TO_28_DAYS:
                return "14-28";
            case FROM_1_TO_3_MONTHS:
                return "28-90";
            case MORE_THAN_3_MONTHS:
                return "90-";
            case MONTHLY:
                return "28-90";
            case NOT_DEFINED:
                return "28-90";
            default:
                logger.warn("Unknown duration value: " + duration.name());
                return "";
        }
    }

    private FreelancerSkillOfferDuration durationFromDays(@Nullable String days) {
        //todo mariachernova: Временный маппер, чтобы не сломался фронт
        if (days == null || days.isEmpty()) {
            return FreelancerSkillOfferDuration.NOT_DEFINED;
        }
        try {
            return FreelancerSkillOfferDuration.valueOf(days);
        } catch (IllegalArgumentException e) {
            // Заменяем тире на дефис
            days = days.replaceAll("—", "-");
            switch (days) {
                case "1-3":
                    return FreelancerSkillOfferDuration.FROM_1_TO_3_DAYS;
                case "3-7":
                    return FreelancerSkillOfferDuration.FROM_3_TO_7_DAYS;
                case "7-14":
                    return FreelancerSkillOfferDuration.FROM_7_TO_14_DAYS;
                case "14-28":
                    return FreelancerSkillOfferDuration.FROM_14_TO_28_DAYS;
                case "28-90":
                    return FreelancerSkillOfferDuration.FROM_1_TO_3_MONTHS;
                case "90-":
                    return FreelancerSkillOfferDuration.MORE_THAN_3_MONTHS;
                case "ежемесячно":
                    return FreelancerSkillOfferDuration.MONTHLY;
                default:
                    logger.warn("Unknown service time value: " + days);
                    return FreelancerSkillOfferDuration.NOT_DEFINED;
            }
        }
    }

    FreelancerFeedback convertFreelancerFeedbackFromGd(Long operatorUid,
                                                       GdAddFeedbackForFreelancerRequest input) {
        return new FreelancerFeedback()
                .withAuthorUid(operatorUid)
                .withFreelancerId(input.getFreelancerId())
                .withFeedbackText(input.getFeedbackText())
                // Задаем время в московской таймзоне
                .withCreatedTime(ZonedDateTime.now().withZoneSameInstant(MSK))
                .withOverallMark(input.getOverallMark())
                .withWillRecommend(input.getWillRecommend())
                .withSkills(emptyList());
    }

    /**
     * Преобразует отзыв в Gd-представление. Если {@code authorUid} совпадает с {@code operatorUid},
     * в отзыве заполняется {@code willRecommend}. Это поле изспользуется в окне редактирования отзыва.
     */
    GdFreelancerFeedback convertFreelancerFeedbackToGd(FreelancerFeedback feedback, Long operatorUid) {
        GdFreelancerFeedback result = new GdFreelancerFeedback()
                .withFeedbackId(feedback.getFeedbackId())
                .withAuthorUid(feedback.getAuthorUid())
                .withFreelancerId(feedback.getFreelancerId())
                .withFeedbackText(feedback.getFeedbackText())
                .withSkills(convertFreelancerSkillListToGd(feedback.getSkills()))
                .withCreatedTime(convertZonedDateTimeToLocal(feedback.getCreatedTime()))
                .withUpdatedTime(convertZonedDateTimeToLocal(feedback.getUpdatedTime()))
                .withOverallMark(feedback.getOverallMark())
                .withModerationStatus(convertModerationStatusToGd(feedback.getModerationStatus()))
                .withComments(emptyList());
        if (operatorUid.equals(feedback.getAuthorUid())) {
            return result.withWillRecommend(feedback.getWillRecommend());
        }
        return result;
    }

    private LocalDateTime convertZonedDateTimeToLocal(@Nullable ZonedDateTime zonedDateTime) {
        if (zonedDateTime == null) {
            return null;
        }
        return zonedDateTime.withZoneSameInstant(MSK).toLocalDateTime();
    }

    /**
     * Преобразует описания услуг из представления ядра ({@link FreelancerSkill}) в Gd-представление.
     */
    @Nonnull
    public List<GdFreelancerSkillType> convertFreelancerSkillListToGd(List<FreelancerSkill> skills) {
        return mapList(skills, this::convertFreelancerSkillToGd);
    }

    @Nonnull
    private GdFreelancerSkillType convertFreelancerSkillToGd(FreelancerSkill freelancerSkill) {
        return new GdFreelancerSkillType()
                .withSkillId(freelancerSkill.getSkillId())
                .withSkillCode(freelancerSkill.getSkillCode())
                .withName(freelancerSkill.getDescription());
    }

    private GdFreelancerUgcModerationStatus convertModerationStatusToGd(
            @Nullable FreelancerUgcModerationStatus status) {
        if (status == null) {
            return GdFreelancerUgcModerationStatus.IN_PROGRESS;
        }
        switch (status) {
            case ACCEPTED:
                return GdFreelancerUgcModerationStatus.ACCEPTED;
            case DECLINED:
                return GdFreelancerUgcModerationStatus.DECLINED;
            case IN_PROGRESS:
                return GdFreelancerUgcModerationStatus.IN_PROGRESS;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + status);
        }
    }

    FreelancerUgcModerationStatus convertModerationStatusFromGd(GdFreelancerUgcModerationStatus status) {
        switch (status) {
            case ACCEPTED:
                return FreelancerUgcModerationStatus.ACCEPTED;
            case DECLINED:
                return FreelancerUgcModerationStatus.DECLINED;
            case IN_PROGRESS:
                return FreelancerUgcModerationStatus.IN_PROGRESS;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + status);
        }
    }

    FreelancerFeedbackComment convertFeedbackCommentFromGd(Long operatorUid,
                                                           GdAddFeedbackCommentRequest input) {
        return new FreelancerFeedbackComment()
                .withAuthorUid(operatorUid)
                .withFeedbackId(input.getFeedbackId())
                .withCommentText(input.getCommentText());
    }

    GdFreelancerFeedbackComment convertFeedbackCommentToGd(FreelancerFeedbackComment feedback) {
        return new GdFreelancerFeedbackComment()
                .withFeedbackId(feedback.getFeedbackId())
                .withCommentId(feedback.getCommentId())
                .withAuthorUserId(feedback.getAuthorUid())
                .withIsYandexSupport(feedback.getIsYandexSupport())
                .withRespondToCommentId(feedback.getRespondToCommentId())
                .withModerationStatus(GdFreelancerUgcModerationStatus.fromSource(feedback.getModerationStatus()))
                .withCreatedTime(feedback.getCreatedTime())
                .withUpdatedTime(feedback.getUpdatedTime());
    }

    GdFreelancerAdvQuality convertToGdFreelancerAdvQuality(Freelancer freelancer) {
        return new GdFreelancerAdvQuality()
                .withFreelancerId(freelancer.getFreelancerId())
                .withAdvQualityRating(freelancer.getAdvQualityRating())
                .withAdvQualityRank(freelancer.getAdvQualityRank());
    }
}
