package ru.yandex.direct.grid.processing.service.freelancer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PostConstruct;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectFilter;
import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.container.SenderKeys;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatar;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarId;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectIdentity;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerStatus;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerCardService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerClientAvatarService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerFeedbackService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerSkillOffersService;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerUpdateService;
import ru.yandex.direct.core.entity.freelancer.utils.ClientAvatarIdUtils;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.grid.processing.model.client.GdUserPublicInfo;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.model.freelancer.GdClientProject;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerAdvQuality;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerCard;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerClient;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFeedback;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFilter;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFull;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerPrivateData;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerProject;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerProjectFilter;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerRegion;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerRegionTranslation;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerRegionTranslationKey;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerSkill;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAcceptFlServiceByFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAcceptFlServiceByFreelancerRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackForFreelancerPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackForFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByClientRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByClientRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByFreelancerRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFreelancerFeedbackPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFreelancerFeedbackRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdRequestFlServiceByClientRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdRequestFlServiceByClientRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFlStatusRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerCard;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerFeedbackPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerFeedbackRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerSkill;
import ru.yandex.direct.grid.processing.service.operator.UserDataLoader;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.sender.YandexSenderClient;
import ru.yandex.direct.sender.YandexSenderTemplateParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Iterables.getOnlyElement;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.activeFreelancers;
import static ru.yandex.direct.core.entity.freelancer.utils.ClientAvatarIdUtils.clientAvatarIdFromCards;
import static ru.yandex.direct.core.entity.freelancer.utils.ClientAvatarIdUtils.clientAvatarIdFromParams;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Сервис, возвращающий данные о фрилансерах
 */
@Service
@ParametersAreNonnullByDefault
public class FreelancerDataService {

    // все услуги фрилансеров, независимо от страны, должны быть оценены в рублях, подробнее здесь: DIRECT-83462
    private static final CurrencyCode CURRENCY_CODE = CurrencyCode.RUB;
    protected static final String CLIENT_CLIENT_ID = "ClientID";
    protected static final String CLIENT_NAME = "client_name";
    protected static final String FREELANCER_NAME = "freelancer_name";
    protected static final String LINK_TO_FREELANCER_CARD = "link_to_freelancer_card";
    protected static final String REQUESTS = "freelancer_requests";
    protected static final String CLIENT_PHONE_NUMBER = "client_phone_number";
    protected static final String CLIENT_EMAIL = "client_email";
    protected static final String CLIENT_LOGIN = "client_login";
    protected static final String LINK_TO_INTERFACE = "link_to_interface";
    protected static final String FEEDBACK_FORM = "link_to_feedback_form";
    private final FreelancerService freelancerService;
    private final FreelancerUpdateService freelancerUpdateService;
    private final ClientService clientService;
    private final UserService userService;
    private final RbacService rbacService;
    private final GridValidationService gridValidationService;
    private final FreelancerConverter converter;
    private final FreelancerProjectConverter projectConverter;
    private final FreelancerClientAvatarService freelancerClientAvatarService;
    private final FreelancerSkillOffersService freelancerSkillOffersService;
    private final FreelancerFeedbackService freelancerFeedbackService;
    private final FreelancerCardService freelancerCardService;
    // data loaders
    private final UserDataLoader userDataLoader;
    private final FreelancerRegionTranslationDataLoader freelancerRegionTranslationDataLoader;
    private final GeoBaseHelper geoBaseHelper;
    private final YandexSenderClient yandexSenderClient;
    private final DirectConfig config;
    private final EnvironmentType environmentType;
    private SenderKeys senderKeys;
    private String freelancerPagePrefix;
    private String freelancerRequests;
    private String linkToFeedbackForm;

    @Autowired
    public FreelancerDataService(FreelancerService freelancerService,
                                 FreelancerUpdateService freelancerUpdateService,
                                 ClientService clientService, UserService userService,
                                 RbacService rbacService,
                                 GridValidationService gridValidationService,
                                 FreelancerConverter converter,
                                 FreelancerProjectConverter projectConverter,
                                 FreelancerClientAvatarService freelancerClientAvatarService,
                                 FreelancerSkillOffersService freelancerSkillOffersService,
                                 FreelancerFeedbackService freelancerFeedbackService,
                                 FreelancerCardService freelancerCardService,
                                 UserDataLoader userDataLoader,
                                 FreelancerRegionTranslationDataLoader freelancerRegionTranslationDataLoader,
                                 GeoBaseHelper geoBaseHelper,
                                 YandexSenderClient yandexSenderClient,
                                 DirectConfig config,
                                 EnvironmentType environmentType) {
        this.freelancerService = freelancerService;
        this.freelancerUpdateService = freelancerUpdateService;
        this.clientService = clientService;
        this.userService = userService;
        this.rbacService = rbacService;
        this.gridValidationService = gridValidationService;
        this.converter = converter;
        this.projectConverter = projectConverter;
        this.freelancerClientAvatarService = freelancerClientAvatarService;
        this.freelancerSkillOffersService = freelancerSkillOffersService;
        this.freelancerFeedbackService = freelancerFeedbackService;
        this.freelancerCardService = freelancerCardService;
        this.userDataLoader = userDataLoader;
        this.freelancerRegionTranslationDataLoader = freelancerRegionTranslationDataLoader;
        this.geoBaseHelper = geoBaseHelper;
        this.yandexSenderClient = yandexSenderClient;
        this.config = config;
        this.environmentType = environmentType;
    }

    @PostConstruct
    private void init() {
        this.senderKeys = new SenderKeys(config.getBranch("freelancers.sender_keys"));
        String host = environmentType.equals(EnvironmentType.DEVTEST) ||
                environmentType.equals(EnvironmentType.DEV7) ?
                System.getProperty("beta.url") :
                config.getString("web.host");
        this.freelancerPagePrefix = host + "/dna/freelancers/";
        this.freelancerRequests = host + "/dna/customers?tab=requested";
        this.linkToFeedbackForm = config.getString("common_links.link_to_feedback_form");
    }

    /**
     * Количество активных специалистов
     *
     * @see FreelancersQueryFilter#activeFreelancers()
     */
    public int getFreelancersCount() {
        List<Freelancer> freelancers =
                freelancerService.getFreelancers(activeFreelancers().build(), LimitOffset.maxLimited());
        return freelancers.size();
    }

    public <T extends GdFreelancer> List<T> getFreelancers(@Nullable GdFreelancerFilter filter,
                                                           Supplier<T> modelSupplier,
                                                           @Nullable Long operatorUid) {
        List<Freelancer> freelancers;
        List<Long> freelancersIds = new ArrayList<>();
        if (filter == null) {
            freelancers = freelancerService.getFreelancers(activeFreelancers().build(), LimitOffset.maxLimited());
            freelancers = StreamEx.of(freelancers)
                    .remove(f -> f.getCard() == null)
                    .filter(freelancerHasValidDirectCert())
                    .toList();
        } else {
            if (filter.getFreelancerIds() != null) {
                freelancersIds.addAll(filter.getFreelancerIds());
            }
            if (filter.getLogins() != null) {
                Map<String, Long> clientIdsByLogin = userService.massGetClientIdsByLogin(filter.getLogins());
                if (clientIdsByLogin.size() < filter.getLogins().size()) {
                    throw new IllegalArgumentException("Nonexistent logins were passed");
                }
                freelancersIds.addAll(clientIdsByLogin.values());
            }
            freelancers = freelancerService.getFreelancers(freelancersIds);
        }
        List<ClientId> freelancerIds = freelancers.stream()
                .map(Freelancer::getId)
                .map(ClientId::fromLong)
                .collect(toList());
        // Получаем ФИО фрилансера как ФИС главного представителя соответствующего клиента
        Map<ClientId, Long> chiefsByFreelancerIds = rbacService.getChiefsByClientIds(freelancerIds);
        Map<Long, User> usersByUid =
                listToMap(userService.massGetUser(chiefsByFreelancerIds.values()), User::getUid);
        Map<Long, User> usersByFreelancerIds = EntryStream.of(chiefsByFreelancerIds)
                .mapValues(usersByUid::get)
                .mapKeys(ClientId::asLong)
                .toMap();

        List<FreelancerCard> freelancerCards = StreamEx.of(freelancers)
                .map(Freelancer::getCard)
                .filter(Objects::nonNull)
                .toList();
        List<ClientAvatarId> avatarIdByClientId = clientAvatarIdFromCards(freelancerCards);
        Map<Long, String> avatarUrlByClientId = freelancerClientAvatarService.massGetUrlSize180(avatarIdByClientId);

        List<T> result = new ArrayList<>(freelancers.size());
        for (Freelancer f : freelancers) {
            User user = usersByFreelancerIds.get(f.getId());
            String avatarUrl = avatarUrlByClientId.get(f.getId());
            T t = converter.convertToGd(f, user, CURRENCY_CODE, modelSupplier, avatarUrl);
            result.add(t);
        }
        if (filter == null) {
            // При выводе общего списка показываем фрилансеров рандомно. Подробности: DIRECT-84992
            // Если operatorUid не задан, перемешиваем каждый час.
            Random rnd = new Random(operatorUid == null ? System.currentTimeMillis() / (1000 * 3600) : operatorUid);
            Collections.shuffle(result, rnd);
        }
        return result;
    }

    /**
     * Условие "имеется ли у специалиста сертификат по Директу"
     */
    private Predicate<Freelancer> freelancerHasValidDirectCert() {
        return f -> StreamEx.of(f.getCertificates())
                .anyMatch(c -> c.getType() == FreelancerCertificateType.DIRECT ||
                        c.getType() == FreelancerCertificateType.DIRECT_PRO);
    }

    private String getAvatarUrl(Freelancer addedFreelancer) {
        Long freelancerId = addedFreelancer.getFreelancerId();
        Long avatarId = addedFreelancer.getCard().getAvatarId();
        ClientAvatar clientAvatarId = clientAvatarIdFromParams(freelancerId, avatarId);
        String url = freelancerClientAvatarService.getUrlSize180(clientAvatarId);
        checkState(url != null,
                "FreelancerClientAvatarService hasn't returned even default avatar-url. FreelancerId=%1$d",
                freelancerId);
        return url;
    }

    Map<Long, GdFreelancerCard> getNewestFreelancerCards(List<Long> freelancerIds) {
        List<FreelancerCard> newestFreelancerCards = freelancerCardService.getNewestFreelancerCards(freelancerIds);
        Collection<ClientAvatarId> clientAvatarIds =
                mapList(newestFreelancerCards, ClientAvatarIdUtils::clientAvatarIdFromCard);
        Map<Long, String> avatarUrlByClientId = freelancerClientAvatarService.massGetUrlSize180(clientAvatarIds);

        Function<FreelancerCard, GdFreelancerCard> cardToGdMapper =
                card -> {
                    GdFreelancerCard gdFreelancerCard =
                            converter.convertToGd(card, avatarUrlByClientId.get(card.getFreelancerId()), null);
                    // очищаем nullable-поле contacts.town
                    // Планируем его удалить в freelancersList.card. Здесь в newestCard его ещё не должны использовать
                    gdFreelancerCard.getContacts().setTown(null);
                    return gdFreelancerCard;
                };
        return StreamEx.of(newestFreelancerCards)
                .mapToEntry(FreelancerCard::getFreelancerId, cardToGdMapper)
                .distinctKeys()
                .toMap();
    }

    /**
     * Обновление карточки фрилансера. Возвращается фрилансер с обновлёнными данными
     */
    GdFreelancerFull updateFreelancerCard(ClientId freelancerId, GdUpdateFreelancerCard input) {
        FreelancerCard card = converter.convertFromGd(freelancerId.asLong(), input.getCard());
        Result<Long> result = freelancerCardService.addChangedFreelancerCard(freelancerId, card);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result.getValidationResult());
        Freelancer freelancer = freelancerService.getFreelancerWithNewestCard(freelancerId.asLong());
        checkNotNull(freelancer, "Can't find freelancer after successful update");
        String avatarUrl = getAvatarUrl(freelancer);
        return converter.convertToGd(freelancer, GdFreelancerFull::new, avatarUrl);
    }

    /**
     * Существующие услуги обновляются переданным списком.
     * Услуги, которых нет в переданном списке, удаляются.
     * Возвращается обновленный список услуг.
     */
    List<GdFreelancerSkill> updateFreelancerSkills(ClientId freelancerId, GdUpdateFreelancerSkill input) {
        List<FreelancerSkillOffer> freelancerSkillOffers =
                mapList(input.getSkills(), skillId -> converter.convertFromGd(freelancerId, skillId));
        List<Long> newSkillOfferIds = mapList(freelancerSkillOffers, FreelancerSkillOffer::getSkillId);

        MassResult<FreelancerSkillOffer> offerResult =
                freelancerSkillOffersService.setFreelancerSkillOffer(freelancerId, freelancerSkillOffers);
        gridValidationService.throwGridValidationExceptionIfHasErrors(offerResult.getValidationResult());

        List<FreelancerSkillOffer> existingSkillsOffers =
                freelancerSkillOffersService.getFreelancerSkillsOffers(singletonList(freelancerId.asLong()));

        List<Long> offerIdsToDelete = StreamEx.of(existingSkillsOffers)
                .map(FreelancerSkillOffer::getSkillId)
                .remove(newSkillOfferIds::contains)
                .toList();
        freelancerSkillOffersService.deleteFreelancerSkillsOffer(freelancerId, offerIdsToDelete);

        return getFreelancersSkills(freelancerId.asLong());
    }

    /**
     * Обновление информации о фрилансере. Возвращается фрилансер с обновлёнными данными
     */
    GdFreelancerFull updateFreelancer(ClientId freelancerId, GdUpdateFreelancer input) {
        FreelancerBase freelancerBase = converter.convertFromGd(freelancerId.asLong(), input);
        return updateFreelancer(freelancerId, freelancerBase);
    }

    /**
     * Обновление статуса фрилансера. Возвращается фрилансер с обновлёнными данными
     */
    GdFreelancerFull updateFreelancerStatus(ClientId freelancerId, GdUpdateFlStatusRequest input) {
        Long freelancerIdLong = freelancerId.asLong();

        FreelancerStatus status = converter.convertFromGd(input.getStatus());
        FreelancerBase freelancerBase = new FreelancerBase()
                .withFreelancerId(freelancerIdLong)
                .withStatus(status);

        return updateFreelancer(freelancerId, freelancerBase);
    }

    private GdFreelancerFull updateFreelancer(ClientId freelancerId, FreelancerBase freelancerBase) {
        Result<Long> result = freelancerUpdateService.updateFreelancer(freelancerId, freelancerBase);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result.getValidationResult());
        Freelancer freelancer = freelancerService.getFreelancerWithNewestCard(freelancerId.asLong());
        checkNotNull(freelancer, "Can't find freelancer after successful update");
        String avatarUrl = getAvatarUrl(freelancer);
        return converter.convertToGd(freelancer, GdFreelancerFull::new, avatarUrl);
    }

    /**
     * bulk-загрузка базовой информации о проектах фрилансера
     */
    List<List<GdFreelancerProject>> getFreelancersProjects(List<Long> freelancerIds) {
        if (freelancerIds.isEmpty()) {
            return emptyList();
        }
        List<FreelancerProject> freelancerProjects = freelancerService.getFreelancersProjects(freelancerIds);
        Map<Long, List<GdFreelancerProject>> projectsByFreelancerId = StreamEx.of(freelancerProjects)
                .map(p -> projectConverter.convert(p, GdFreelancerProject::new))
                .groupingBy(GdFreelancerProject::getFreelancerId);
        return StreamEx.of(freelancerIds)
                .map(id -> projectsByFreelancerId.getOrDefault(id, emptyList()))
                .toList();
    }

    public List<GdFreelancerFeedback> getFreelancerFeedbacks(Long operatorUid, Long freelancerId) {
        List<FreelancerFeedback> freelancerFeedbacks =
                freelancerFeedbackService.getFreelancerFeedbackList(freelancerId);
        return freelancerFeedbacks
                .stream()
                .filter(feedback -> feedback.getModerationStatus() != null &&
                        feedback.getModerationStatus().equals(FreelancerUgcModerationStatus.ACCEPTED) ||
                        feedback.getAuthorUid().equals(operatorUid))
                .map(feedback -> converter.convertFreelancerFeedbackToGd(feedback, operatorUid))
                .collect(Collectors.toList());
    }

    public CompletableFuture<GdUserPublicInfo> getFeedbackAuthorInfo(GdFreelancerFeedback feedback) {
        return userDataLoader.get().load(feedback.getAuthorUid())
                .thenApply(GdUserPublicInfo.class::cast);
    }

    private List<GdFreelancerProject> getFreelancersProjectsByProjectIds(List<Long> projectIds) {
        if (projectIds.isEmpty()) {
            return emptyList();
        }
        List<FreelancerProject> freelancerProjects = freelancerService.getFreelancersProjectsByProjectIds(projectIds);
        return StreamEx.of(freelancerProjects)
                .map(p -> projectConverter.convert(p, GdFreelancerProject::new))
                .toList();
    }

    /**
     * bulk-загрузка базовой информации о клиентах
     */
    List<GdFreelancerClient> getFreelancerClients(List<Long> clientLongIds) {
        Set<ClientId> affectedClientIds = StreamEx.of(clientLongIds)
                .map(ClientId::fromLong)
                .toSet();
        Map<Long, Client> clients = StreamEx.of(clientService.massGetClient(affectedClientIds))
                .toMap(Client::getId, Function.identity());

        List<Long> clientsChiefUid = EntryStream.of(clients)
                .mapValues(Client::getChiefUid)
                .values()
                .toList();
        Map<Long, User> users = StreamEx.of(userService.massGetUser(clientsChiefUid))
                .toMap(User::getUid, Function.identity());
        return StreamEx.of(clientLongIds)
                .map(id -> createClient(id, clients, users))
                .toList();
    }

    private GdFreelancerClient createClient(Long id, Map<Long, Client> clients, Map<Long, User> users) {
        Client client = clients.get(id);
        if (client == null) {
            return null;
        }
        User chiefUser = users.get(client.getChiefUid());
        return new GdFreelancerClient()
                .withClientId(client.getClientId())
                .withChiefLogin(chiefUser.getLogin())
                .withFio(chiefUser.getFio())
                .withPhoneNumber(chiefUser.getPhone());
    }

    List<GdClientProject> getClientProjects(Long clientId, @Nullable GdFreelancerProjectFilter gdFilter) {
        FreelancerProjectFilter filter = convertProjectFilterFromGd(gdFilter);
        List<FreelancerProject> freelancerProjects = freelancerService.getClientProjects(clientId, filter);
        return StreamEx.of(freelancerProjects)
                .map(p -> projectConverter.convert(p, GdClientProject::new))
                .toList();
    }

    /**
     * Преобразовывает гридовую модель в ядровую.
     * Если передан freelancerLogin, то он преобразуется в freelancerId, иначе используется сразу freelancerId
     */
    private FreelancerProjectFilter convertProjectFilterFromGd(@Nullable GdFreelancerProjectFilter filter) {
        if (filter == null) {
            return new FreelancerProjectFilter();
        }
        Long freelancerId = null;
        if (filter.getFreelancerLogin() != null) {
            User freelancerUser = userService.getUserByLogin(filter.getFreelancerLogin());
            if (freelancerUser != null) {
                freelancerId = freelancerUser.getClientId().asLong();
            }
        }
        if (freelancerId == null) {
            freelancerId = filter.getFreelancerId();
        }

        return new FreelancerProjectFilter()
                .withFreelancerId(freelancerId)
                .withLimit(filter.getFirst())
                .withIsStarted(filter.getIsStarted())
                .withIsActive(filter.getIsActive());
    }

    /**
     * Для указанных {@code freelancerIds} возвращает списки предложений услуг.
     * Если услуг по заданному ID не найдено, в ответе на соответствующей позиции будет пустой список.
     */
    List<List<GdFreelancerSkill>> getFreelancersSkills(List<Long> freelancerIds) {
        List<FreelancerSkillOffer> skillsOffers = freelancerSkillOffersService.getFreelancerSkillsOffers(freelancerIds);
        Map<Long, List<GdFreelancerSkill>> grouping = StreamEx.of(skillsOffers)
                .mapToEntry(FreelancerSkillOffer::getFreelancerId)
                .invert()
                .mapValues(converter::convertToGd)
                .grouping();
        return freelancerIds.stream()
                .map(id -> grouping.getOrDefault(id, emptyList()))
                .collect(toList());
    }

    /**
     * Не-bulk'овая версия {@link #getFreelancersSkills(List)}
     */
    @Nonnull
    private List<GdFreelancerSkill> getFreelancersSkills(Long freelancerId) {
        List<List<GdFreelancerSkill>> freelancersSkills = getFreelancersSkills(singletonList(freelancerId));
        return freelancersSkills.get(0);
    }

    public Money getSkillPriceMoney(GdFreelancerSkill skill) {
        return Money.valueOf(skill.getPrice(), CURRENCY_CODE);
    }

    GdRequestFlServiceByClientRequestPayload requestFlServiceByClient(User client,
                                                                      GdRequestFlServiceByClientRequest input) {
        ModelItemValidationBuilder<GdRequestFlServiceByClientRequest> b =
                ModelItemValidationBuilder.of(input);
        b.item(GdRequestFlServiceByClientRequest.FREELANCER_ID)
                .check(validId());
        ValidationResult<GdRequestFlServiceByClientRequest, Defect> vr = b.getResult();
        gridValidationService.throwGridValidationExceptionIfHasErrors(vr);

        Long freelancerId = input.getFreelancerId();
        User freelancer = getChiefByClientId(ClientId.fromLong(freelancerId));

        Result<Long> result = freelancerService.requestFreelancerService(client, freelancerId);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        Long projectId = result.getResult();
        List<GdFreelancerProject> freelancerProjects = getFreelancersProjectsByProjectIds(singletonList(projectId));
        checkState(!freelancerProjects.isEmpty(), "Can't find project with id %s", projectId);
        GdFreelancerProject project = freelancerProjects.stream().findFirst().get();

        String freelancerPage = freelancerPagePrefix + freelancer.getLogin();
        sendEmail(client.getEmail(), senderKeys.getOnRequestToClient(), ImmutableMap.of(
                CLIENT_NAME, client.getFio(),
                FREELANCER_NAME, freelancer.getFio(),
                LINK_TO_FREELANCER_CARD, freelancerPage,
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));
        sendEmail(freelancer.getEmail(), senderKeys.getOnRequestToFreelancer(), Map.of(
                FREELANCER_NAME, freelancer.getFio(),
                REQUESTS, freelancerRequests,
                CLIENT_NAME, client.getFio(),
                CLIENT_PHONE_NUMBER, Optional.ofNullable(client.getPhone()).orElse("-"),
                CLIENT_EMAIL, client.getEmail(),
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));

        return new GdRequestFlServiceByClientRequestPayload()
                .withFreelancerProject(project)
                .withClientId(client.getClientId().asLong())
                .withFreelancerId(freelancerId);
    }

    GdAcceptFlServiceByFreelancerRequestPayload acceptFlServiceByFreelancer(User freelancer,
                                                                            GdAcceptFlServiceByFreelancerRequest input) {
        long projectId = input.getProjectId();
        ClientId clientId = ClientId.fromLong(input.getClientId());
        Result<FreelancerProjectIdentity> result = freelancerService.acceptFreelancerProject(
                new FreelancerProject()
                        .withClientId(clientId.asLong())
                        .withFreelancerId(freelancer.getClientId().asLong())
                        .withId(projectId));
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        List<GdFreelancerProject> freelancerProjects = getFreelancersProjectsByProjectIds(singletonList(projectId));
        checkState(!freelancerProjects.isEmpty(), "Can't find project with %s", projectId);
        GdFreelancerProject project = freelancerProjects.stream().findFirst().get();

        User client = getChiefByClientId(clientId);
        String freelancerPage = freelancerPagePrefix + freelancer.getLogin();
        sendEmail(client.getEmail(), senderKeys.getOnAcceptToClient(), ImmutableMap.of(
                FREELANCER_NAME, freelancer.getFio(),
                LINK_TO_FREELANCER_CARD, freelancerPage,
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));
        sendEmail(freelancer.getEmail(), senderKeys.getOnAcceptToFreelancer(), ImmutableMap.of(
                FREELANCER_NAME, freelancer.getFio(),
                CLIENT_LOGIN, client.getLogin(),
                LINK_TO_INTERFACE, freelancerPage,
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));

        return new GdAcceptFlServiceByFreelancerRequestPayload()
                .withFreelancerProject(project)
                .withClientId(clientId.asLong())
                .withFreelancerId(freelancer.getClientId().asLong());
    }

    GdCancelFlServiceByClientRequestPayload cancelFlServiceByClient(ClientId clientId,
                                                                    GdCancelFlServiceByClientRequest input) {
        long projectId = input.getProjectId();
        ClientId freelancerId = ClientId.fromLong(input.getFreelancerId());
        boolean relationExists = freelancerService.isAnyFreelancerRelation(clientId, freelancerId);
        Result<FreelancerProjectIdentity> result = freelancerService
                .cancelFreelancerProjectByClient(new FreelancerProject()
                        .withClientId(clientId.asLong())
                        .withFreelancerId(freelancerId.asLong())
                        .withId(projectId));
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        List<GdFreelancerProject> freelancerProjects = getFreelancersProjectsByProjectIds(singletonList(projectId));
        checkState(!freelancerProjects.isEmpty(), "Can't find project with id %s", projectId);
        GdFreelancerProject project = freelancerProjects.stream().findFirst().get();

        sendCancelEmails(clientId, freelancerId, relationExists);

        return new GdCancelFlServiceByClientRequestPayload()
                .withFreelancerProject(project)
                .withClientId(clientId.asLong())
                .withFreelancerId(freelancerId.asLong());
    }

    GdCancelFlServiceByFreelancerRequestPayload cancelFlServiceByFreelancer(ClientId freelancerId,
                                                                            GdCancelFlServiceByFreelancerRequest input) {
        long projectId = input.getProjectId();
        ClientId clientId = ClientId.fromLong(input.getClientId());
        boolean relationExists = freelancerService.isAnyFreelancerRelation(clientId, freelancerId);
        Result<FreelancerProjectIdentity> result = freelancerService
                .cancelFreelancerProjectByFreelancer(new FreelancerProject()
                        .withClientId(clientId.asLong())
                        .withFreelancerId(freelancerId.asLong())
                        .withId(projectId));
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        List<GdFreelancerProject> freelancerProjects = getFreelancersProjectsByProjectIds(singletonList(projectId));
        checkState(!freelancerProjects.isEmpty(), "Can't find project with id %s", projectId);
        GdFreelancerProject project = freelancerProjects.stream().findFirst().get();

        sendCancelEmails(clientId, freelancerId, relationExists);

        return new GdCancelFlServiceByFreelancerRequestPayload()
                .withFreelancerProject(project)
                .withClientId(clientId.asLong())
                .withFreelancerId(freelancerId.asLong());
    }

    @Nonnull
    private User getChiefByClientId(ClientId freelancerId) {
        long chiefUid = rbacService.getChiefByClientId(freelancerId);
        return checkNotNull(userService.getUser(chiefUid));
    }

    private void sendEmail(String toEmail, String campaignSlug, Map<String, String> args) {
        YandexSenderTemplateParams params = new YandexSenderTemplateParams.Builder()
                .withToEmail(toEmail)
                .withCampaignSlug(campaignSlug)
                .withAsync(true)
                .withArgs(args)
                .build();
        yandexSenderClient.sendTemplate(params);
    }

    private void sendCancelEmails(ClientId clientId, ClientId freelancerId, boolean relationExists) {
        freelancerService.isAnyFreelancerRelation(clientId, freelancerId);
        User freelancer = getChiefByClientId(freelancerId);
        User client = getChiefByClientId(clientId);

        String freelancerName = freelancer.getFio();
        String letterToClient;
        String letterToFreelancer;
        if (relationExists) {
            letterToClient = senderKeys.getOnRescindToClient();
            letterToFreelancer = senderKeys.getOnRescindToFreelancer();
        } else {
            letterToClient = senderKeys.getOnCancelToClient();
            letterToFreelancer = senderKeys.getOnCancelToFreelancer();
        }
        sendEmail(client.getEmail(), letterToClient, ImmutableMap.of(
                FREELANCER_NAME, freelancerName,
                FEEDBACK_FORM, linkToFeedbackForm,
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));
        sendEmail(freelancer.getEmail(), letterToFreelancer, ImmutableMap.of(
                FREELANCER_NAME, freelancerName,
                CLIENT_LOGIN, client.getLogin(),
                FEEDBACK_FORM, linkToFeedbackForm,
                CLIENT_CLIENT_ID, client.getClientId().toString()
        ));
    }

    /**
     * Возвращает {@link CompletableFuture} с {@link GdFreelancerRegion}, который заполнит
     * {@link FreelancerRegionTranslationDataLoader} bulk'ово для всех переданных {@code regionId}
     */
    public CompletableFuture<GdFreelancerRegion> getRegionTranslationFuture(@Nullable Long regionId, GdLanguage lang) {
        if (regionId == null) {
            return CompletableFuture.completedFuture(null);
        }
        GdFreelancerRegionTranslationKey regionTranslationKey =
                new GdFreelancerRegionTranslationKey().withRegionId(regionId).withLang(lang);
        CompletableFuture<GdFreelancerRegionTranslation> translation =
                freelancerRegionTranslationDataLoader.get().load(regionTranslationKey);

        return translation.thenApply(tr -> new GdFreelancerRegion()
                .withRegionId(regionId)
                .withTranslation(tr));
    }

    Map<GdFreelancerRegionTranslationKey, GdFreelancerRegionTranslation> getRegionTranslations(
            List<GdFreelancerRegionTranslationKey> keys) {
        Map<GdLanguage, List<Long>> groupByLang = StreamEx.of(keys)
                .mapToEntry(GdFreelancerRegionTranslationKey::getLang, GdFreelancerRegionTranslationKey::getRegionId)
                .grouping();
        Map<GdFreelancerRegionTranslationKey, GdFreelancerRegionTranslation> result = new HashMap<>(keys.size());
        groupByLang.forEach((lang, ids) -> result.putAll(getRegionTranslations(lang, ids)));
        return result;
    }

    private Map<GdFreelancerRegionTranslationKey, GdFreelancerRegionTranslation> getRegionTranslations(GdLanguage lang,
                                                                                                       List<Long> keys) {
        Map<Long, Long> countryByRegionId = StreamEx.of(keys)
                .distinct()
                .mapToEntry(geoBaseHelper::getCountryId)
                .mapValues(Integer::longValue)
                .toMap();
        Map<Long, String> namesById = StreamEx.ofValues(countryByRegionId)
                .append(keys)
                .distinct()
                .mapToEntry(id -> geoBaseHelper.getRegionName(id, lang.name()))
                .toMap();

        return StreamEx.of(keys)
                .distinct()
                .mapToEntry(id -> {
                    Long countryId = countryByRegionId.get(id);
                    return new GdFreelancerRegionTranslation()
                            .withLang(lang)
                            .withRegionId(id)
                            .withCountryRegionId(countryId)
                            .withRegion(namesById.get(id))
                            .withCountry(namesById.get(countryId));
                })
                .mapKeys(key -> new GdFreelancerRegionTranslationKey().withRegionId(key).withLang(lang))
                .toMap();
    }

    /**
     * Возвращает {@link GdFreelancerPrivateData} с заполненным {@code freelancerId},
     * если у оператора есть права на просмотр приватных данных фрилансера.
     * Иначе возвращается {@code null}.
     */
    @Nullable
    GdFreelancerPrivateData getPrivateData(Long freelancerId, User operator) {
        long freelancerChiefUid = rbacService.getChiefByClientId(ClientId.fromLong(freelancerId));
        if (!rbacService.isOwner(operator.getUid(), freelancerChiefUid)) {
            return null;
        }
        return new GdFreelancerPrivateData().withFreelancerId(freelancerId);
    }

    @Nullable
    public GdFreelancerSkill getMainSkill(Long freelancerId) {
        FreelancerSkillOffer mainSkillsOffer = freelancerSkillOffersService.getFreelancerMainSkillsOffer(freelancerId);
        return mainSkillsOffer != null ? converter.convertToGd(mainSkillsOffer) : null;
    }

    GdAddFeedbackForFreelancerPayload addFeedbackForFreelancer(Long operatorUid, ClientId clientId,
                                                               GdAddFeedbackForFreelancerRequest input) {
        FreelancerFeedback freelancerFeedback = converter.convertFreelancerFeedbackFromGd(operatorUid, input);
        Result<String> result =
                freelancerFeedbackService.addFeedbackForFreelancer(clientId, freelancerFeedback);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        String feedbackId = result.getResult();
        freelancerFeedback.withFeedbackId(feedbackId);
        GdFreelancerFeedback gdFreelancerFeedback =
                converter.convertFreelancerFeedbackToGd(freelancerFeedback, operatorUid);
        return new GdAddFeedbackForFreelancerPayload()
                .withFeedback(gdFreelancerFeedback);
    }

    GdDeleteFreelancerFeedbackPayload deleteFreelancerFeedback(Long operatorUid,
                                                               GdDeleteFreelancerFeedbackRequest input) {
        String feedbackId = input.getFeedbackId();
        Long freelancerId = input.getFreelancerId();
        Result<String> result =
                freelancerFeedbackService.deleteFreelancerFeedback(operatorUid, feedbackId, freelancerId);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        return new GdDeleteFreelancerFeedbackPayload().withFeedbackId(result.getResult());
    }

    GdUpdateFreelancerFeedbackPayload updateFreelancerFeedback(Long operatorUid,
                                                               GdUpdateFreelancerFeedbackRequest input) {
        String feedbackId = input.getFeedbackId();
        Long freelancerId = input.getFreelancerId();
        FreelancerFeedback feedback = new FreelancerFeedback()
                .withFeedbackId(feedbackId)
                .withFreelancerId(freelancerId)
                .withFeedbackText(input.getFeedbackText())
                .withOverallMark(input.getOverallMark())
                .withWillRecommend(input.getWillRecommend());
        Result<String> result = freelancerFeedbackService.updateFreelancerFeedback(operatorUid, feedback);
        gridValidationService.throwGridValidationExceptionIfHasErrors(result, path());

        FreelancerFeedback freelancerFeedback =
                freelancerFeedbackService.getFreelancerFeedback(feedbackId, freelancerId);
        GdFreelancerFeedback updatedFeedback = converter.convertFreelancerFeedbackToGd(freelancerFeedback, operatorUid);
        return new GdUpdateFreelancerFeedbackPayload().withFeedback(updatedFeedback);
    }

    @SuppressWarnings("unused")
    GdAddFeedbackCommentPayload addFeedbackComment(Long operatorUid,
                                                   GdAddFeedbackCommentRequest input) {
        return new GdAddFeedbackCommentPayload();
    }

    @SuppressWarnings("unused")
    GdUpdateFeedbackCommentPayload updateFeedbackComment(Long operatorUid,
                                                         GdUpdateFeedbackCommentRequest input) {
        return new GdUpdateFeedbackCommentPayload();
    }

    @SuppressWarnings("unused")
    GdDeleteFeedbackCommentPayload deleteFeedbackComment(Long operatorUid,
                                                         GdDeleteFeedbackCommentRequest input) {
        return new GdDeleteFeedbackCommentPayload();
    }

    /**
     * Возвращает ОКР-рейтинг фрилансера.
     * Когда понадобится bulk-овое получение рейтингов предполагается перенести их поля в GdFreelancer.
     */
    GdFreelancerAdvQuality getAdvQuality(Long freelancerId) {
        List<Freelancer> freelancers = freelancerService.getFreelancers(singletonList(freelancerId));
        Freelancer freelancer = getOnlyElement(freelancers);
        return converter.convertToGdFreelancerAdvQuality(freelancer);
    }

    String getCertLogin(Long freelancerId) {
        List<Freelancer> freelancers = freelancerService.getFreelancers(singletonList(freelancerId));
        Freelancer freelancer = getOnlyElement(freelancers);
        return freelancer.getCertLogin();
    }
}
