package ru.yandex.direct.grid.processing.service.freelancer;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdUserPublicInfo;
import ru.yandex.direct.grid.processing.model.constants.GdLanguage;
import ru.yandex.direct.grid.processing.model.freelancer.GdClientProject;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerAdvQuality;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerBase;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerCard;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerClient;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFeedback;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFilter;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerFull;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerPrivateData;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerProject;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerProjectFilter;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerRegion;
import ru.yandex.direct.grid.processing.model.freelancer.GdFreelancerSkill;
import ru.yandex.direct.grid.processing.model.freelancer.GdProject;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAcceptFlServiceByFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAcceptFlServiceByFreelancerRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackForFreelancerPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdAddFeedbackForFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByClientRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByClientRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByFreelancerRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdCancelFlServiceByFreelancerRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFreelancerFeedbackPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdDeleteFreelancerFeedbackRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdRequestFlServiceByClientRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdRequestFlServiceByClientRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFeedbackCommentPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFeedbackCommentRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFlStatusRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancer;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerCard;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerCardRequestPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerFeedbackPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerFeedbackRequest;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerPayload;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerSkill;
import ru.yandex.direct.grid.processing.model.freelancer.mutation.GdUpdateFreelancerSkillsPayload;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.grid.processing.util.TextConstants.INPUT;

/**
 * Сервис про фирилансеров
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class FreelancerGraphQlService {

    private final FreelancerDataService freelancerDataService;
    // На самом деле это request-scope бин, доступный через proxy
    private final FreelancerDataLoader freelancerDataLoader;
    private final FreelancerSkillDataLoader freelancerSkillDataLoader;
    private final FreelancerProjectDataLoader freelancerProjectDataLoader;
    private final FreelancerClientDataLoader freelancerClientDataLoader;

    @Autowired
    public FreelancerGraphQlService(
            FreelancerDataService freelancerDataService,
            FreelancerDataLoader freelancerDataLoader,
            FreelancerSkillDataLoader freelancerSkillDataLoader,
            FreelancerProjectDataLoader freelancerProjectDataLoader,
            FreelancerClientDataLoader freelancerClientDataLoader) {
        this.freelancerDataService = freelancerDataService;
        this.freelancerDataLoader = freelancerDataLoader;
        this.freelancerSkillDataLoader = freelancerSkillDataLoader;
        this.freelancerProjectDataLoader = freelancerProjectDataLoader;
        this.freelancerClientDataLoader = freelancerClientDataLoader;
    }

    /**
     * Получение списка фрилансеров с общедоступной информацией
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "freelancersList")
    public List<GdFreelancerBase> allFreelancers(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLArgument(name = "filter") GdFreelancerFilter filter) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.getFreelancers(filter, GdFreelancerBase::new, operatorUid);
    }

    @GraphQLQuery(name = "freelancer")
    public GdFreelancerFull getFreelancer(@GraphQLRootContext GridGraphQLContext context) {
        User client = context.getSubjectUser();
        GdFreelancerFilter filter =
                new GdFreelancerFilter().withFreelancerIds(singletonList(client.getClientId().asLong()));
        return freelancerDataService.getFreelancers(filter, GdFreelancerFull::new, null)
                .stream().findFirst()
                .orElse(null);
    }

    /**
     * Регион фрилансера. Отдельным запросом подтягиваем данные о стране и названии региона фрилансера.
     */
    @GraphQLQuery(
            name = "region",
            description = "Данные о стране и названии региона фрилансера.")
    public CompletableFuture<GdFreelancerRegion> getFreelancerRegion(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancer freelancer,
            @GraphQLNonNull @GraphQLArgument(name = "lang") GdLanguage lang) {
        Long regionId = freelancer.getRegionId();
        return freelancerDataService.getRegionTranslationFuture(regionId, lang);
    }

    /**
     * Приватные данные фрилансера. Отдельными запросами подтягиваем новейшую карточку и проекты
     */
    @GraphQLQuery(
            name = "privateData",
            description = "Приватные данные фрилансера. null, если у оператора нет прав на просмотр")
    public GdFreelancerPrivateData getFreelancerPrivateData(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancer freelancer) {
        User operator = context.getOperator();
        return freelancerDataService.getPrivateData(freelancer.getFreelancerId(), operator);
    }

    /**
     * Последняя версия карточки вне зависимости от статуса.
     * <p>
     * Метод не-bulk'овый, поскольку обычный клиент видит только свою карточку,
     * а в интерфейсе для суперпользователей пока нет необходимости получать это значение для нескольких специалистов.
     */
    @GraphQLQuery(
            name = "newestCard",
            description = "Последняя версия карточки вне зависимости от статуса")
    public GdFreelancerCard getFreelancerNewestCard(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancerPrivateData privateData) {
        Long freelancerId = privateData.getFreelancerId();
        return freelancerDataService.getNewestFreelancerCards(singletonList(freelancerId)).get(freelancerId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "projects")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerProject>> getFreelancersProjects(
            @GraphQLContext GdFreelancerPrivateData freelancer) {
        return freelancerProjectDataLoader.get().load(freelancer.getFreelancerId());
    }

    @GraphQLQuery(
            name = "advQuality",
            description = "ОКР-рейтинг специалиста.")
    public GdFreelancerAdvQuality getFreelancerAdvQuality(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancerPrivateData privateData) {
        Long freelancerId = privateData.getFreelancerId();
        return freelancerDataService.getAdvQuality(freelancerId);
    }

    @GraphQLQuery(
            name = "certLogin",
            description = "Логин для сертификатов")
    public String getFreelancerCertLogin(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancerPrivateData privateData) {
        Long freelancerId = privateData.getFreelancerId();
        return freelancerDataService.getCertLogin(freelancerId);
    }

    @GraphQLQuery(name = "mainSkill")
    public GdFreelancerSkill getFreelancerMainSkill(
            @GraphQLContext GdFreelancer freelancer) {
        return freelancerDataService.getMainSkill(freelancer.getFreelancerId());
    }

    @GraphQLQuery(name = "feedbacks")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerFeedback>> getFreelancerFeedbacks(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancer freelancer) {
        Long operatorUid = context.getOperator().getUid();
        return CompletableFuture
                .completedFuture(
                        freelancerDataService.getFreelancerFeedbacks(operatorUid, freelancer.getFreelancerId()));
    }

    @GraphQLQuery(name = "authorInfo")
    public CompletableFuture<GdUserPublicInfo> getFreelancerFeedbacksAuthorInfo(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancerFeedback feedback) {
        return freelancerDataService.getFeedbackAuthorInfo(feedback);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "projects")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerProject>> getFreelancersProjects(
            @GraphQLContext GdFreelancerFull freelancer) {
        return freelancerProjectDataLoader.get().load(freelancer.getFreelancerId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "client")
    public CompletableFuture<GdFreelancerClient> getFreelancerClients(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdFreelancerProject project) {
        return freelancerClientDataLoader.get().load(project.getClientId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "freelancerProjects")
    public List<@GraphQLNonNull GdClientProject> getClientFreelancerProjects(
            @GraphQLContext GdClient client,
            @Nullable @GraphQLArgument(name = "filter") GdFreelancerProjectFilter filter) {
        return freelancerDataService.getClientProjects(client.getInfo().getId(), filter);
    }

    @GraphQLQuery(name = "freelancer")
    public CompletableFuture<GdFreelancer> getClientFreelancer(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdProject project) {
        // todo maxlog: при желании тут можно превратить FreelancerFull в FreelancerBase
        return freelancerDataLoader.get().load(project.getFreelancerId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "skills")
    public CompletableFuture<List<@GraphQLNonNull GdFreelancerSkill>> getFreelancersSkills(
            @GraphQLContext GdFreelancer freelancer) {
        return freelancerSkillDataLoader.get().load(freelancer.getFreelancerId());
    }

    /**
     * Добавляет {@link GdFreelancerSkill} свойство {@code priceMoney} типа {@link Money}
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "priceMoney")
    public Money getFreelancerSkillPriceMoney(@GraphQLContext GdFreelancerSkill skill) {
        return freelancerDataService.getSkillPriceMoney(skill);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "requestFlServiceByClient")
    @PreAuthorizeWrite
    public GdRequestFlServiceByClientRequestPayload requestFlServiceByClient(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdRequestFlServiceByClientRequest input
    ) {
        return freelancerDataService.requestFlServiceByClient(context.getSubjectUser(), input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "acceptFlServiceByFreelancer")
    @PreAuthorizeWrite
    public GdAcceptFlServiceByFreelancerRequestPayload acceptFlServiceByFreelancer(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdAcceptFlServiceByFreelancerRequest input
    ) {
        return freelancerDataService.acceptFlServiceByFreelancer(context.getSubjectUser(), input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "cancelFlServiceByClient")
    @PreAuthorizeWrite
    public GdCancelFlServiceByClientRequestPayload cancelFlServiceByClient(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdCancelFlServiceByClientRequest input
    ) {
        return freelancerDataService.cancelFlServiceByClient(context.getSubjectUser().getClientId(), input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "cancelFlServiceByFreelancer")
    @PreAuthorizeWrite
    public GdCancelFlServiceByFreelancerRequestPayload cancelFlServiceByFreelancer(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdCancelFlServiceByFreelancerRequest input
    ) {
        return freelancerDataService.cancelFlServiceByFreelancer(context.getSubjectUser().getClientId(), input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFreelancerCard")
    @PreAuthorizeWrite
    public GdUpdateFreelancerCardRequestPayload updateFreelancerCard(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFreelancerCard input
    ) {
        ClientId freelancerId = context.getSubjectUser().getClientId();
        GdFreelancer freelancer = freelancerDataService.updateFreelancerCard(freelancerId, input);
        return new GdUpdateFreelancerCardRequestPayload().withFreelancer(freelancer);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFreelancerSkills")
    @PreAuthorizeWrite
    public GdUpdateFreelancerSkillsPayload updateFreelancerSkills(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFreelancerSkill input
    ) {
        ClientId freelancerId = context.getSubjectUser().getClientId();
        List<GdFreelancerSkill> freelancerSkills = freelancerDataService.updateFreelancerSkills(freelancerId, input);
        return new GdUpdateFreelancerSkillsPayload().withSkills(freelancerSkills);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFreelancer")
    @PreAuthorizeWrite
    public GdUpdateFreelancerPayload updateFreelancer(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFreelancer input
    ) {
        ClientId freelancerId = context.getSubjectUser().getClientId();
        GdFreelancerFull freelancer = freelancerDataService.updateFreelancer(freelancerId, input);
        return new GdUpdateFreelancerPayload().withFreelancer(freelancer);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFreelancerStatus")
    @PreAuthorizeWrite
    public GdUpdateFreelancerCardRequestPayload updateFreelancerStatus(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFlStatusRequest input
    ) {
        ClientId freelancerId = context.getSubjectUser().getClientId();
        GdFreelancer freelancer = freelancerDataService.updateFreelancerStatus(freelancerId, input);
        return new GdUpdateFreelancerCardRequestPayload().withFreelancer(freelancer);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "addFeedbackForFreelancer")
    @PreAuthorizeWrite
    public GdAddFeedbackForFreelancerPayload addFeedbackForFreelancer(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdAddFeedbackForFreelancerRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = context.getSubjectUser().getClientId();
        return freelancerDataService.addFeedbackForFreelancer(operatorUid, clientId, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "deleteFreelancerFeedback")
    @PreAuthorizeWrite
    public GdDeleteFreelancerFeedbackPayload deleteFreelancerFeedback(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdDeleteFreelancerFeedbackRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.deleteFreelancerFeedback(operatorUid, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFreelancerFeedback")
    @PreAuthorizeWrite
    public GdUpdateFreelancerFeedbackPayload updateFreelancerFeedback(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFreelancerFeedbackRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.updateFreelancerFeedback(operatorUid, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "addFeedbackComment")
    @PreAuthorizeWrite
    public GdAddFeedbackCommentPayload addFeedbackComment(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdAddFeedbackCommentRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.addFeedbackComment(operatorUid, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "updateFeedbackComment")
    @PreAuthorizeWrite
    public GdUpdateFeedbackCommentPayload updateFeedbackComment(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdUpdateFeedbackCommentRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.updateFeedbackComment(operatorUid, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = "deleteFeedbackComment")
    @PreAuthorizeWrite
    public GdDeleteFeedbackCommentPayload deleteFeedbackComment(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = INPUT) GdDeleteFeedbackCommentRequest input
    ) {
        Long operatorUid = context.getOperator().getUid();
        return freelancerDataService.deleteFeedbackComment(operatorUid, input);
    }
}
