package ru.yandex.direct.grid.processing.service.goal;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.EnumUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.appmetrika.AppMetrikaClient;
import ru.yandex.direct.appmetrika.model.AppMetrikaClientException;
import ru.yandex.direct.appmetrika.model.Platform;
import ru.yandex.direct.appmetrika.model.response.TypedEventInfo;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventSubtype;
import ru.yandex.direct.core.entity.mobileapp.model.AppmetrikaEventType;
import ru.yandex.direct.core.entity.mobileapp.model.ExternalTrackerEventName;
import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService;
import ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser;
import ru.yandex.direct.core.entity.mobilegoals.model.AppmetrikaInternalEvent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.processing.model.goal.GdAppmetrikaApplication;
import ru.yandex.direct.grid.processing.model.goal.GdAppmetrikaApplicationsPayload;
import ru.yandex.direct.grid.processing.model.goal.GdErrorCause;
import ru.yandex.direct.grid.processing.model.goal.GdMobileEvent;
import ru.yandex.direct.grid.processing.model.goal.GdMobileEventsPayload;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.lang.Math.max;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.containsIgnoreCase;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static ru.yandex.direct.core.entity.mobilecontent.model.OsType.ANDROID;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class AppmetrikaService {
    private static final Logger logger = LoggerFactory.getLogger(AppmetrikaService.class);

    private static final Pattern ENTITY_NOT_FOUND_PATTERN = Pattern.compile("Entity not found");
    private static final Pattern OBJECT_NOT_FOUND_PATTERN = Pattern.compile("Object with id = \\d+ not found");

    private final AppMetrikaClient appMetrikaClient;
    private final MobileContentService mobileContentService;

    @Autowired
    public AppmetrikaService(AppMetrikaClient appMetrikaClient, MobileContentService mobileContentService) {
        this.appMetrikaClient = appMetrikaClient;
        this.mobileContentService = mobileContentService;
    }

    public List<GdAppmetrikaApplication> getApplications(long uid, ClientId clientId, @Nullable String url,
                                                         @Nullable String mask,
                                                         @Nullable Integer limit, @Nullable Integer offset) {
        String bundleId = null;
        Platform platform = null;

        if (isNotEmpty(url)) {
            MobileAppStoreUrl parsedUrl = MobileAppStoreUrlParser.parse(url).orElse(null);
            if (parsedUrl == null) {
                return emptyList();
            }

            MobileContent mobileContent = mobileContentService.getMobileContent(clientId, url, parsedUrl, true, true)
                    .orElse(null);
            if (mobileContent == null) {
                return emptyList();
            }

            OsType osType = mobileContent.getOsType();
            bundleId = osType == ANDROID ? mobileContent.getStoreContentId() : mobileContent.getBundleId();
            platform = Platform.valueOf(osType.name().toLowerCase());
        }

        return mapList(appMetrikaClient.getApplications(uid, bundleId, platform, mask, limit, offset),
                a -> new GdAppmetrikaApplication().withId(a.getId()).withName(a.getName()));
    }

    public GdAppmetrikaApplicationsPayload getApplicationsSafe(long uid, ClientId clientId, @Nullable String url,
                                                               @Nullable String mask,
                                                               @Nullable Integer limit, @Nullable Integer offset) {
        String bundleId = null;
        Platform platform = null;

        if (isNotEmpty(url)) {
            MobileAppStoreUrl parsedUrl = MobileAppStoreUrlParser.parse(url).orElse(null);
            if (parsedUrl == null) {
                return makeEmptyPayload(true);
            }

            MobileContent mobileContent = mobileContentService.getMobileContent(clientId, url, parsedUrl, true, true)
                    .orElse(null);
            if (mobileContent == null) {
                return makeEmptyPayload(true);
            }

            OsType osType = mobileContent.getOsType();
            bundleId = osType == ANDROID ? mobileContent.getStoreContentId() : mobileContent.getBundleId();
            platform = Platform.valueOf(osType.name().toLowerCase());
        }

        try {
            return makePayload(
                    mapList(appMetrikaClient.getApplications(uid, bundleId, platform, mask, limit, offset),
                            a -> new GdAppmetrikaApplication().withId(a.getId()).withName(a.getName()))
            );
        } catch (AppMetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika applications for clientId: " + clientId, e);
            return makeEmptyPayload(false);
        }
    }

    public List<GdMobileEvent> getEvents(long uid, @Nullable Long metrikaAppId, @Nullable String mask,
                                         @Nullable Integer limit, @Nullable Integer offset) {
        limit = nvl(limit, Integer.MAX_VALUE);
        offset = nvl(offset, 0);
        List<GdMobileEvent> events = new ArrayList<>();
        if (metrikaAppId == null) {
            // Возвращаем список событий внешних трекеров
            List<GdMobileEvent> externalTrackerEvents = Stream.of(ExternalTrackerEventName.values())
                    .map(e -> new GdMobileEvent().withName(e.name()).withIsInternal(true))
                    .filter(e -> isEmpty(mask) || containsIgnoreCase(e.getName(), mask))
                    .skip(offset)
                    .limit(limit)
                    .collect(toList());
            events.addAll(externalTrackerEvents);
        } else {
            // Объединяем системные и клиентские события аппметрики с лимитом и оффсетом
            List<GdMobileEvent> typedEvents =
                    filterList(typedEventsToMobileEvents(appMetrikaClient.getTypedEvents(uid, metrikaAppId)),
                            e -> isEmpty(mask) || containsIgnoreCase(e.getName(), mask));
            events.addAll(typedEvents.stream()
                    .skip(offset)
                    .limit(limit)
                    .collect(toList()));

            Integer clientEventsLimit = null;
            Integer clientEventsOffset = null;
            if (limit < Integer.MAX_VALUE) {
                int upperClientEventsBound = limit + offset - typedEvents.size();
                clientEventsOffset = max(upperClientEventsBound - limit, 0);
                clientEventsLimit = upperClientEventsBound - clientEventsOffset;
            }

            if (clientEventsLimit == null || clientEventsLimit > 0) {
                events.addAll(mapList(appMetrikaClient.getClientEvents(uid, metrikaAppId, mask, clientEventsLimit,
                                clientEventsOffset),
                        e -> new GdMobileEvent().withName(e).withIsInternal(false)));
            }
        }

        return events;
    }

    public GdMobileEventsPayload getEventsSafe(long uid, ClientId clientId, @Nullable Long metrikaAppId,
                                               @Nullable String mask, @Nullable Integer limit,
                                               @Nullable Integer offset) {
        limit = nvl(limit, Integer.MAX_VALUE);
        offset = nvl(offset, 0);
        List<GdMobileEvent> events = new ArrayList<>();
        if (metrikaAppId == null) {
            // Возвращаем список событий внешних трекеров
            List<GdMobileEvent> externalTrackerEvents = Stream.of(ExternalTrackerEventName.values())
                    .map(e -> new GdMobileEvent().withName(e.name()).withIsInternal(true))
                    .filter(e -> isEmpty(mask) || containsIgnoreCase(e.getName(), mask))
                    .skip(offset)
                    .limit(limit)
                    .collect(toList());
            events.addAll(externalTrackerEvents);
            return new GdMobileEventsPayload()
                    .withIsAppmetrikaAvailable(true)
                    .withMobileAppEvents(events);
        } else {
            try {
                // Объединяем системные и клиентские события аппметрики с лимитом и оффсетом
                List<GdMobileEvent> typedEvents =
                        filterList(typedEventsToMobileEvents(appMetrikaClient.getTypedEvents(uid, metrikaAppId)),
                                e -> isEmpty(mask) || containsIgnoreCase(e.getName(), mask));
                events.addAll(typedEvents.stream()
                        .skip(offset)
                        .limit(limit)
                        .collect(toList()));

                Integer clientEventsLimit = null;
                Integer clientEventsOffset = null;
                if (limit < Integer.MAX_VALUE) {
                    int upperClientEventsBound = limit + offset - typedEvents.size();
                    clientEventsOffset = max(upperClientEventsBound - limit, 0);
                    clientEventsLimit = upperClientEventsBound - clientEventsOffset;
                }

                if (clientEventsLimit == null || clientEventsLimit > 0) {
                    events.addAll(mapList(appMetrikaClient.getClientEvents(uid, metrikaAppId, mask, clientEventsLimit,
                                    clientEventsOffset),
                            e -> new GdMobileEvent().withName(e).withIsInternal(false)));
                }

                return new GdMobileEventsPayload()
                        .withIsAppmetrikaAvailable(true)
                        .withMobileAppEvents(events);
            } catch (AppMetrikaClientException | InterruptedRuntimeException e) {
                logger.warn("Got an exception when querying for appmetrika mobile app events for client: "
                        + clientId, e);

                GdErrorCause errorCause = GdErrorCause.APPMETRIKA_NOT_AVAILABLE;
                String errorMessage = e.getMessage();
                if (errorMessage != null) {
                    if (ENTITY_NOT_FOUND_PATTERN.matcher(errorMessage).find()) {
                        errorCause = GdErrorCause.ENTITY_NOT_FOUND;
                    } else if (OBJECT_NOT_FOUND_PATTERN.matcher(errorMessage).find()) {
                        errorCause = GdErrorCause.OBJECT_NOT_FOUND;
                    }
                }

                return new GdMobileEventsPayload()
                        .withIsAppmetrikaAvailable(false)
                        .withErrorCause(errorCause)
                        .withMobileAppEvents(List.of());
            }
        }
    }

    private List<GdMobileEvent> typedEventsToMobileEvents(List<TypedEventInfo> typedEvents) {
        return typedEvents.stream()
                .peek(e -> e.setType(e.getType().replace("EVENT_", "")))
                .filter(e -> EnumUtils.isValidEnum(AppmetrikaEventType.class, e.getType())
                        && EnumUtils.isValidEnum(AppmetrikaEventSubtype.class, e.getSubtype()))
                .map(e -> AppmetrikaInternalEvent.fromTypeAndSubtype(AppmetrikaEventType.valueOf(e.getType()),
                        AppmetrikaEventSubtype.valueOf(e.getSubtype()), false))
                .filter(Objects::nonNull)
                .map(e -> new GdMobileEvent().withName(e.name()).withIsInternal(true))
                .collect(toList());
    }

    private static GdAppmetrikaApplicationsPayload makePayload(List<GdAppmetrikaApplication> applications) {
        return new GdAppmetrikaApplicationsPayload()
                .withApplications(applications)
                .withIsAppmetrikaAvailable(true);
    }

    private static GdAppmetrikaApplicationsPayload makeEmptyPayload(boolean isAppmetrikaAvailable) {
        return new GdAppmetrikaApplicationsPayload()
                .withApplications(emptyList())
                .withIsAppmetrikaAvailable(isAppmetrikaAvailable);
    }
}
