package ru.yandex.direct.grid.processing.service.goal

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.common.TranslationService
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestItem
import ru.yandex.direct.core.entity.retargeting.model.CryptaGoalsSuggestSegment
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.GoalType
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaGoalsSuggestItem
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaGoalsSuggestSegment
import ru.yandex.direct.grid.processing.model.crypta.GdCryptaGoalsSuggestType
import ru.yandex.direct.grid.processing.model.goal.GdGoal
import ru.yandex.direct.grid.processing.model.goal.GdGoalTruncated
import ru.yandex.direct.grid.processing.model.goal.GdGoalType
import ru.yandex.direct.utils.FunctionalUtils.mapList

@Component
class CryptaGoalsConverter(private val translationService: TranslationService) {

    companion object {
        private const val BUNDLE_NAME = "ru.yandex.direct.core.entity.crypta.CryptaGoalTranslations"
        private val logger = LoggerFactory.getLogger(CryptaGoalsConverter::class.java)

        @JvmStatic
        fun toGdCryptaGoalsSuggestItem(item: CryptaGoalsSuggestItem): GdCryptaGoalsSuggestItem =
            GdCryptaGoalsSuggestItem()
                .withText(item.text)
                .withDescription(item.description)
                .withType(GdCryptaGoalsSuggestType.fromSource(item.type))
                .withSegments(mapList(item.segments, this::toGdCryptaGoalsSuggestSegment))

        @JvmStatic
        fun toCryptaGoalsSuggestItem(item: GdCryptaGoalsSuggestItem): CryptaGoalsSuggestItem = CryptaGoalsSuggestItem()
            .withText(item.text)
            .withType(GdCryptaGoalsSuggestType.toSource(item.type))
            .withSegments(mapList(item.segments, this::toCryptaGoalsSuggestSegment))

        private fun toGdCryptaGoalsSuggestSegment(segment: CryptaGoalsSuggestSegment): GdCryptaGoalsSuggestSegment =
            GdCryptaGoalsSuggestSegment()
                .withSegmentId(segment.segmentId)
                .withKeywordId(segment.keywordId)

        private fun toCryptaGoalsSuggestSegment(segment: GdCryptaGoalsSuggestSegment): CryptaGoalsSuggestSegment =
            CryptaGoalsSuggestSegment()
                .withSegmentId(segment.segmentId)
                .withKeywordId(segment.keywordId)

        private fun toTranslatedGdGoalTruncated(
            goal: Goal,
            name: String,
            description: String? = null,
            audienceTypeName: String? = null,
        ): GdGoalTruncated =
            GdGoal()
                .withId(goal.id)
                .withType(GdGoalType.fromSource(goal.type))
                .withName(name)
                .withDescription(description)
                .withAudienceTypeName(audienceTypeName)
    }

    fun convertToTranslatedGdGoal(goals: Collection<Goal>) = goals.mapNotNull { translateGoal(it) }

    private fun translateGoal(goal: Goal) = when (goal.type) {
        GoalType.HOST -> translateHost(goal)
        else -> translateInterest(goal)
    }

    private fun translateHost(goal: Goal) = toTranslatedGdGoalTruncated(goal, goal.name)

    private fun translateInterest(goal: Goal): GdGoalTruncated? {
        val name = getSafeTranslation(goal.tankerNameKey)
        val description = getSafeTranslation(goal.tankerDescriptionKey)
        val audienceTypeName = getSafeTranslation(goal.tankerAudienceTypeKey)

        if (name == null || description == null) {
            logger.warn(
                "Goal with id {} will be filtered. Unable to find translation using the following keys: [{}, {}]",
                goal.id,
                goal.tankerNameKey,
                goal.tankerDescriptionKey
            )
            return null
        }

        return toTranslatedGdGoalTruncated(goal, name, description, audienceTypeName)
    }

    private fun getSafeTranslation(key: String?): String? = when {
        key.isNullOrBlank() -> null
        else -> {
            try {
                val translate = translationService.translate(BUNDLE_NAME, key)
                /*
                В translationService всегда инжектится StubTranslator, который берет дефолтное значение из аннотаций.
                Сделанно это было для того, чтобы у нас всегда были какие-то переводы.
                Но в случае с Криптой он возвращает сам ключ, поэтому добавляем такую проверку
                 */
                if (translate.equals(key)) null else translate
            } catch (e: Exception) {
                logger.warn("Unable to find translation", e)
                null
            }
        }
    }
}
