package ru.yandex.direct.grid.processing.service.goal;

import java.time.Duration;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.lettuce.LettuceExecuteException;
import ru.yandex.direct.grid.core.entity.goal.model.GoalsConversionsCacheRecord;
import ru.yandex.direct.utils.HashingUtils;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;

@Service
@ParametersAreNonnullByDefault
public class GoalConversionsCacheService {
    private static final Logger logger = LoggerFactory.getLogger(GoalConversionsCacheService.class);
    static final String CACHE_KEY_PATTERN = "goals-combined-conversions-cache-v2-%s";

    private final LettuceConnectionProvider lettuceCache;

    public static final Duration EXPIRATION_TIME = Duration.ofHours(4);

    public GoalConversionsCacheService(@Qualifier(LETTUCE) LettuceConnectionProvider lettuceCache) {
        this.lettuceCache = lettuceCache;
    }

    private static String generateKey(Long clientId, Set<Long> counterIds, boolean withUnavailableGoals) {
        Set<Long> keySet = new HashSet<>();
        keySet.add(clientId);
        keySet.addAll(counterIds);
        keySet.add(withUnavailableGoals ? 1L : 0L);

        String md5Hash = HashingUtils.getMd5HashUtf8AsHexString(StreamEx.of(keySet).sorted().joining("_"));
        return String.format(CACHE_KEY_PATTERN, md5Hash);
    }

    public GoalsConversionsCacheRecord getFromCache(Long clientId, Set<Long> counterIds, boolean withUnavailableGoals) {
        String key = generateKey(clientId, counterIds, withUnavailableGoals);
        String jsonValue = redisGet(key);

        return Optional.ofNullable(jsonValue)
                .map(v -> JsonUtils.fromJson(v, GoalsConversionsCacheRecord.class))
                .orElse(null);
    }

    public void saveToCache(GoalsConversionsCacheRecord cacheRecord, boolean withUnavailableGoals) {
        String key = generateKey(cacheRecord.getClientId(), cacheRecord.getCounterIds(), withUnavailableGoals);
        String jsonValue = JsonUtils.toJson(cacheRecord);
        redisSetex(key, jsonValue);
    }

    @Nullable
    private String redisGet(String key) {
        try {
            return lettuceCache.call("redis:get", cmd -> cmd.get(key));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisGet", e);
            return null;
        }
    }

    private void redisSetex(String key, String value) {
        try {
            lettuceCache.call("redis:setex", cmd -> cmd.setex(key, EXPIRATION_TIME.getSeconds(), value));
        } catch (LettuceExecuteException e) {
            logger.warn("error in redisSetex", e);
        }
    }

}
