package ru.yandex.direct.grid.processing.service.goal;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.metrika.container.CreateCounterGoalContainer;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.retargeting.model.ConversionLevel;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.model.PresetsByCounter;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.grid.core.entity.goal.model.GoalConversionVisit;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiGoalCostPerAction;
import ru.yandex.direct.grid.processing.model.goal.ConversionGrade;
import ru.yandex.direct.grid.processing.model.goal.GdCounterWithGoals;
import ru.yandex.direct.grid.processing.model.goal.GdCpaSource;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.goal.GdGoalConversionVisitsCount;
import ru.yandex.direct.grid.processing.model.goal.GdGoalFilter;
import ru.yandex.direct.grid.processing.model.goal.GdGoalTruncated;
import ru.yandex.direct.grid.processing.model.goal.GdGoalType;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsContext;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsRecommendedCostPerActionByCampaignId;
import ru.yandex.direct.grid.processing.model.goal.GdLalSegment;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaCounterGoalType;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaSegment;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaSegmentPreset;
import ru.yandex.direct.grid.processing.model.goal.GdRecommendedGoalCostPerAction;
import ru.yandex.direct.grid.processing.model.goal.GoalSubtype;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoal;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalType;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValue;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdPresetsByCounter;
import ru.yandex.direct.grid.processing.model.retargeting.GdGoalMinimal;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.metrika.client.model.response.CreateCounterGoal;
import ru.yandex.direct.metrika.client.model.response.GoalConditionType;
import ru.yandex.direct.metrika.client.model.response.GoalConversionInfo;
import ru.yandex.direct.metrika.client.model.response.Segment;
import ru.yandex.direct.utils.JsonUtils;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MIN_SUFFICIENT_GOAL_CONVERSION_COUNT;
import static ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService.composeGoalName;
import static ru.yandex.direct.core.entity.retargeting.service.common.GoalUtilsService.composeGoalSimpleName;
import static ru.yandex.direct.grid.processing.model.goal.GdMetrikaCounterGoalType.URL;
import static ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValueType.SPECIFIED_VALUE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.NumberUtils.greaterThanZero;

@ParametersAreNonnullByDefault
public class GoalDataConverter {
    static GdGoalsContext toGdGoalsContext(List<GdGoal> rowset) {
        return new GdGoalsContext()
                .withRowset(rowset)
                .withTotalCount(rowset.size());
    }

    static List<GdGoal> toFilteredRowset(List<Goal> goals, GdGoalFilter filter) {
        return StreamEx.of(goals)
                .filter(GoalBase::getAllowToUse)
                .map(goal -> toGdGoal(goal, null))
                .filter(goal -> filter.getGoalTypeIn() == null || filter.getGoalTypeIn().contains(goal.getType()))
                .toList();
    }

    static List<GdGoal> toGdGoals(List<Goal> goals) {
        Map<Long, Goal> goalsMap = listToMap(goals, GoalBase::getId);
        return mapList(goals, goal -> toGdGoal(goal, goalsMap.get(goal.getParentId())));
    }

    public static Set<GdGoal> toGdGoals(Set<Goal> goals) {
        Map<Long, Goal> goalsMap = listToMap(goals, GoalBase::getId);
        return listToSet(goals, goal -> toGdGoal(goal, goalsMap.get(goal.getParentId())));
    }

    static List<GdCounterWithGoals> toGdCounterWithGoals(Map<Long, Set<Goal>> availableGoalsForCounterId) {
        return EntryStream.of(availableGoalsForCounterId)
                .mapValues(GoalDataConverter::toGdGoals)
                .map(longSetEntry -> new GdCounterWithGoals()
                        .withCounterId(longSetEntry.getKey())
                        .withGoals(longSetEntry.getValue())
                ).toList();
    }

    public static GdGoal toGdGoal(Goal goal, @Nullable Goal parentGoal) {
        // DIRECT-124421: quickfix: если не знаем тип, отдаём на фронт 'URL'
        GdMetrikaCounterGoalType metrikaGoalType =
                nvl(GdMetrikaCounterGoalType.fromSource(goal.getMetrikaCounterGoalType()), URL);
        //пока конвертируем только цели метрики
        return new GdGoal()
                .withId(goal.getId())
                .withParentId(goal.getParentId())
                //возможно не все из этих целей нужно отдавать и светить наружу
                .withType(convertToGdGoalType(goal.getType()))
                .withSubtype(GoalSubtype.fromTypedValue(goal.getSubtype()))
                .withMetrikaGoalType(metrikaGoalType)
                .withUploadingSourceId(goal.getUploadingSourceId())
                //для подцели добавляем название родительской цели в начало
                .withName(composeGoalName(goal, parentGoal))
                .withSimpleName(composeGoalSimpleName(goal))
                .withDomain(isEmpty(goal.getDomain()) ? null : goal.getDomain())
                .withCounterName(goal.getCounterName())
                .withCounterId(goal.getCounterId())
                .withMobileAppId(goal.getMobileAppId())
                .withMobileAppName(goal.getMobileAppName())
                .withTime(goal.getTime())
                .withPercent(goal.getPercent())
                .withHasPrice(calculateHasPrice(goal))
                .withSectionId(goal.getSectionId())
                .withSectionName(goal.getSectionName())
                .withCounterIds(goal.getCounterIds())
                .withIsMobileGoal(goal.getIsMobileGoal())
                .withUnionWithId(goal.getUnionWithId())
                .withIsPerfectGoal(goal.getConversionLevel() == ConversionLevel.PERFECT)
                .withConversionGrade(getConversionGrade(goal.getConversionVisitsCount()))
                .withConversionVisitsCount(goal.getConversionVisitsCount());
    }

    private static ConversionGrade getConversionGrade(@Nullable Long conversionVisitsCount) {
        return conversionVisitsCount == null || conversionVisitsCount < MIN_SUFFICIENT_GOAL_CONVERSION_COUNT
                ? ConversionGrade.LOW_CONVERSION
                : ConversionGrade.ENOUGH_CONVERSIONS;
    }

    static Boolean calculateHasPrice(Goal goal) {
        //т.к. сейчас метрика ничего не знает про ecom, считаем что у таких целей всегда есть доход
        if (goal.getType() == GoalType.ECOMMERCE) {
            return true;
        }

        //ценность конверсии определяется наличием дохода по цели или наличием установленной пользователем цены для цели
        return nvl(goal.getHasRevenue(), false) || greaterThanZero(goal.getDefaultPrice());
    }

    static GdLalSegment toGdLalSegment(Goal goal) {
        return new GdLalSegment()
                .withId(goal.getId())
                .withName(goal.getName())
                .withParentId(goal.getParentId());
    }

    public static GdGoalTruncated toGdGoalTruncated(Goal internalGoal) {
        return new GdGoal()
                .withId(internalGoal.getId())
                .withType(convertToGdGoalType(internalGoal.getType()))
                .withTime(internalGoal.getTime())
                .withUnionWithId(internalGoal.getUnionWithId())
                .withName(nvl(internalGoal.getName(), ""))
                .withLalParentType(convertToGdGoalType(internalGoal.getLalParentType()));
    }

    public static Goal toCoreGoal(GdGoalMinimal gridGoal) {
        Goal goal = new Goal();
        goal.setId(gridGoal.getId());
        goal.setTime(gridGoal.getTime());
        goal.setUnionWithId(gridGoal.getUnionWithId());
        return goal;
    }

    private static GdGoalType convertToGdGoalType(GoalType goalType) {
        return GdGoalType.fromSource(goalType);
    }

    static GdMetrikaSegment toGdSegment(Segment segment) {
        return new GdMetrikaSegment()
                .withId(segment.getId())
                .withName(segment.getName())
                .withCounterId(segment.getCounterId());
    }

    static PresetsByCounter toCorePresetsByCounter(GdPresetsByCounter presetsByCounter) {
        return new PresetsByCounter()
                .withCounterId(presetsByCounter.getCounterId())
                .withPresetIds(presetsByCounter.getPresetIds());
    }

    static GdMetrikaSegmentPreset toGdMetrikaSegmentPreset(MetrikaSegmentPreset corePreset) {
        return new GdMetrikaSegmentPreset()
                .withCounterId(corePreset.getCounterId())
                .withPresetId(corePreset.getPresetId())
                .withName(corePreset.getName())
                .withDomain(corePreset.getDomain());
    }

    public static Set<GdGoalConversionVisitsCount> toGoalsConversionVisitsCount(Map<Long, Long> conversionCountByGoalId) {
        return EntryStream.of(conversionCountByGoalId)
                .mapKeyValue((goalId, conversionCount) -> new GdGoalConversionVisitsCount()
                        .withId(goalId)
                        .withConversionVisitsCount(conversionCount))
                .toSet();
    }

    /**
     * Получить итоговый набор данных о конверсиях для целей на основе ответа из Метрики и статистики из БК.
     * Если цель является не составной, то в качестве значения будет:
     * <ul>
     *      <li>данные из метрики</li>
     *      <li>если метрика не вернула значение (еком цель), то смотрим в статистику из БК</li>
     *      <li>если нет статистики из БК, то устанавливаем значение в 0</li>
     * </ul>
     * Если цель составная, то значение для конверсии будет браться из БК.
     * ВАЖНО! конверсии для составной цели может не быть из-за неконсистентности статистики (значение для родительской
     * цели не соответствует минимальному значению ее шагов), поэтому такие данные мы отфильтровываем.
     *
     * @param metrikaGoals - общий список целей из метрики
     * @param goalIdsWithoutCombinedGoals - список ids целей, без составных
     * @param metrikaConversionCountByGoalId - данные по конверсиям из метрики, содержит в себе также информацию о составных
     * @param combinedAndEcomGoalsStatByGoalId - данные по конверсиям из БК для составных и еком целей
     * @return
     */
    public static Map<Long, GoalConversionVisit> mergeConversionVisitsCount(Set<Goal> metrikaGoals,
                                                                            Set<Long> goalIdsWithoutCombinedGoals,
                                                                            Map<Long, GoalConversionInfo> metrikaConversionCountByGoalId,
                                                                            Map<Long, Long> combinedAndEcomGoalsStatByGoalId) {
        return StreamEx.of(metrikaGoals)
                .mapToEntry(GoalBase::getId, goal -> getGoalConversionVisit(goalIdsWithoutCombinedGoals,
                        metrikaConversionCountByGoalId,
                        combinedAndEcomGoalsStatByGoalId,
                        goal)
                ).removeValues(conversionData -> conversionData.getCount() == null)
                .toMap();
    }

    private static GoalConversionVisit getGoalConversionVisit(Set<Long> goalIdsWithoutCombinedGoals,
                                                              Map<Long, GoalConversionInfo> metrikaConversionCountByGoalId,
                                                              Map<Long, Long> combinedAndEcomGoalsStatByGoalId,
                                                              Goal goal) {
        var goalId = goal.getId();
        var conversionFromMetrika = metrikaConversionCountByGoalId
                .getOrDefault(goalId, new GoalConversionInfo());
        Long conversionCount = calculateConversionCount(
                goalIdsWithoutCombinedGoals,
                combinedAndEcomGoalsStatByGoalId,
                goalId,
                conversionFromMetrika
        );
        return new GoalConversionVisit()
                .withCount(conversionCount)
                .withHasPrice(conversionFromMetrika.isHasPrice());
    }

    @Nullable
    private static Long calculateConversionCount(Set<Long> goalIdsWithoutCombinedGoals,
                                                 Map<Long, Long> combinedAndEcomGoalsStatByGoalId,
                                                 Long goalId,
                                                 GoalConversionInfo conversionFromMetrika) {
        Long conversionCount;
        if (goalIdsWithoutCombinedGoals.contains(goalId)) {
            conversionCount = nvl(
                    conversionFromMetrika.getCount(),
                    combinedAndEcomGoalsStatByGoalId.getOrDefault(goalId, 0L)
            );
        } else {
            conversionCount = combinedAndEcomGoalsStatByGoalId.get(goalId);
        }
        return conversionCount;
    }

    /**
     * Получить итоговый набор данных о конверсиях для целей на основе ответа из Метрики и статистики из БК.
     * Если данных нет в статистике из БК, то значением для конверсии будут данные из Метрики
     * ВАЖНО! Значение конверсии может быть null, поэтому такие значения мы отфильтровываем
     *
     * @param metrikaGoals - общий список целей из метрики
     * @param filteredMetrikaConversionCountByGoalId - данные из метрики без составных и еком целей
     * @param combinedAndEcomGoalsStatByGoalId - данные из БК для составных и еком целей
     * @return
     */
    public static Map<Long, GoalConversionVisit> mergeConversionVisitsCount(
            Set<Goal> metrikaGoals,
            Map<Long, GoalConversionInfo> filteredMetrikaConversionCountByGoalId,
            Map<Long, Long> combinedAndEcomGoalsStatByGoalId) {

        return StreamEx.of(metrikaGoals)
                .mapToEntry(Goal::getId, goal -> getGoalConversionVisit(
                        filteredMetrikaConversionCountByGoalId,
                        combinedAndEcomGoalsStatByGoalId,
                        goal)
                ).removeValues(conversionData -> conversionData.getCount() == null)
                .toMap();
    }

    private static GoalConversionVisit getGoalConversionVisit(Map<Long, GoalConversionInfo> metrikaConversionCountByGoalId,
                                                              Map<Long, Long> combinedAndEcomGoalsStatByGoalId,
                                                              Goal goal) {
        var goalId = goal.getId();
        var conversionFromMetrika = metrikaConversionCountByGoalId
                .getOrDefault(goalId, new GoalConversionInfo());
        var conversionCountFromBk = combinedAndEcomGoalsStatByGoalId.get(goalId);
        var conversionCountResult = Optional.ofNullable(conversionCountFromBk)
                .orElseGet(conversionFromMetrika::getCount);
        return new GoalConversionVisit()
                .withCount(conversionCountResult)
                .withHasPrice(conversionFromMetrika.isHasPrice());
    }

    public static Set<GdGoalConversionVisitsCount> toSetOfGdGoalConversionVisitsCount(Map<Long, GoalConversionVisit> conversionVisitsCountByGoalId) {
        return mapSet(conversionVisitsCountByGoalId.keySet(), goalId ->
                new GdGoalConversionVisitsCount()
                        .withId(goalId)
                        .withConversionVisitsCount(conversionVisitsCountByGoalId.get(goalId).getCount()));
    }

    public static GdGoalConversionVisitsCount toGdGoalConversionVisitsCount(Long goalId,
                                                                            Map<Long, Long> conversionCountByGoalId,
                                                                            Map<Long, Long> combinedAndEcomGoalsStatByGoalId) {
        Long conversionCount = combinedAndEcomGoalsStatByGoalId.getOrDefault(goalId,
                conversionCountByGoalId.getOrDefault(goalId, 0L));

        return new GdGoalConversionVisitsCount()
                .withId(goalId)
                .withConversionVisitsCount(conversionCount);
    }

    public static GdRecommendedGoalCostPerAction toGdRecommendedGoalCostPerActionForLogin(Long goalId,
                                                                                          Map<Long, BigDecimal> averageCostPerActionByGoalId) {
        return new GdRecommendedGoalCostPerAction()
                .withId(goalId)
                .withCostPerAction(averageCostPerActionByGoalId.get(goalId))
                .withCostPerActionSource(
                        ifNotNull(averageCostPerActionByGoalId.get(goalId), cost -> GdCpaSource.LOGIN));
    }

    public static GdRecommendedGoalCostPerAction toGdRecommendedGoalCostPerActionForCategory(Long goalId,
                                                                                             Map<Long, BigDecimal> averageCostPerActionByGoalId) {
        return new GdRecommendedGoalCostPerAction()
                .withId(goalId)
                .withCostPerAction(averageCostPerActionByGoalId.get(goalId))
                .withCostPerActionSource(
                        ifNotNull(averageCostPerActionByGoalId.get(goalId), cost -> GdCpaSource.CATEGORY));
    }


    public static GdRecommendedGoalCostPerAction toGdRecommendedGoalCostPerAction(GdiGoalCostPerAction goalCost,
                                                                                  CurrencyCode currencyCode,
                                                                                  boolean limitThePrice) {
        BigDecimal optimalAndLimitedCost = getOptimalAndLimitedCost(
                                    goalCost.getCostPerAction(),
                                    currencyCode,
                                    limitThePrice);
        return new GdRecommendedGoalCostPerAction()
                .withId(goalCost.getGoalId())
                .withCostPerAction(optimalAndLimitedCost)
                .withCostPerActionSource(GdCpaSource.fromSource(goalCost.getSource()));
    }

    private static BigDecimal getOptimalAndLimitedCost(
            BigDecimal costPerAction,
            CurrencyCode currencyCode,
            boolean limitThePrice) {
        if (!limitThePrice) {
            return costPerAction;
        } else {
            BigDecimal maxCost = currencyCode
                    .getCurrency()
                    .getAutobudgetPayForConversionAvgCpaWarningIncreased();
            return costPerAction.compareTo(maxCost) <= 0 ? costPerAction : maxCost;
        }
    }

    public static GdGoalsRecommendedCostPerActionByCampaignId toGdGoalsRecommendedCostPerActionByCampaignId(Long cid,
                                                                                                            List<GdRecommendedGoalCostPerAction> costs) {
        return new GdGoalsRecommendedCostPerActionByCampaignId()
                .withCampaignId(cid).withRecommendedGoalsCostPerAction(costs);
    }

    public static String costPerActionByCampaignIdToString(
            List<GdGoalsRecommendedCostPerActionByCampaignId> costPerActionByCampaignIdList) {
        return JsonUtils.toDeterministicJson(costPerActionByCampaignIdList);
    }

    public static String costPerActionToString(List<GdRecommendedGoalCostPerAction> costPerActionList) {
        return JsonUtils.toDeterministicJson(costPerActionList);
    }

    public static CounterGoal.Type toCounterGoalType(GdCreateMetrikaGoalType goalType) {
        return CounterGoal.Type.valueOf(goalType.name());
    }

    public static List<CreateCounterGoal.GoalCondition> toGoalConditions(GdCreateMetrikaGoalValue metrikaGoalValue) {
        if (metrikaGoalValue.getValueType() == SPECIFIED_VALUE) {
            return List.of(toGoalCondition(GoalConditionType.EXACT, metrikaGoalValue.getValue()));
        }
        return List.of();
    }

    public static CreateCounterGoal.GoalCondition toGoalCondition(GoalConditionType goalConditionType, String url) {
        return new CreateCounterGoal.GoalCondition()
                .withType(goalConditionType)
                .withUrl(url);
    }

    public static CreateCounterGoal toCreateCounterGoal(GdCreateMetrikaGoal createMetrikaGoalInput,
                                                        String generatedGoalName) {
        return new CreateCounterGoal()
                .withName(generatedGoalName)
                .withRetargeting(false)
                .withType(toCounterGoalType(createMetrikaGoalInput.getType()))
                .withConditions(toGoalConditions(createMetrikaGoalInput.getGoalValue()));
    }

    public static CreateCounterGoalContainer toCreateCounterGoalContainer(GdCreateMetrikaGoal createMetrikaGoalInput,
                                                                          String generatedGoalName) {
        return new CreateCounterGoalContainer()
                .withCounterId(createMetrikaGoalInput.getCounterId())
                .withCounterGoal(toCreateCounterGoal(createMetrikaGoalInput, generatedGoalName));
    }

    public static GdGoal toGdGoal(CreateCounterGoalContainer createCounterGoalContainer, Map<Long,
            MetrikaCounterWithAdditionalInformation> metrikaCounterWithAdditionalInformation) {
        var counterId = createCounterGoalContainer.getCounterId();
        var counterDomain = metrikaCounterWithAdditionalInformation.get(counterId).getDomain();
        return new GdGoal()
                .withCounterId(counterId.intValue())
                .withId(createCounterGoalContainer.getCounterGoal().getId())
                .withType(GdGoalType.GOAL)
                .withDomain(counterDomain)
                .withName(createCounterGoalContainer.getCounterGoal().getName())
                .withSimpleName(createCounterGoalContainer.getCounterGoal().getName())
                .withMetrikaGoalType(toGdMetrikaCounterGoalType(createCounterGoalContainer.getCounterGoal().getType()));
    }

    public static GdMetrikaCounterGoalType toGdMetrikaCounterGoalType(CounterGoal.Type counterGoalType) {
        return GdMetrikaCounterGoalType.valueOf(counterGoalType.name());
    }
}
