package ru.yandex.direct.grid.processing.service.goal;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsConversionService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaSegmentService;
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService;
import ru.yandex.direct.core.entity.metrika.utils.MetrikaGoalsUtils;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.mobileapp.model.MobileGoalConversions;
import ru.yandex.direct.core.entity.mobilegoals.repository.MobileGoalsStatisticRepository;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.GoalsSuggestion;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService;
import ru.yandex.direct.grid.core.entity.goal.model.GoalConversionVisit;
import ru.yandex.direct.grid.core.entity.goal.model.GoalsConversionsCacheRecord;
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignGoalsCostPerAction;
import ru.yandex.direct.grid.processing.model.goal.GdAvailableGoalsContainer;
import ru.yandex.direct.grid.processing.model.goal.GdAvailableGoalsContext;
import ru.yandex.direct.grid.processing.model.goal.GdBroadMatchContainer;
import ru.yandex.direct.grid.processing.model.goal.GdBroadMatchGoals;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignGoalFilter;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignGoals;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignsGoals;
import ru.yandex.direct.grid.processing.model.goal.GdCountersWithGoals;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.goal.GdGoalFilter;
import ru.yandex.direct.grid.processing.model.goal.GdGoalType;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsConversionVisitsCount;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsRecommendedCostPerActionByCampaignId;
import ru.yandex.direct.grid.processing.model.goal.GdMeaningfulGoalContainer;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaSegmentPresets;
import ru.yandex.direct.grid.processing.model.goal.GdRecommendedGoalCostPerAction;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.GoalConversionInfo;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.utils.InterruptedRuntimeException;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsConversionService.DAYS_WITH_CONVERSION_VISITS;
import static ru.yandex.direct.feature.FeatureName.CATEGORY_CPA_SOURCE_FOR_CONVERSION_PRICE_RECOMMENDATION;
import static ru.yandex.direct.feature.FeatureName.CREATION_OF_METRIKA_SEGMENTS_BY_PRESETS_ENABLED;
import static ru.yandex.direct.feature.FeatureName.ENABLE_GOAL_CONVERSION_STATISTICS_FOR_7_DAYS;
import static ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter.toCampaignType;
import static ru.yandex.direct.grid.processing.service.goal.GoalConstant.SUPPORTED_GOAL_SUB_TYPE_VALUES;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.costPerActionByCampaignIdToString;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.costPerActionToString;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.mergeConversionVisitsCount;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toFilteredRowset;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdCounterWithGoals;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoal;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoals;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdRecommendedGoalCostPerActionForCategory;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdRecommendedGoalCostPerActionForLogin;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toSetOfGdGoalConversionVisitsCount;
import static ru.yandex.direct.grid.processing.service.goal.GoalServiceUtils.getComparator;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.isValidId;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class GoalDataService {
    private static final Logger logger = LoggerFactory.getLogger(GoalDataService.class);
    private static final int DAYS_FOR_PROCESSING_AVARAGE_COST_PER_CLICK = 7;
    private static final int DAYS_FOR_PROCESSING_TOTAL_ACTIONS_PER_CLIENT_CAMPAIGNS = 7;

    private final MetrikaGoalsService metrikaGoalsService;
    private final MetrikaSegmentService metrikaSegmentService;
    private final CampaignGoalsService campaignGoalsService;
    private final RetargetingConditionService retargetingConditionService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final MetrikaGoalsConversionService metrikaGoalsConversionService;
    private final MobileGoalsStatisticRepository mobileGoalsStatisticRepository;
    private final GridCampaignService gridCampaignService;
    private final FeatureService featureService;
    private final GoalConversionsCacheService goalConversionsCacheService;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;
    @Autowired
    public GoalDataService(MetrikaGoalsService metrikaGoalsService,
                           MetrikaSegmentService metrikaSegmentService,
                           CampaignGoalsService campaignGoalsService,
                           RetargetingConditionService retargetingConditionService,
                           CampMetrikaCountersService campMetrikaCountersService,
                           MetrikaGoalsConversionService metrikaGoalsConversionService,
                           MobileGoalsStatisticRepository mobileGoalsStatisticRepository,
                           GridCampaignService gridCampaignService,
                           FeatureService featureService,
                           GoalConversionsCacheService goalConversionsCacheService,
                           CampaignRepository campaignRepository,
                           ShardHelper shardHelper) {
        this.metrikaGoalsService = metrikaGoalsService;
        this.metrikaSegmentService = metrikaSegmentService;
        this.campaignGoalsService = campaignGoalsService;
        this.retargetingConditionService = retargetingConditionService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.metrikaGoalsConversionService = metrikaGoalsConversionService;
        this.mobileGoalsStatisticRepository = mobileGoalsStatisticRepository;
        this.gridCampaignService = gridCampaignService;
        this.featureService = featureService;
        this.goalConversionsCacheService = goalConversionsCacheService;
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
    }

    public List<GdGoal> getClientGoals(ClientId clientId, GdGoalFilter filter) {
        List<Goal> goals = retargetingConditionService.getMetrikaGoalsForRetargeting(clientId);
        return toFilteredRowset(filterSupportedGoals(goals), filter);
    }

    /**
     * Получает список целей и сортирует с {@link GoalDataService#goalComparator} (для порядка отображения)
     * todo: https://st.yandex-team.ru/DIRECT-115007,
     */
    public List<GdGoal> getCampaignsGoals(Long operatorUid, ClientId clientId, GdCampaignGoalFilter filter,
                                          boolean expandGoalData) {
        List<Goal> goals;
        if (featureService.isEnabledForClientId(clientId, FeatureName.MIGRATING_TO_NEW_METHODS_IN_GET_CAMPAIGN_GOALS)) {
            // получение счетчиков кампаний
            Set<Long> campaignIds = nvl(filter.getCampaignIdIn(), Set.of());
            try {
                goals = getCampaignGoalsFromCampaignServices(operatorUid, clientId, campaignIds);
            } catch (Exception e) {
                goals = Collections.emptyList();
                logger.warn("Got an exception when querying for goals for clientId: {}",
                        clientId, e);
            }
        } else if (featureService.isEnabledForClientId(clientId, FeatureName.GOALS_ONLY_WITH_CAMPAIGN_COUNTERS_USED)) {
            try {
                goals = metrikaGoalsService.getGoalsWithCounters(operatorUid, clientId, filter.getCampaignIdIn(),
                        expandGoalData);
            } catch (MetrikaClientException | InterruptedRuntimeException e) {
                goals = Collections.emptyList();
                logger.warn("Got an exception when querying for metrika counters and goals for clientId: {}",
                        clientId, e);
            }
        } else {
            goals = metrikaGoalsService.getGoals(operatorUid, clientId, filter.getCampaignIdIn());
        }
        goals.sort(goalComparator());
        return toGdGoals(filterSupportedGoals(goals));
    }

    @NotNull
    private List<Goal> getCampaignGoalsFromCampaignServices(Long operatorUid, ClientId clientId, Set<Long> campaignIds) {
        // получение типов компаний по их id
        Map<Long, CampaignType> campaignIdToCampaignType = campaignRepository.getCampaignsTypeMap(
                shardHelper.getShardByClientId(clientId),
                campaignIds);
        // получение счётчиков компаний доступных клиенту по компаниям
        Map<Long, List<Long>> counterByCampaignIds = campMetrikaCountersService.getCounterByCampaignIds(clientId, campaignIds);
        // сборка мапы из айди компаний в связку (тип компании + счётчики компании)
        Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId =
                listToMap(campaignIds, identity(),
                        cid -> getLongCampaignTypeWithCounterIds(
                                            campaignIdToCampaignType,
                                            counterByCampaignIds, cid));

        // получение доступных целей для клиента по мапе из компаний в связки (тип компании + счётчики компании)
        Map<Long, Set<Goal>> availableGoalsForCampaignId = campaignGoalsService.getAvailableGoalsForCampaignId(
                operatorUid,
                clientId,
                campaignTypeWithCounterIdsByCampaignId,
                null);
        return StreamEx.of(availableGoalsForCampaignId.values())
                .flatMap(Set<Goal>::stream)
                .distinct()
                .collect(Collectors.toList());
    }

    @NotNull
    private CampaignTypeWithCounterIds getLongCampaignTypeWithCounterIds(
            Map<Long, CampaignType> campaignIdToCampaignType,
            Map<Long, List<Long>> counterByCampaignIds,
            Long campaignId) {
        return new CampaignTypeWithCounterIds()
                .withCampaignType(campaignIdToCampaignType.getOrDefault(campaignId, CampaignType.TEXT))
                .withCounterIds(new HashSet<>(counterByCampaignIds.getOrDefault(campaignId, List.of())));
    }

    /**
     * Сортируем цели так, чтобы дочерние цели шли сразу после родительской
     */
    private static Comparator<Goal> goalComparator() {
        Function<Goal, Long> getParentIfExists = goal ->
                isValidId(goal.getParentId()) ? goal.getParentId() : goal.getId();
        return Comparator.comparing(getParentIfExists)
                .thenComparing(goal -> isValidId(goal.getParentId()) ? goal.getId() : 0L);
    }

    private List<Goal> filterSupportedGoals(List<Goal> goals) {
        return filterList(goals, this::goalSubtypeIsSupported);
    }

    private boolean goalSubtypeIsSupported(Goal goal) {
        return goal.getSubtype() == null || SUPPORTED_GOAL_SUB_TYPE_VALUES.contains(goal.getSubtype());
    }

    /**
     * Возвращает ключевые цели для существующих кампаний.
     *
     * @param operatorUid uid оператора
     * @param clientId    id клиента
     * @param containers  список редактируемых кампаний
     * @return Список целей по кампаниям
     */
    public GdCampaignsGoals getGdMeaningfulGoals(Long operatorUid, ClientId clientId,
                                                 List<GdMeaningfulGoalContainer> containers) {
        try {
            Map<Long, Set<Goal>> availableGoalsForCampaignId;
            Map<Long, CampaignTypeWithCounterIds> campaignTypeWithCounterIdsByCampaignId = StreamEx.of(containers)
                    .mapToEntry(GdMeaningfulGoalContainer::getCampaignId,
                            campaign -> new CampaignTypeWithCounterIds()
                                    .withCampaignType(toCampaignType(campaign.getCampaignType()))
                                    .withCounterIds(campaign.getCounterIds()))
                    .filterValues(c -> c.getCounterIds() != null)
                    .toMap();

            availableGoalsForCampaignId = campaignGoalsService
                    .getAvailableGoalsForCampaignId(operatorUid, clientId,
                            campaignTypeWithCounterIdsByCampaignId, null);

            List<GdCampaignGoals> gdCampaignGoals = EntryStream.of(availableGoalsForCampaignId)
                    .mapKeyValue((campaignId, goals) -> new GdCampaignGoals()
                            .withCampaignId(campaignId)
                            .withGoals(toGdGoals(goals)))
                    .toList();

            return new GdCampaignsGoals()
                    .withCampaignGoals(gdCampaignGoals)
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for meaningful goals for clientId: " + clientId, e);
            return new GdCampaignsGoals()
                    .withCampaignGoals(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    /**
     * Возвращает цели по счетчикам.
     *
     * @param counterIds список id счетчиков
     * @param clientId   id клиента
     * @return Список целей по счетчикам
     */
    public GdCountersWithGoals getGdGoalsByCounterIds(Long operatorUid,
                                                      Set<Long> counterIds,
                                                      ClientId clientId) {
        if (SetUtils.emptyIfNull(counterIds).isEmpty()) {
            return new GdCountersWithGoals()
                    .withCounterWithGoals(null)
                    .withIsMetrikaAvailable(true);
        }
        try {
            //Достаем цели только для счетчиков к которым есть доступ
            Set<Long> availableCounterIds =
                    campMetrikaCountersService.getAvailableCounterIdsByClientId(clientId, counterIds);
            Map<Long, Set<Goal>> availableGoalsForCounterId = metrikaGoalsService
                    .getMetrikaGoalsByCounterIds(clientId, availableCounterIds);

            return toGdCountersWithGoals(operatorUid, clientId, availableGoalsForCounterId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            return new GdCountersWithGoals()
                    .withCounterWithGoals(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdAvailableGoalsContext getAvailableGoals(
            Long operatorUid, ClientId clientId, @Nullable GdAvailableGoalsContainer input
    ) {
        try {
            Set<Goal> availableGoals =
                    metrikaGoalsService.getAvailableMetrikaGoalsForClient(operatorUid, clientId);
            receiveAndAddConversionVisitsToGoals(operatorUid, clientId, availableGoals, false);

            GoalsSuggestion goalsSuggestion = metrikaGoalsService.getGoalsSuggestion(clientId, availableGoals);

            var goalStream = StreamEx.of(goalsSuggestion.getSortedGoalsToSuggestion())
                    .map(goal -> toGdGoal(goal, null));
            if (input != null) {
                if (input.getOrderBy() != null) {
                    goalStream = goalStream.sorted(getComparator(input.getOrderBy()));
                }
                if (input.getLimitOffset() != null) {
                    goalStream = goalStream
                            .skip(input.getLimitOffset().getOffset())
                            .limit(input.getLimitOffset().getLimit());
                }
            }
            var rowset = goalStream.toList();

            return new GdAvailableGoalsContext()
                    .withRowset(rowset)
                    .withTotalCount(rowset.size())
                    .withTop1GoalId(goalsSuggestion.getTop1GoalId())
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdAvailableGoalsContext()
                    .withRowset(Collections.emptyList())
                    .withTotalCount(0)
                    .withTop1GoalId(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    /**
     * Скрываем конверсии для недоступных целей
     */
    public Set<GdGoal> toGdGoalsWithRemoveConversionsForUnavailable(GoalsSuggestion goalsSuggestion,
                                                                    Set<Long> availableCounterIds,
                                                                    boolean isUnavailableGoalsAllowed) {
        Set<Goal> goals = StreamEx.of(goalsSuggestion.getSortedGoalsToSuggestion()).toSet();
        return toGdGoalsWithRemoveConversionsForUnavailable(goals, availableCounterIds, isUnavailableGoalsAllowed);
    }

    public Set<GdGoal> toGdGoalsWithRemoveConversionsForUnavailable(Set<Goal> goals,
                                                                    Set<Long> availableCounterIds,
                                                                    boolean isUnavailableGoalsAllowed) {
        Set<GdGoal> gdGoals = toGdGoals(goals);
        if (!isUnavailableGoalsAllowed) {
            return gdGoals;
        }
        Map<Long, Set<Long>> goalIdToCouterIds = StreamEx.of(goals)
                .mapToEntry(Goal::getId, MetrikaGoalsUtils::counterIdsFromGoal)
                .toMap();

        StreamEx.of(gdGoals)
                .filter(gdGoal -> gdGoal.getType() != GdGoalType.MOBILE
                        && Sets.intersection(goalIdToCouterIds.get(gdGoal.getId()), availableCounterIds).isEmpty())
                .forEach(gdGoal -> gdGoal.withConversionVisitsCount(null));
        return gdGoals;
    }

    public GdCountersWithGoals getAbSegmentsByCounterIds(Long operatorUid,
                                                         Set<Long> counterIds,
                                                         ClientId clientId) {
        counterIds = SetUtils.emptyIfNull(counterIds);
        try {
            var availableCounterIds = StreamEx.of(campMetrikaCountersService
                    .getAvailableCounterIdsByClientId(clientId, counterIds))
                    .toSet();
            //Достаем цели только для счетчиков к которым есть доступ
            Map<Long, Set<Goal>> availableGoalsByCounterId = metrikaGoalsService
                    .getAbSegmentsByCounterIds(clientId, availableCounterIds);

            return toGdCountersWithGoals(operatorUid, clientId, availableGoalsByCounterId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika goals for clientId: " + clientId, e);
            return new GdCountersWithGoals()
                    .withCounterWithGoals(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdCampaignsGoals getAbSegments(List<GdMeaningfulGoalContainer> containers, ClientId clientId) {
        List<Long> campaignIds = mapList(containers, GdMeaningfulGoalContainer::getCampaignId);
        Map<Long, Set<Long>> counterIdsByCampaignId = StreamEx.of(containers)
                .mapToEntry(GdMeaningfulGoalContainer::getCampaignId,
                        GdMeaningfulGoalContainer::getCounterIds)
                .nonNullValues()
                .toMap();

        try {
            Map<Long, Set<Goal>> abSegments = metrikaGoalsService.getAbSegments(clientId, counterIdsByCampaignId);
            List<GdCampaignGoals> campaignGoals = mapList(campaignIds, campaignId -> new GdCampaignGoals()
                    .withGoals(toGdGoals(abSegments.get(campaignId)))
                    .withCampaignId(campaignId));
            return new GdCampaignsGoals()
                    .withCampaignGoals(campaignGoals)
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdCampaignsGoals()
                    .withCampaignGoals(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdBroadMatchGoals getGdBroadMatches(List<@GraphQLNonNull GdBroadMatchContainer> containers,
                                               Long operatorUid, ClientId clientId) {
        var campaignIdToCampaignType = listToMap(containers, GdBroadMatchContainer::getCampaignId,
                c -> toCampaignType(c.getCampaignType()));
        try {
            Map<Long, Set<Goal>> availableGoalsForCampaignId = metrikaGoalsService
                    .getAvailableBroadMatchesForCampaignId(operatorUid, clientId, campaignIdToCampaignType);

            List<GdCampaignGoals> gdCampaignGoals = EntryStream.of(availableGoalsForCampaignId)
                    .mapKeyValue((campaignId, goals) -> new GdCampaignGoals()
                            .withCampaignId(campaignId)
                            .withGoals(toGdGoals(goals)))
                    .toList();

            return new GdBroadMatchGoals()
                    .withCampaignGoals(gdCampaignGoals)
                    .withIsMetrikaAvailable(true);

        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdBroadMatchGoals()
                    .withCampaignGoals(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public GdMetrikaSegmentPresets getSegmentPresets(ClientId clientId) {
        if (!featureService.isEnabledForClientId(clientId, CREATION_OF_METRIKA_SEGMENTS_BY_PRESETS_ENABLED)) {
            return new GdMetrikaSegmentPresets()
                    .withPresets(List.of())
                    .withIsMetrikaAvailable(true);
        }

        List<MetrikaSegmentPreset> presets;
        try {
            presets = metrikaSegmentService.getSegmentPresets(clientId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got and exception when querying for metrika segment presets for clientId: " + clientId, e);
            return new GdMetrikaSegmentPresets()
                    .withPresets(List.of())
                    .withIsMetrikaAvailable(false);
        }
        var presetsList = mapList(presets, GoalDataConverter::toGdMetrikaSegmentPreset);
        return new GdMetrikaSegmentPresets()
                .withPresets(presetsList)
                .withIsMetrikaAvailable(true);
    }

    public GdGoalsConversionVisitsCount getGdGoalsConversionVisitsCountForAllGoals(Long operatorUid, ClientId clientId,
                                                                                   Set<Long> counterIds) {
        Map<Long, GoalConversionVisit> conversionVisitsCountByGoalId =
                getGoalsConversionVisitsCountForAllGoals(operatorUid, clientId, counterIds);

        if (conversionVisitsCountByGoalId == null) {
            return new GdGoalsConversionVisitsCount()
                    .withIsMetrikaAvailable(false)
                    .withGoalsConversionVisitsCount(emptySet());
        }

        return new GdGoalsConversionVisitsCount()
                .withIsMetrikaAvailable(true)
                .withGoalsConversionVisitsCount(toSetOfGdGoalConversionVisitsCount(conversionVisitsCountByGoalId));
    }

    /**
     * Получить данные о конверсиях по всем доступным целям заданных счетчиков, включая составные цели и ecommerce.
     * Для ecommerce и составных целей считаем среднее кол-во конверсий по логину. (Метрика не возвращает статистику)
     * <p>
     * В отличии от getGoalsConversionVisitsCount возвращает цели с нулевой статистикой.
     */
    @Nullable
    public Map<Long, GoalConversionVisit> getGoalsConversionVisitsCountForAllGoals(Long operatorUid, ClientId clientId,
                                                                                   Set<Long> counterIds) {
        // получение информации о доступных счетчиках клиента, по переданному списку id счетчиков
        Function<Set<Long>, Set<MetrikaCounterWithAdditionalInformation>> getCounterInfoForStat = (metrikaCounterIds) ->
                getAvailableCountersForRequest(clientId, metrikaCounterIds);

        // получение доступных целей по всем переданным счетчикам
        Function<Set<MetrikaCounterWithAdditionalInformation>, Set<Goal>> getGoals = (counterInformations) ->
                metrikaGoalsService.getAvailableMetrikaGoalsForClient(operatorUid, clientId, counterInformations);

        return getGoalsConversionVisitsCountForAllGoals(clientId, counterIds, getCounterInfoForStat, getGoals, false);
    }

    /**
     * Получить данные о конверсиях по всем переданным целям
     * <p>
     * В отличии от getGoalsConversionVisitsCount возвращает цели с нулевой статистикой.
     */
    @Nullable
    public Map<Long, GoalConversionVisit> getGoalsConversionVisitsCountForAllGoals(ClientId clientId,
                                                                                   Set<Goal> goals) {
        // получение информации о счетчиках клиента (только с id счетчика), по переданному списку id счетчиков
        Function<Set<Long>, Set<MetrikaCounterWithAdditionalInformation>> getCounterIdsForStat = (metrikaCounterIds) ->
                mapSet(metrikaCounterIds, counterId -> new MetrikaCounterWithAdditionalInformation()
                        .withId(counterId));

        // получение всех переданных целей
        Function<Set<MetrikaCounterWithAdditionalInformation>, Set<Goal>> getGoals = (counterInformations) -> goals;

        Set<Long> counterIds = FunctionalUtils.filterAndMapToSet(goals, goal -> goal.getCounterId() != null,
                goal -> goal.getCounterId().longValue());

        return getGoalsConversionVisitsCountForAllGoals(clientId, counterIds, getCounterIdsForStat, getGoals, true);
    }

    /**
     * Получить данные о конверсиях по целям
     *
     * @param clientId              id клиента
     * @param counterIds            список id счетчиков
     * @param getCounterInfoForStat функция получения информации по счетчикам, по которым будет собираться статистика
     * @param getGoals              функция получения целей по которым будет собираться статистика
     * @param withUnavailableGoals  флаг показа недоступных целей, в зависимости от которого будут собираться
     *                              конверсии для недоступных целей
     */
    private Map<Long, GoalConversionVisit> getGoalsConversionVisitsCountForAllGoals(
            ClientId clientId,
            Set<Long> counterIds,
            Function<Set<Long>, Set<MetrikaCounterWithAdditionalInformation>> getCounterInfoForStat,
            Function<Set<MetrikaCounterWithAdditionalInformation>, Set<Goal>> getGoals,
            boolean withUnavailableGoals
    ) {
        try {
            GoalsConversionsCacheRecord cachedConversionVisitsCount =
                    goalConversionsCacheService.getFromCache(clientId.asLong(), counterIds, withUnavailableGoals);
            if (cachedConversionVisitsCount != null) {
                return cachedConversionVisitsCount.getConversionVisitsCount();
            }

            Set<MetrikaCounterWithAdditionalInformation> countersToRequest = getCounterInfoForStat.apply(counterIds);
            Set<Goal> metrikaGoals = getGoals.apply(countersToRequest);

            var counterIdsForStat = mapSet(countersToRequest, MetrikaCounterWithAdditionalInformation::getId);

            //получить статистику конверсий по целям из метричной ручки
            Map<Long, GoalConversionInfo> conversionInfoByGoalId =
                    metrikaGoalsConversionService.getGoalsConversion(clientId, counterIdsForStat, withUnavailableGoals);

            //под этой же фичей включаем раскраску красным целей без статистики (DIRECT-135080)
            var conversionVisitsCountByGoalId = enrichMetrikaGoalsConversions(clientId,
                    metrikaGoals, conversionInfoByGoalId, featureService.isEnabledForClientId(clientId,
                            ENABLE_GOAL_CONVERSION_STATISTICS_FOR_7_DAYS));

            GoalsConversionsCacheRecord cacheRecord = new GoalsConversionsCacheRecord()
                    .withClientId(clientId.asLong())
                    .withCounterIds(counterIds)
                    .withConversionVisitsCount(conversionVisitsCountByGoalId);
            goalConversionsCacheService.saveToCache(cacheRecord, withUnavailableGoals);

            return conversionVisitsCountByGoalId;
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got and exception when querying for metrika goals conversion visits for clientId: " + clientId, e);
            return null;
        }
    }

    /**
     * Обогатить статистику по кол-ву достижений целей из Метрики данными из БК
     *
     * @param clientId                                          клиент
     * @param metrikaGoals                                      цели
     * @param metrikaConversionCountByGoalId                    статистика по целям, полученная из Метрики
     * @param setZeroConversionsForNonCombinedGoalsWithoutStats нужно ли проставить нулевое кол-во достижений
     *                                                          несоставных целей,
     *                                                          для которых нет статистики (раскрашиваем цель в красный)
     * @return мапа id цели -> кол-во достижений цели по объединенной статистике из Метрики и БК
     */
    public Map<Long, GoalConversionVisit> enrichMetrikaGoalsConversions(ClientId clientId, Set<Goal> metrikaGoals,
                                                                        Map<Long, GoalConversionInfo> metrikaConversionCountByGoalId,
                                                                        boolean setZeroConversionsForNonCombinedGoalsWithoutStats) {

        //статистику по составным и ecom-целям запрашиваем из БК, а из ответа метрики исключаем
        Map<Long, Long> bsConversionCountByGoalId = getBsStatForCombinedAndEcomGoals(clientId, metrikaGoals);

        if (setZeroConversionsForNonCombinedGoalsWithoutStats) {
            var goalIdsWithoutCombinedGoals = getGoalsWithoutCombinedGoals(metrikaGoals);
            return mergeConversionVisitsCount(metrikaGoals,
                    goalIdsWithoutCombinedGoals,
                    metrikaConversionCountByGoalId, bsConversionCountByGoalId);
        } else {
            //Метрика не возвращает статистику по составным целям. Но в редких случаях это происходит
            //Защищаемся от таких случаев, чтобы не смешивать конверсии с ответом БК.
            var filteredMetrikaConversionCountByGoalId =
                    getGoalsConversionsWithoutEcomAndCombinedGoals(metrikaGoals, metrikaConversionCountByGoalId);
            return mergeConversionVisitsCount(metrikaGoals,
                    filteredMetrikaConversionCountByGoalId, bsConversionCountByGoalId);
        }
    }

    private static Map<Long, GoalConversionInfo> getGoalsConversionsWithoutEcomAndCombinedGoals(
            Set<Goal> metrikaGoals,
            Map<Long, GoalConversionInfo> metrikaConversionCountByGoalId) {
        Set<Long> combinedAndEcomGoalIds = getCombinedAndEcomGoalIds(metrikaGoals);
        return EntryStream.of(metrikaConversionCountByGoalId)
                .filterKeys(goalId -> !combinedAndEcomGoalIds.contains(goalId))
                .toMap();
    }

    private static Set<Long> getGoalsWithoutCombinedGoals(Set<Goal> metrikaGoals) {
        Set<Long> combinedAndEcomGoalIds = getCombinedGoalIds(metrikaGoals);
        return filterAndMapToSet(metrikaGoals, goal -> !combinedAndEcomGoalIds.contains(goal.getId()), Goal::getId);
    }

    private static Set<Long> getCombinedAndEcomGoalIds(Set<Goal> metrikaGoals) {
        return filterAndMapToSet(metrikaGoals,
                g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.ECOMMERCE
                        || g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.STEP
                        || nvl(g.getParentId(), 0L) != 0L,
                Goal::getId);
    }

    private static Set<Long> getCombinedGoalIds(Set<Goal> metrikaGoals) {
        return filterAndMapToSet(metrikaGoals,
                g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.STEP
                        || nvl(g.getParentId(), 0L) != 0L,
                Goal::getId);
    }

    /**
     * Ограничить набор счетчиков только доступными счетчиками
     * Если исходный набор счетчиков пуст, возвращает набор всех доступных счетчиков
     */
    private Set<MetrikaCounterWithAdditionalInformation> getAvailableCountersForRequest(ClientId clientId,
                                                                                        Set<Long> counterIds) {
        if (counterIds.isEmpty()) {
            return campMetrikaCountersService.getAvailableCountersForGoals(clientId);
        }
        Set<MetrikaCounterWithAdditionalInformation> availableCounters =
                campMetrikaCountersService.getAvailableAndFilterInputCountersInMetrikaForGoals(clientId, counterIds);
        return filterToSet(availableCounters, counterInfo -> counterIds.contains(counterInfo.getId()));
    }

    /**
     * Возвращает статистику по составным и ecom-целям из таблиц БК.
     * Валидирует значения конверсий в составных целях.
     * В случае, если составная цель содержит некорректную статистику, такая цель возвращаться не будет (вместе с
     * подцелями)
     */
    private Map<Long, Long> getBsStatForCombinedAndEcomGoals(ClientId clientId, Set<Goal> metrikaGoals) {
        //по составным и ecom целями запрашиваем статистику из БК на логин
        Set<Long> combinedAndEcomGoalIds = getCombinedAndEcomGoalIds(metrikaGoals);
        if (combinedAndEcomGoalIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, Goal> combinedGoalsByIds = StreamEx.of(metrikaGoals)
                .filter(g -> g.getMetrikaCounterGoalType() == MetrikaCounterGoalType.STEP
                        || nvl(g.getParentId(), 0L) != 0L)
                .toMap(Goal::getId, identity());

        Pair<LocalDate, LocalDate> startDateEndDate =
                getBoundaryDatesForTodayMinusDays(DAYS_FOR_PROCESSING_TOTAL_ACTIONS_PER_CLIENT_CAMPAIGNS);
        //По составным целям статистика конверсий есть только на последний шаг.
        //поэтому проставляем значение последего шага всем шагам составной цели
        //последним шагом считаем цель с минимальным кол-вом конверсий //todo: добавить subgoal_index (DIRECT-128260)
        Map<Long, Long> conversionsCountByGoalId =
                gridCampaignService.getTotalActionsCountForGoalsInAllClientCampaigns(clientId,
                        combinedAndEcomGoalIds, startDateEndDate.getLeft(),
                        startDateEndDate.getRight());

        //если нет статистики по родительской цели, дописываем в нее минимальную статистику из составляющих ее шагов
        Map<Long, Long> minGoalConversionByParentGoalId = EntryStream.of(conversionsCountByGoalId)
                .filterKeys(combinedGoalsByIds::containsKey)
                .filterKeys(goalId -> nvl(combinedGoalsByIds.get(goalId).getParentId(), 0L) != 0)
                .mapKeys(goalId -> combinedGoalsByIds.get(goalId).getParentId())
                .nonNullKeys()
                .toMap(BinaryOperator.minBy(Comparator.comparing(Long::longValue)));

        combinedGoalsByIds.forEach((id, goal) -> {
            var parentGoalId = nvl(goal.getParentId(), 0L) != 0L ? goal.getParentId() : id;
            var conversionNum = minGoalConversionByParentGoalId.getOrDefault(parentGoalId, 0L);
            conversionsCountByGoalId.putIfAbsent(id, conversionNum);
        });

        //исключаем составные цели и их шаги, если кол-во конверсий в них не консистентно.
        Set<Long> goalIdsToExclude = getGoalsWithInconsistentConversions(combinedGoalsByIds, conversionsCountByGoalId,
                minGoalConversionByParentGoalId);
        goalIdsToExclude.forEach(conversionsCountByGoalId::remove);

        return conversionsCountByGoalId;
    }

    /**
     * Метод нужен для защиты от некорректных данных в статистике БК.
     * Проверяет совпадает ли статистика в родительской цели с минимальной статистикой в ее подцелях.
     * Возвращает все родительские цели (с подцелями), в которых статистика не совпадает.
     */
    private Set<Long> getGoalsWithInconsistentConversions(Map<Long, Goal> combinedGoalsByIds,
                                                          Map<Long, Long> conversionsCountByGoalId,
                                                          Map<Long, Long> minConversionsByParentGoalId) {

        Set<Long> parentGoalIds = filterAndMapToSet(combinedGoalsByIds.values(),
                goal -> goal.isStepGoalParent() &&
                        isParentGoalHasInconsistencyWithSubGoals(goal.getId(), conversionsCountByGoalId,
                                minConversionsByParentGoalId),
                GoalBase::getId);

        if (!parentGoalIds.isEmpty()) {
            return filterAndMapToSet(combinedGoalsByIds.values(),
                    goal -> parentGoalIds.contains(goal.getId())
                            || parentGoalIds.contains(goal.getParentId()),
                    Goal::getId);
        }
        return emptySet();
    }

    private static boolean isParentGoalHasInconsistencyWithSubGoals(Long goalId,
                                                                    Map<Long, Long> conversionsCountByGoalId,
                                                                    Map<Long, Long> minConversionsByParentGoalId) {
        return !Objects.equals(conversionsCountByGoalId.get(goalId),
                minConversionsByParentGoalId.getOrDefault(goalId, 0L));
    }


    /**
     * Рассчет (рекомендации) стоимости конверсии для целей в заданной кампании
     * Если в заданной кампании цель не достигалась и стоимость конверсии 0, то для данной цели будет рассчитана
     * средняя стоимость конверсии в рамках всех кампаний клиена, в которых она достигалась
     *
     * @param clientId            - id клиента
     * @param goalIdsByCampaignId - мапа с кампаниями и целями, для которых нужно получить рекомендуемую (среднюю)
     * @param currencyCode        - инфорация о валюте (передается, чтобы узнавать ограничения на макс. цены)
     */
    public List<GdGoalsRecommendedCostPerActionByCampaignId> getCampaignsGoalsCostPerActionRecommendations(
            Long operatorUid, ClientId clientId,
            Map<Long, List<Long>> goalIdsByCampaignId,
            CurrencyCode currencyCode,
            Map<Long, String> urlByCampaignId,
            boolean removeUnavailableGoals) {
        LocalDate endDate = LocalDate.now();
        LocalDate startDate = endDate.minusDays(DAYS_FOR_PROCESSING_AVARAGE_COST_PER_CLICK);

        Map<Long, List<Long>> availableGoalIdsByCampaignId = removeUnavailableGoals ?
                removeUnavailableGoals(operatorUid, clientId, goalIdsByCampaignId) : goalIdsByCampaignId;
        Map<Long, GdiCampaignGoalsCostPerAction> averageCostPerActionForCampaignGoals =
                gridCampaignService.getAverageCostPerActionForGoalsInSpecifiedCampaigns(clientId,
                        availableGoalIdsByCampaignId, urlByCampaignId, startDate, endDate);

        boolean isLimitationOfCostEnabledForClient =
                featureService.isEnabledForClientId(clientId, FeatureName.MAXIMUM_RECOMMENDED_GOAL_COST_BOUNDARY);

        List<GdGoalsRecommendedCostPerActionByCampaignId> result = EntryStream.of(averageCostPerActionForCampaignGoals)
                .mapValues(GdiCampaignGoalsCostPerAction::getGoalsCostPerAction)
                .mapValues(goalsCost -> mapList(goalsCost,
                        goalCost -> GoalDataConverter.toGdRecommendedGoalCostPerAction(
                                goalCost, currencyCode, isLimitationOfCostEnabledForClient)))
                .mapKeyValue(GoalDataConverter::toGdGoalsRecommendedCostPerActionByCampaignId)
                .toList();

        logger.info("Recommended CPA for clientId {}: {}", clientId, costPerActionByCampaignIdToString(result));
        return result;
    }

    /**
     * Расчет (рекомендуемой) средней стоимости конверсии для заданных целей в рамках всех кампаний клиента
     * и по категории.
     *
     * @param clientId - id клиента
     * @param goalIds  - список целей, для которых нужно посчитать среднюю стоимость конверсии
     */
    public List<GdRecommendedGoalCostPerAction> getGoalsCostPerActionRecommendations(
            Long operatorUid, ClientId clientId,
            List<Long> goalIds, @Nullable String url,
            boolean removeUnavailableGoals) {
        LocalDate endDate = LocalDate.now();
        LocalDate startDate = endDate.minusDays(DAYS_FOR_PROCESSING_AVARAGE_COST_PER_CLICK);

        Set<Long> availableGoalIds = removeUnavailableGoals ?
                removeUnavailableGoals(operatorUid, clientId, goalIds) : Set.copyOf(goalIds);
        Map<Long, BigDecimal> averageCostPerActionWithLoginCpaSource =
                gridCampaignService.getAverageCostPerActionForGoalsByLoginStats(clientId,
                        availableGoalIds, startDate, endDate);
        Set<Long> loginCpaSourceNonNullGoalIds = averageCostPerActionWithLoginCpaSource.keySet();

        List<GdRecommendedGoalCostPerAction> result =
                allToGdRecommendedGoalCostPerActionForLogin(loginCpaSourceNonNullGoalIds,
                        averageCostPerActionWithLoginCpaSource);

        String nonNullUrl = nvl(url, "");
        boolean categoryCpaSourceFeatureEnabled = featureService.isEnabledForClientId(clientId,
                CATEGORY_CPA_SOURCE_FOR_CONVERSION_PRICE_RECOMMENDATION);
        boolean allGoalsHaveStats = allGoalsHaveStats(loginCpaSourceNonNullGoalIds, availableGoalIds);
        if (!allGoalsHaveStats && categoryCpaSourceFeatureEnabled) {
            Set<Long> categoryCpaSourceGoalIds = StreamEx.of(availableGoalIds)
                    .filter(gid -> !loginCpaSourceNonNullGoalIds.contains(gid))
                    .toSet();

            result.addAll(recommendedGoalCostWithCategoryCpaSource(categoryCpaSourceGoalIds, clientId, nonNullUrl));
        }

        logger.info("Recommended CPA for clientId {}: {}", clientId, costPerActionToString(result));
        return result;
    }

    private Set<Long> removeUnavailableGoals(Long operatorUid, ClientId clientId, List<Long> goalIds) {
        Set<Long> availableGoalIds = listToSet(
                metrikaGoalsService.getAvailableMetrikaGoalsForClient(operatorUid, clientId), Goal::getId);
        return filterToSet(goalIds, availableGoalIds::contains);
    }

    private Map<Long, List<Long>> removeUnavailableGoals(Long operatorUid, ClientId clientId,
                                                         Map<Long, List<Long>> goalIdsByCampaignId) {
        Set<Long> availableGoalIds = listToSet(
                metrikaGoalsService.getAvailableMetrikaGoalsForClient(operatorUid, clientId), Goal::getId);
        return EntryStream.of(goalIdsByCampaignId)
                .mapValues(goalIds -> filterList(goalIds, availableGoalIds::contains))
                .toMap();
    }

    private boolean allGoalsHaveStats(Set<Long> loginCpaSourceNonNullGoalIds,
                                      Set<Long> goalIds) {
        return loginCpaSourceNonNullGoalIds.size() == goalIds.size();
    }

    private List<GdRecommendedGoalCostPerAction> allToGdRecommendedGoalCostPerActionForLogin(Set<Long> goalIds,
                                                                                             Map<Long, BigDecimal> averageCostPerActionWithLoginCpaSource) {
        return mapList(goalIds,
                goalId -> toGdRecommendedGoalCostPerActionForLogin(goalId, averageCostPerActionWithLoginCpaSource));
    }

    private List<GdRecommendedGoalCostPerAction> recommendedGoalCostWithCategoryCpaSource(Set<Long> categoryCpaSourceGoalIds,
                                                                                          ClientId clientId,
                                                                                          String url) {
        Map<Long, BigDecimal> averageCostPerActionWithCategoryCpaSource =
                gridCampaignService.getAverageCostPerActionForGoalsByWithCategoryCpaSource(clientId,
                        categoryCpaSourceGoalIds, url).getPriceByGoalId();

        return mapList(
                averageCostPerActionWithCategoryCpaSource.keySet(),
                goalId -> toGdRecommendedGoalCostPerActionForCategory(goalId,
                        averageCostPerActionWithCategoryCpaSource));
    }

    private Pair<LocalDate, LocalDate> getBoundaryDatesForTodayMinusDays(int numOfDays) {
        LocalDate endDate = LocalDate.now();
        LocalDate startDate = endDate.minusDays(numOfDays);
        return Pair.of(startDate, endDate);
    }

    /**
     * Выполнить запрос в метрику и БК за информацией о конверсиях и проставить данные в переданный список целей.
     */
    public void receiveAndAddConversionVisitsToGoals(Long operatorUid,
                                                     ClientId clientId,
                                                     Set<Goal> goals,
                                                     boolean isUnavailableGoalsAllowed) {
        if (goals.isEmpty()) {
            return;
        }

        Map<Long, GoalConversionVisit> goalsConversionVisitsCount;
        if (isUnavailableGoalsAllowed) {
            goalsConversionVisitsCount = getGoalsConversionVisitsCountForAllGoals(clientId, goals);
        } else {
            Set<Long> counterIds = FunctionalUtils.filterAndMapToSet(goals, goal -> goal.getCounterId() != null,
                    goal -> goal.getCounterId().longValue());
            goalsConversionVisitsCount = getGoalsConversionVisitsCountForAllGoals(operatorUid, clientId, counterIds);
        }

        Set<Long> mobileGoalIds = FunctionalUtils.filterAndMapToSet(
                goals, goal -> goal.getType() == GoalType.MOBILE, GoalBase::getId);
        var mobileGoalsConversionVisitsCount = getMobileGoalsConversions(mobileGoalIds);

        var metrikaGoalsConversionVisitsCount = EntryStream.of(goalsConversionVisitsCount)
                .removeKeys(mobileGoalIds::contains)
                .append(mobileGoalsConversionVisitsCount)
                .toMap();

        populateGoalsByConversionVisits(goals, metrikaGoalsConversionVisitsCount);
    }

    /**
     * Получить конверсии по мобильным целям (новым, не фиксированным)
     *
     * @param mobileGoalIds коллекция идентификаторов мобильных целей для получения конверсий (метод не проверяет
     *                      "мобильность")
     * @return информация о конверсиях по номеру цели
     */
    public Map<Long, GoalConversionVisit> getMobileGoalsConversions(Collection<Long> mobileGoalIds) {
        List<MobileGoalConversions> eventsStats = mobileGoalsStatisticRepository.getEventsStats(
                mobileGoalIds, DAYS_WITH_CONVERSION_VISITS);
        Map<Long, MobileGoalConversions> goalConversions = listToMap(eventsStats, MobileGoalConversions::getGoalId);

        return StreamEx.of(mobileGoalIds)
                .mapToEntry(goalConversions::get)
                .mapValues(conversions -> ifNotNullOrDefault(
                        conversions, MobileGoalConversions::getAttributedConversions, 0L))
                .mapValues(conversionCount -> new GoalConversionVisit()
                        .withCount(conversionCount)
                        .withHasPrice(Boolean.FALSE))
                .toMap();
    }

    /**
     * Проставляет переданноому списку целей значения по конверсиям и наличию дохода,
     * которые мы получили из метрики или БК. Если данные не найдены, то в цель записывается null.
     *
     * @param goals                  - список целей
     * @param goalConversionDataById - данные со статистикой
     */
    public void populateGoalsByConversionVisits(Collection<Goal> goals,
                                                @Nullable Map<Long, GoalConversionVisit> goalConversionDataById) {
        if (CollectionUtils.isEmpty(goalConversionDataById) || CollectionUtils.isEmpty(goals)) {
            return;
        }

        goals.forEach(goal -> {
            var goalStatistics = goalConversionDataById.getOrDefault(goal.getId(), new GoalConversionVisit());
            goal.setConversionVisitsCount(goalStatistics.getCount());
            goal.setHasRevenue(goalStatistics.getHasPrice());
        });
    }

    private GdCountersWithGoals toGdCountersWithGoals(Long operatorUid,
                                                      ClientId clientId,
                                                      Map<Long, Set<Goal>> availableGoalsByCounterId) {
        var availableGoals = StreamEx.of(availableGoalsByCounterId.values())
                .flatCollection(Function.identity())
                .toSet();

        // Добавляем конверсии для целей
        receiveAndAddConversionVisitsToGoals(operatorUid, clientId, availableGoals, true);

        return new GdCountersWithGoals()
                .withCounterWithGoals(toGdCounterWithGoals(availableGoalsByCounterId))
                .withIsMetrikaAvailable(true);
    }
}
