package ru.yandex.direct.grid.processing.service.goal;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.crypta.service.CryptaSuggestService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.goal.GdAppmetrikaApplication;
import ru.yandex.direct.grid.processing.model.goal.GdAppmetrikaApplicationsPayload;
import ru.yandex.direct.grid.processing.model.goal.GdAvailableGoalsContainer;
import ru.yandex.direct.grid.processing.model.goal.GdAvailableGoalsContext;
import ru.yandex.direct.grid.processing.model.goal.GdBroadMatchGoals;
import ru.yandex.direct.grid.processing.model.goal.GdBroadMatchesContainer;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignGoalsContainer;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignGoalsRecommendedCostPerActionForNewCampaign;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignGoalsRecommendedCostPerActionInputItem;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignsGoals;
import ru.yandex.direct.grid.processing.model.goal.GdCampaignsGoalsRecommendedCostPerActionInput;
import ru.yandex.direct.grid.processing.model.goal.GdCountersWithGoals;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.goal.GdGoalFilter;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsContainer;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsContext;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsConversionVisitsCount;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsRecommendedCostPerActionByCampaignId;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsRecommendedCostPerActionByCampaignIdPayload;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsRecommendedCostPerActionForNewCampaignPayload;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsSuggestContext;
import ru.yandex.direct.grid.processing.model.goal.GdGoalsSuggestFilter;
import ru.yandex.direct.grid.processing.model.goal.GdMeaningfulGoalsContainer;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaGoalsFilterUnion;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaSegmentPresets;
import ru.yandex.direct.grid.processing.model.goal.GdMobileEvent;
import ru.yandex.direct.grid.processing.model.goal.GdMobileEventsPayload;
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalConversions;
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalConversionsReq;
import ru.yandex.direct.grid.processing.model.goal.GdRecommendedGoalCostPerAction;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaGoalsByCounter;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaGoalsByCounterPayload;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoalsContext;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Сервис, возвращающий данные о целях
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class GoalGraphQlService {
    public static final String CAMPAIGN_GOALS_RESOLVER_NAME = "campaignGoals";

    private final GoalDataService goalDataService;
    private final GridShortenerService gridShortenerService;
    private final GoalMutationService goalMutationService;
    private final MobileGoalConversionsDataService mobileGoalConversionsDataService;
    private final GoalUsageParamsDataLoader goalUsageParamsDataLoader;
    private final AppmetrikaService appmetrikaService;
    private final CryptaSuggestService cryptaSuggestService;
    private final FeatureService featureService;
    private final CryptaGoalsConverter cryptaGoalsConverter;

    @Autowired
    public GoalGraphQlService(GoalDataService goalDataService, GridShortenerService gridShortenerService,
                              GoalMutationService goalMutationService,
                              MobileGoalConversionsDataService mobileGoalConversionsDataService,
                              GoalUsageParamsDataLoader goalUsageParamsDataLoader,
                              AppmetrikaService appmetrikaService, CryptaSuggestService cryptaSuggestService,
                              FeatureService featureService, CryptaGoalsConverter cryptaGoalsConverter) {
        this.goalDataService = goalDataService;
        this.gridShortenerService = gridShortenerService;
        this.goalMutationService = goalMutationService;
        this.mobileGoalConversionsDataService = mobileGoalConversionsDataService;
        this.goalUsageParamsDataLoader = goalUsageParamsDataLoader;
        this.appmetrikaService = appmetrikaService;
        this.cryptaSuggestService = cryptaSuggestService;
        this.featureService = featureService;
        this.cryptaGoalsConverter = cryptaGoalsConverter;
    }

    /**
     * GraphQL подзапрос. Получает информацию об привязанных целях клиента
     *
     * @param context       контекст
     * @param clientContext клиент, к которому относятся рекомендации
     * @param input         параметры фильтра и вывода
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAMPAIGN_GOALS_RESOLVER_NAME)
    public GdGoalsContext getCampaignGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignGoalsContainer input) {
        GdClientInfo client = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(client.getId());
        Long operatorUid = context.getOperator().getUid();

        return toGdGoalsContext(goalDataService.getCampaignsGoals(operatorUid, clientId, input.getFilter(),
                context.getFetchedFieldsReslover().getCampaignGoal().getDomain()));
    }


    /**
     * Получение списка всех целей ретаргетинга на клиента
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "goals")
    public GdGoalsContext getGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient clientContext,
            @GraphQLArgument(name = "input") GdGoalsContainer input) {
        /*
        Проверить, какие типы целей нужно отдавать фронту.
        Сделать фильтр по типу целей
        */

        GdClientInfo client = context.getQueriedClient();
        ClientId clientId = ClientId.fromLong(client.getId());

        if (input.getFilterKey() != null) {
            GdGoalFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    clientId,
                    GdGoalFilter.class,
                    () -> new GdGoalFilter().withCounterIdsIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdGoalsContext gdGoalsContext = toGdGoalsContext(goalDataService.getClientGoals(clientId, input.getFilter()));
        gdGoalsContext.setFilter(input.getFilter());

        return gdGoalsContext;
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "retargetingGoalsSuggest")
    public GdGoalsSuggestContext getGoalsSuggest(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdGoalsSuggestFilter input
    ) {
        var suggest = cryptaSuggestService.getRetargetingGoalsSuggest(input.getText());
        var translatedSuggest = cryptaGoalsConverter.convertToTranslatedGdGoal(suggest);

        return new GdGoalsSuggestContext()
                .withRowset(translatedSuggest);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "meaningfulGoals")
    public GdCampaignsGoals getMeaningfulGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMeaningfulGoalsContainer input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        Long operatorUid = context.getOperator().getUid();

        return goalDataService.getGdMeaningfulGoals(operatorUid, clientId, input.getMeaningfulGoalContainers());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "availableGoals")
    public GdAvailableGoalsContext getAvailableGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "input") GdAvailableGoalsContainer input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        Long operatorUid = context.getOperator().getUid();
        return goalDataService.getAvailableGoals(operatorUid, clientId, input);
    }

    /**
     * Получение доступных целей по счетчикам
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "goalsByCounterIds")
    public GdCountersWithGoals getGoalsByCounterIds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") Set<@GraphQLNonNull Long> counterIdsIn) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        Long operatorUid = context.getOperator().getUid();

        return goalDataService.getGdGoalsByCounterIds(operatorUid, counterIdsIn, clientId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "broadMatchGoals")
    public GdBroadMatchGoals getBroadMatchGoals(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdBroadMatchesContainer input) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        Long operatorUid = context.getOperator().getUid();

        return goalDataService.getGdBroadMatches(input.getBroadMatchContainers(), operatorUid, clientId);
    }

    @GraphQLQuery(name = "abSegments")
    public GdCampaignsGoals getAbSegments(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMeaningfulGoalsContainer input) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        return goalDataService.getAbSegments(input.getMeaningfulGoalContainers(), clientId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "abSegmentsByCounterIds")
    public GdCountersWithGoals getAbSegmentsByCounterIds(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") Set<@GraphQLNonNull Long> counterIdsIn) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());
        Long operatorUid = context.getOperator().getUid();

        return goalDataService.getAbSegmentsByCounterIds(operatorUid, counterIdsIn, clientId);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "segmentPresets")
    public GdMetrikaSegmentPresets getSegmentPresets(@GraphQLContext GdClient client) {
        GdClientInfo info = client.getInfo();
        ClientId clientId = ClientId.fromLong(info.getId());

        return goalDataService.getSegmentPresets(clientId);
    }

    /**
     * Получение целей по счетчикам, которые можно использовать на кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "getMetrikaGoalsByCounter")
    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByCounter(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetrikaGoalsByCounter input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Long operatorUid = context.getOperator().getUid();

        return goalMutationService.getMetrikaGoalsByCounterIds(operatorUid, clientId, input);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getMetrikaGoalsByFilter")
    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByFilter(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMetrikaGoalsFilterUnion input
    ) {
        ClientId clientId = context.getSubjectUser().getClientId();
        Long operatorUid = context.getOperator().getUid();

        return goalMutationService.getMetrikaGoalsByFilter(
                operatorUid,
                clientId,
                input);
    }


    @GraphQLNonNull
    @GraphQLQuery(name = "conversionVisitsCountForCountersGoals")
    public GdGoalsConversionVisitsCount getConversionVisitsCountForCountersGoals(
            @GraphQLContext GdClient client,
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") Set<@GraphQLNonNull Long> counterIdsIn
    ) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = context.getSubjectUser().getClientId();

        return goalDataService.getGdGoalsConversionVisitsCountForAllGoals(operatorUid, clientId, counterIdsIn);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "conversionsForMobileGoals")
    public GdMobileGoalConversions getConversionsForMobileGoals(
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdMobileGoalConversionsReq request
    ) {
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        var mobileGoalConversions = mobileGoalConversionsDataService.getMobileGoalConversions(
                clientId, request.getMobileAppId(), request.getGoalIds(), request.getDaysNum());
        return new GdMobileGoalConversions().withGoalConversions(mobileGoalConversions);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getRecommendedCampaignsGoalsCostPerAction")
    public GdGoalsRecommendedCostPerActionByCampaignIdPayload getRecommendedCampaignsGoalsCostPerAction(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCampaignsGoalsRecommendedCostPerActionInput input
    ) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());
        CurrencyCode clientCurrencyCode = client.getInfo().getWorkCurrency();

        Map<Long, List<Long>> goalIdsByCampaignId = listToMap(input.getItems(),
                GdCampaignGoalsRecommendedCostPerActionInputItem::getCampaignId,
                GdCampaignGoalsRecommendedCostPerActionInputItem::getGoalIds);

        Map<Long, String> urlByCampaignId = listToMap(input.getItems(),
                GdCampaignGoalsRecommendedCostPerActionInputItem::getCampaignId,
                GdCampaignGoalsRecommendedCostPerActionInputItem::getUrl);

        boolean unavailableGoalsEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED);
        List<GdGoalsRecommendedCostPerActionByCampaignId> result =
                goalDataService.getCampaignsGoalsCostPerActionRecommendations(
                        operatorUid, clientId, goalIdsByCampaignId,
                        clientCurrencyCode, urlByCampaignId, unavailableGoalsEnabled);

        return new GdGoalsRecommendedCostPerActionByCampaignIdPayload()
                .withRecommendedCampaignsGoalsCostPerAction(result);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "getRecommendedGoalsCostPerActionForNewCampaign")
    public GdGoalsRecommendedCostPerActionForNewCampaignPayload getRecommendedGoalsCostPerActionForNewCamp(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdClient client,
            @GraphQLNonNull @GraphQLArgument(name = "input")
                    GdCampaignGoalsRecommendedCostPerActionForNewCampaign input
    ) {
        Long operatorUid = context.getOperator().getUid();
        ClientId clientId = ClientId.fromLong(client.getInfo().getId());

        boolean unavailableGoalsEnabled = featureService.isEnabledForClientId(clientId,
                FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED);
        List<GdRecommendedGoalCostPerAction> result = goalDataService.getGoalsCostPerActionRecommendations(
                operatorUid, clientId, input.getGoalIds(), input.getUrl(), unavailableGoalsEnabled);

        return new GdGoalsRecommendedCostPerActionForNewCampaignPayload()
                .withRecommendedGoalsCostPerAction(result);
    }

    /**
     * GraphQL подзапрос. Возвращает список доступных приложений аппметрики.
     * Все аргументы, кроме client, опциональны и нужны для фильтрации
     *
     * @param client клиент, для которого выполняется запрос
     * @param url    url мобильного приложения
     * @param mask   подстрока, которая должна присутствовать в названиях
     * @param limit  ограничение на размер списка
     * @param offset размер сдвига
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "appmetrikaApplications")
    public List<@GraphQLNonNull GdAppmetrikaApplication> getAppmetrikaApplications(
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "url") @Nullable String url,
            @GraphQLArgument(name = "mask") @Nullable String mask,
            @GraphQLArgument(name = "limit") @Nullable Integer limit,
            @GraphQLArgument(name = "offset") @Nullable Integer offset) {
        return appmetrikaService.getApplications(client.getInfo().getChiefUserId(),
                ClientId.fromLong(client.getInfo().getId()), url, mask, limit, offset);
    }

    /**
     * GraphQL подзапрос. Возвращает объект, содержащий список доступных приложений аппметрики и флаг доступности
     * сервиса аппметрики. При недоступности сервиса список пуст.
     * Все аргументы, кроме client, опциональны и нужны для фильтрации
     *
     * @param client клиент, для которого выполняется запрос
     * @param url    url мобильного приложения
     * @param mask   подстрока, которая должна присутствовать в названиях
     * @param limit  ограничение на размер списка
     * @param offset размер сдвига
     */
    @GraphQLNonNull
    @GraphQLQuery(name = "appmetrikaApplicationsSafe")
    public GdAppmetrikaApplicationsPayload getAppmetrikaApplicationsSafe(
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "url") @Nullable String url,
            @GraphQLArgument(name = "mask") @Nullable String mask,
            @GraphQLArgument(name = "limit") @Nullable Integer limit,
            @GraphQLArgument(name = "offset") @Nullable Integer offset) {
        return appmetrikaService.getApplicationsSafe(client.getInfo().getChiefUserId(),
                ClientId.fromLong(client.getInfo().getId()), url, mask, limit, offset);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "mobileAppEvents")
    public List<@GraphQLNonNull GdMobileEvent> getMobileAppEvents(
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "metrikaAppId") Long metrikaAppId,
            @GraphQLArgument(name = "mask") @Nullable String mask,
            @GraphQLArgument(name = "limit") @Nullable Integer limit,
            @GraphQLArgument(name = "offset") @Nullable Integer offset) {
        return appmetrikaService.getEvents(client.getInfo().getChiefUserId(), metrikaAppId, mask, limit, offset);
    }

    @GraphQLNonNull
    @GraphQLQuery(name = "mobileAppEventsSafe")
    public GdMobileEventsPayload getMobileAppEventsSafe(
            @GraphQLContext GdClient client,
            @GraphQLArgument(name = "metrikaAppId") Long metrikaAppId,
            @GraphQLArgument(name = "mask") @Nullable String mask,
            @GraphQLArgument(name = "limit") @Nullable Integer limit,
            @GraphQLArgument(name = "offset") @Nullable Integer offset) {
        return appmetrikaService.getEventsSafe(client.getInfo().getChiefUserId(),
                ClientId.fromLong(client.getInfo().getId()), metrikaAppId, mask, limit, offset);
    }

    @GraphQLQuery(name = "isMeaningfulGoal")
    public CompletableFuture<Boolean> isMeaningfulGoal(
            @GraphQLContext GdGoal goal
    ) {
        return goalUsageParamsDataLoader.get().load(goal.getId())
                .thenApply(goalCampUsages -> goalCampUsages == null ? null : goalCampUsages.isMeaningful());
    }

    @GraphQLQuery(name = "isUsedInCampaignStrategy")
    public CompletableFuture<Boolean> isUsedInCampaignStrategy(
            @GraphQLContext GdGoal goal
    ) {
        return goalUsageParamsDataLoader.get().load(goal.getId())
                .thenApply(goalCampUsages -> goalCampUsages == null ? null : goalCampUsages.isUsedInCampaignStrategy());
    }
}
