package ru.yandex.direct.grid.processing.service.goal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLNonNull;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.campaign.service.CampMetrikaCountersService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.container.CreateCounterGoalContainer;
import ru.yandex.direct.core.entity.metrika.service.ForCampaignType;
import ru.yandex.direct.core.entity.metrika.service.ForStrategy;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaSegmentService;
import ru.yandex.direct.core.entity.metrika.service.MobileGoalsFilter;
import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaGoalsValidationService;
import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaSegmentValidationService;
import ru.yandex.direct.core.entity.metrikacounter.model.MetrikaCounterWithAdditionalInformation;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalStatus;
import ru.yandex.direct.core.entity.retargeting.model.GoalsSuggestion;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaSegmentPreset;
import ru.yandex.direct.core.entity.retargeting.model.PresetsByCounter;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.core.util.CoreHttpUtil;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.model.entity.campaign.converter.CampaignDataConverter;
import ru.yandex.direct.grid.processing.model.api.GdValidationResult;
import ru.yandex.direct.grid.processing.model.goal.GdGoal;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaGoalsFilterUnion;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoal;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoals;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalsPayload;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdLalSegments;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdLalSegmentsPayload;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaGoalsByCounter;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaGoalsByCounterPayload;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdMetrikaSegmentsByPresetsPayload;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdPresetsByCounter;
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.metrika.client.MetrikaClientException;
import ru.yandex.direct.metrika.client.model.response.GetExistentCountersResponseItem;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.amazonaws.util.CollectionUtils.isNullOrEmpty;
import static ru.yandex.direct.core.entity.metrika.service.validation.MetrikaCountersValidationService.validateCounterId;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.feature.FeatureName.CREATION_OF_METRIKA_SEGMENTS_BY_PRESETS_ENABLED;
import static ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValueType.ALL_VALUES;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toCreateCounterGoalContainer;
import static ru.yandex.direct.grid.processing.service.goal.GoalDataConverter.toGdGoal;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.validation.result.PathHelper.emptyPath;
import static ru.yandex.direct.validation.result.ValidationResult.failed;

@Service
@ParametersAreNonnullByDefault
public class GoalMutationService {
    private static final Logger logger = LoggerFactory.getLogger(GoalMutationService.class);

    private final MetrikaGoalsService metrikaGoalsService;
    private final MetrikaSegmentService metrikaSegmentService;
    private final MetrikaGoalsValidationService metrikaGoalsValidationService;
    private final MetrikaSegmentValidationService metrikaSegmentValidationService;
    private final RetargetingConditionService retargetingConditionService;
    private final GridValidationResultConversionService validationResultConverter;
    private final FeatureService featureService;
    private final TranslationService translationService;
    private final GoalMutationValidationService goalMutationValidationService;
    private final CampMetrikaCountersService campMetrikaCountersService;
    private final CampaignTypedRepository campaignTypedRepository;
    private final ShardHelper shardHelper;
    private final GoalDataService goalDataService;
    private final NetAcl netAcl;

    private static final GoalNameTranslations GOAL_NAME_TRANSLATIONS = GoalNameTranslations.INSTANCE;

    @Autowired
    public GoalMutationService(MetrikaGoalsService metrikaGoalsService,
                               MetrikaSegmentService metrikaSegmentService,
                               MetrikaGoalsValidationService metrikaGoalsValidationService,
                               MetrikaSegmentValidationService metrikaSegmentValidationService,
                               RetargetingConditionService retargetingConditionService,
                               GridValidationResultConversionService validationResultConverter,
                               FeatureService featureService, TranslationService translationService,
                               GoalMutationValidationService goalMutationValidationService,
                               CampMetrikaCountersService campMetrikaCountersService,
                               CampaignTypedRepository campaignTypedRepository, ShardHelper shardHelper,
                               GoalDataService goalDataService,
                               NetAcl netAcl) {
        this.metrikaGoalsService = metrikaGoalsService;
        this.metrikaSegmentService = metrikaSegmentService;
        this.metrikaGoalsValidationService = metrikaGoalsValidationService;
        this.metrikaSegmentValidationService = metrikaSegmentValidationService;
        this.retargetingConditionService = retargetingConditionService;
        this.validationResultConverter = validationResultConverter;
        this.featureService = featureService;
        this.translationService = translationService;
        this.goalMutationValidationService = goalMutationValidationService;
        this.campMetrikaCountersService = campMetrikaCountersService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.shardHelper = shardHelper;
        this.goalDataService = goalDataService;
        this.netAcl = netAcl;
    }

    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByCounterIds(Long operatorUid, ClientId clientId,
                                                                      GdMetrikaGoalsByCounter container) {
        List<@GraphQLNonNull Long> counterIds = container.getCounterIds();
        var campaignId = container.getCampaignId();
        var campaignType =
                Optional.ofNullable(container.getCampaignType()).map(CampaignDataConverter::toCampaignType);
        var mobileGoalFilter = campaignType.map(ForCampaignType::new);
        Map<Long, CampaignType> campaignTypeMap = Map.of();
        if (campaignId != null && campaignType.isPresent()) {
            campaignTypeMap = Map.of(campaignId, campaignType.get());
        }
        return getMetrikaGoalsByCounterIds(
                operatorUid,
                clientId,
                counterIds,
                campaignTypeMap,
                mobileGoalFilter.orElse(null));
    }

    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByFilter(Long operatorUid,
                                                                  ClientId clientId,
                                                                  GdMetrikaGoalsFilterUnion union) {
        var validationResult = goalMutationValidationService.validateMetrikaGoalsFilterUnion(union);
        if (validationResult.hasAnyErrors()) {
            var gdVr = validationResultConverter.buildGridValidationResult(validationResult);
            return new GdMetrikaGoalsByCounterPayload()
                    .withValidationResult(gdVr);
        }
        //Фильтр определен, проверили на уровне валидации
        if (union.getMetrikaGoalsByStrategyFilter() != null) {
            var filter = union.getMetrikaGoalsByStrategyFilter();
            var campaignTypes = getCampaignTypes(clientId, filter.getCampaignIds());
            List<Long> counterIds = filter.getCounterIds() != null ? filter.getCounterIds() : List.of();
            var mobileGoalFilter = campaignTypes.isEmpty() ? ForStrategy.INSTANCE : null;
            return getMetrikaGoalsByCounterIds(
                    operatorUid,
                    clientId,
                    counterIds,
                    campaignTypes,
                    mobileGoalFilter
            );
        } else {
            throw new IllegalArgumentException("Undefined goals filter");
        }
    }

    private Map<Long, CampaignType> getCampaignTypes(ClientId clientId, List<Long> campaignIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        if (!isNullOrEmpty(campaignIds)) {
            var campaignTypes = campaignTypedRepository.getSafely(shard, campaignIds, CommonCampaign.class);
            return listToMap(campaignTypes, CommonCampaign::getId, CommonCampaign::getType);
        } else {
            return Map.of();
        }
    }

    /**
     * Получить цели с доступных клиенту счетчиков, а также цели с недоступных счетчиков на кампаниях
     */
    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByAvailableAndCampaignCounters(
            Long operatorUid, ClientId clientId, List<Long> campaignIds) {
        Map<Long, List<Long>> existingCounterIdsByCampaignIds =
                campMetrikaCountersService.getCounterByCampaignIds(clientId, campaignIds);
        return getMetrikaGoalsByAvailableAndCampaignCounters(operatorUid, clientId, existingCounterIdsByCampaignIds);
    }

    /**
     * Получить цели с доступных клиенту счетчиков, а также цели с недоступных счетчиков на кампаниях
     */
    public GdMetrikaGoalsByCounterPayload getMetrikaGoalsByAvailableAndCampaignCounters(
            Long operatorUid, ClientId clientId, Map<Long, List<Long>> existingCounterIdsByCampaignIds) {
        var campaignIds = existingCounterIdsByCampaignIds.keySet();
        var availableCounterIds =
                campMetrikaCountersService.getAvailableCounterIdsFromCampaignIdsForGoals(clientId, campaignIds);
        List<Long> counterIds = StreamEx.ofValues(existingCounterIdsByCampaignIds)
                .flatMap(StreamEx::of)
                .append(availableCounterIds)
                .distinct()
                .toList();
        return getMetrikaGoalsByCounterIds(operatorUid, clientId, counterIds, Map.of(), null);
    }

    private GdMetrikaGoalsByCounterPayload getMetrikaGoalsByCounterIds(
            Long operatorUid,
            ClientId clientId,
            List<Long> counterIds,
            Map<Long, CampaignType> campaignTypeMap,
            @Nullable MobileGoalsFilter mobileGoalsFilter
    ) {
        boolean requestFromInternalNetwork = Optional.ofNullable(CoreHttpUtil.getRemoteAddressFromAuthOrDefault())
                .map(netAcl::isInternalIp)
                .orElse(false);
        boolean ucBetaDisabled = featureService.isEnabledForClientId(clientId,
                FeatureName.UNIVERSAL_CAMPAIGNS_BETA_DISABLED);
        boolean isUnavailableGoalsAllowed = featureService.isEnabledForClientId(clientId,
                FeatureName.DIRECT_UNAVAILABLE_GOALS_ALLOWED);
        boolean returnAllGoals = !ucBetaDisabled && requestFromInternalNetwork || isUnavailableGoalsAllowed;

        ValidationResult<Void, Defect> vr =
                metrikaGoalsValidationService.validateMetrikaGoalsByCounters(operatorUid, campaignTypeMap.keySet());

        try {
            //Todo: Распилить на несколько резолверов и оптимизировать запросы в метрику DIRECT-113408
            if (vr.hasAnyErrors()) {
                GdValidationResult gdValidationResult = validationResultConverter.buildGridValidationResult(vr);
                return new GdMetrikaGoalsByCounterPayload()
                        .withValidationResult(gdValidationResult)
                        .withGoals(null)
                        .withTop1GoalId(null)
                        .withIsMetrikaAvailable(true);
            }

            Set<Long> validCounterIds;
            Set<Long> unavailableEcommerceCounterIds = Set.of();
            var availableCounterIds = metrikaGoalsValidationService.getAvailableCounterIds(clientId, counterIds);
            if (returnAllGoals) {
                Set<Long> unavailableCounterIds = StreamEx.of(counterIds)
                        .remove(availableCounterIds::contains)
                        .toSet();
                Set<GetExistentCountersResponseItem> allowedUnavailableCounters =
                        campMetrikaCountersService.getAllowedInaccessibleCounters(unavailableCounterIds);
                Set<Long> allowedUnavailableCounterIds =
                        mapSet(allowedUnavailableCounters, GetExistentCountersResponseItem::getCounterId);
                unavailableEcommerceCounterIds = filterAndMapToSet(allowedUnavailableCounters,
                        c -> nvl(c.getEcommerce(), false), GetExistentCountersResponseItem::getCounterId);
                validCounterIds = StreamEx.of(counterIds)
                        .remove(c -> unavailableCounterIds.contains(c) && !allowedUnavailableCounterIds.contains(c))
                        .toSet();
            } else {
                validCounterIds = StreamEx.of(counterIds)
                        .remove(c -> !availableCounterIds.contains(c))
                        .toSet();
            }
            var availableCounterValidationResult = getAvailableCounterValidationResult(counterIds, validCounterIds);

            boolean removeUnavailableGoals = !returnAllGoals && featureService.isEnabledForClientId(clientId,
                    FeatureName.CAMPAIGN_MODIFICATION_GOALS_VALIDATION_TIGHTENING);
            Set<Goal> goals = doGetMetrikaGoalsByCounterIds(
                    operatorUid,
                    clientId,
                    validCounterIds,
                    unavailableEcommerceCounterIds,
                    campaignTypeMap,
                    mobileGoalsFilter,
                    removeUnavailableGoals
            );
            goalDataService.receiveAndAddConversionVisitsToGoals(
                    operatorUid, clientId, goals, isUnavailableGoalsAllowed);

            GoalsSuggestion goalsSuggestion = metrikaGoalsService.getGoalsSuggestion(clientId, goals);
            Set<GdGoal> gdGoals = goalDataService.toGdGoalsWithRemoveConversionsForUnavailable(
                    goalsSuggestion, availableCounterIds, isUnavailableGoalsAllowed);

            return new GdMetrikaGoalsByCounterPayload()
                    .withGoals(new ArrayList<>(gdGoals))
                    .withTop1GoalId(goalsSuggestion.getTop1GoalId())
                    .withIsMetrikaAvailable(true)
                    .withValidationResult(availableCounterValidationResult);

        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika counters for clientId: " + clientId, e);
            return new GdMetrikaGoalsByCounterPayload()
                    .withGoals(null)
                    .withTop1GoalId(null)
                    .withIsMetrikaAvailable(false);
        }
    }

    public Set<Goal> doGetMetrikaGoalsByCounterIds(Long operatorUid,
                                                   ClientId clientId,
                                                   Collection<Long> counterIds,
                                                   Set<Long> unavailableEcommerceCounterIds,
                                                   Map<Long, CampaignType> campaignTypeMap,
                                                   @Nullable MobileGoalsFilter mobileGoalsFilter,
                                                   boolean removeUnavailableGoals) {
        Set<Goal> metrikaGoals = metrikaGoalsService.getMetrikaGoalsByCounters(
                operatorUid,
                clientId,
                counterIds,
                unavailableEcommerceCounterIds,
                campaignTypeMap,
                mobileGoalsFilter,
                removeUnavailableGoals,
                true);
        metrikaGoals = removeDeletedGoals(metrikaGoals);

        return metrikaGoals;
    }

    public GdLalSegmentsPayload createLalSegments(ClientId clientId, GdLalSegments input) {
        var parentIds = input.getParentGoalIds();
        List<Goal> clientGoals;
        try {
            clientGoals = retargetingConditionService.getMetrikaGoalsForRetargeting(clientId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when querying for metrika for client: " + clientId, e);
            return new GdLalSegmentsPayload()
                    .withSegments(List.of())
                    .withIsMetrikaAvailable(false);
        }

        var validationResult = metrikaGoalsValidationService.validateGoalsForLalSegmentCreation(parentIds, clientGoals);
        if (validationResult.hasAnyErrors()) {
            GdValidationResult gdValidationResult =
                    validationResultConverter.buildGridValidationResult(validationResult);
            return new GdLalSegmentsPayload()
                    .withSegments(List.of())
                    .withValidationResult(gdValidationResult)
                    .withIsMetrikaAvailable(true);
        }

        var createdLalSegments = metrikaGoalsService.createLalSegments(parentIds);
        setLalNames(createdLalSegments, clientGoals);
        return new GdLalSegmentsPayload()
                .withSegments(mapList(createdLalSegments, GoalDataConverter::toGdLalSegment))
                .withIsMetrikaAvailable(true);
    }

    public GdCreateMetrikaGoalsPayload createMetrikaGoals(Long operatorUid, ClientId clientId,
                                                          GdCreateMetrikaGoals input) {
        var counterIds = StreamEx.of(input.getGoals())
                .map(GdCreateMetrikaGoal::getCounterId)
                .toSet();
        var availableForEditingCounters = campMetrikaCountersService
                .getAvailableForEditingCounters(operatorUid, counterIds);
        Set<Long> counterIdsAvailableForEditing = mapSet(availableForEditingCounters,
                MetrikaCounterWithAdditionalInformation::getId);
        var validationResult =
                goalMutationValidationService.getValidationResultForCreateGoalInput(
                        counterIdsAvailableForEditing, input);
        if (validationResult.hasAnyErrors()) {
            GdValidationResult gdValidationResult =
                    validationResultConverter.buildGridValidationResult(validationResult, emptyPath());
            return new GdCreateMetrikaGoalsPayload()
                    .withIsMetrikaAvailable(true)
                    .withGoals(Set.of())
                    .withValidationResult(gdValidationResult);
        }

        List<CreateCounterGoalContainer> counterGoalsToCreate = mapList(input.getGoals(),
                goal -> toCreateCounterGoalContainer(goal, defineNameOfCreatedGoal(goal)));

        if (!featureService.isEnabledForClientId(clientId, FeatureName.CREATE_METRIKA_GOALS_VIA_DIRECT_INTERFACE)) {
            return new GdCreateMetrikaGoalsPayload()
                    .withGoals(Set.of())
                    .withIsMetrikaAvailable(true);
        }

        try {
            List<CreateCounterGoalContainer> createdGoals =
                    metrikaGoalsService.createMetrikaGoals(counterGoalsToCreate);
            var metrikaCountersWithAdditionalInformationByCounterId = listToMap(availableForEditingCounters,
                    MetrikaCounterWithAdditionalInformation::getId, Function.identity());
            Set<GdGoal> goalsResult = listToSet(createdGoals, goalContainer -> toGdGoal(goalContainer,
                    metrikaCountersWithAdditionalInformationByCounterId));

            return new GdCreateMetrikaGoalsPayload()
                    .withGoals(goalsResult)
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            logger.warn("Got an exception when creating metrika goal for client: " + clientId, e);
            return new GdCreateMetrikaGoalsPayload()
                    .withIsMetrikaAvailable(false)
                    .withGoals(Set.of());
        }
    }

    public String defineNameOfCreatedGoal(GdCreateMetrikaGoal createMetrikaGoalInput) {
        boolean isGoalForAllValues = createMetrikaGoalInput.getGoalValue().getValueType().equals(ALL_VALUES);
        var value = createMetrikaGoalInput.getGoalValue().getValue();
        switch (createMetrikaGoalInput.getType()) {
            case EMAIL:
                return isGoalForAllValues ?
                        translate(GOAL_NAME_TRANSLATIONS.clickOnAnyEmail()) :
                        translate(GOAL_NAME_TRANSLATIONS.clickOnSpecifiedEmail(value));
            case PHONE:
                return isGoalForAllValues ?
                        translate(GOAL_NAME_TRANSLATIONS.clickOnAnyPhone()) :
                        translate(GOAL_NAME_TRANSLATIONS.clickOnSpecifiedPhone(value));
            case FORM:
                return translate(GOAL_NAME_TRANSLATIONS.submitAnyForm());
        }
        return null;
    }

    private String translate(Translatable t) {
        return translationService.translate(t);
    }

    private void setLalNames(List<Goal> lals, List<Goal> allClientGoals) {
        var goalsByIds = listToMap(allClientGoals, Goal::getId);
        lals.forEach(lal -> lal.setName(goalsByIds.get(lal.getParentId()).getName()));
    }

    private Set<Goal> removeDeletedGoals(Collection<Goal> goals) {
        return goals.stream()
                .filter(goal -> goal.getStatus() != GoalStatus.DELETED)
                .collect(Collectors.toSet());
    }

    public GdMetrikaSegmentsByPresetsPayload createMetrikaSegmentByPreset(
            ClientId clientId, List<GdPresetsByCounter> items) {
        if (!featureService.isEnabledForClientId(clientId, CREATION_OF_METRIKA_SEGMENTS_BY_PRESETS_ENABLED)) {
            var vr = validationResultConverter.buildGridValidationResult(failed(items, noRights()));
            return new GdMetrikaSegmentsByPresetsPayload()
                    .withValidationResult(vr)
                    .withSegments(List.of())
                    .withIsMetrikaAvailable(true);
        }

        var corePresetsByCounter = mapList(items, GoalDataConverter::toCorePresetsByCounter);

        List<MetrikaSegmentPreset> availablePresets;
        try {
            availablePresets = metrikaSegmentService.getSegmentPresets(clientId);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return unavailableMetrikaResponse(
                    "Got an exception when querying for metrika for client: " + clientId, e);
        }

        var vr = metrikaSegmentValidationService.validateSegmentCreationParameters(corePresetsByCounter,
                availablePresets);
        if (vr.hasAnyErrors()) {
            GdValidationResult gdValidationResult = validationResultConverter.buildGridValidationResult(vr);
            return new GdMetrikaSegmentsByPresetsPayload()
                    .withValidationResult(gdValidationResult)
                    .withSegments(List.of())
                    .withIsMetrikaAvailable(true);
        }

        try {
            var metrikaCountersByCounterId = StreamEx.of(availablePresets)
                    .mapToEntry(MetrikaSegmentPreset::getCounterId, Function.identity())
                    .grouping();
            var metrikaPresetByCounter = listToMap(corePresetsByCounter,
                    PresetsByCounter::getCounterId,
                    corePreset -> {
                        var metrikaPresets = metrikaCountersByCounterId.get(corePreset.getCounterId());
                        var neededIds = corePreset.getPresetIds();
                        return filterList(metrikaPresets,
                                metrikaPreset -> neededIds.contains(metrikaPreset.getPresetId()));
                    });
            var createdSegments = metrikaSegmentService.createMetrikaSegmentsByPresets(metrikaPresetByCounter);
            return new GdMetrikaSegmentsByPresetsPayload()
                    .withSegments(mapList(createdSegments, GoalDataConverter::toGdSegment))
                    .withIsMetrikaAvailable(true);
        } catch (MetrikaClientException | InterruptedRuntimeException e) {
            return unavailableMetrikaResponse(
                    "Got an exception when creating metrika segments for client: " + clientId, e);
        }
    }

    private static GdMetrikaSegmentsByPresetsPayload unavailableMetrikaResponse(String logMessage, Exception cause) {
        logger.warn(logMessage, cause);
        return new GdMetrikaSegmentsByPresetsPayload()
                .withSegments(List.of())
                .withIsMetrikaAvailable(false);
    }

    @Nullable
    private GdValidationResult getAvailableCounterValidationResult(List<Long> counterIds, Set<Long> validCounterIds) {
        if (counterIds.size() == validCounterIds.size()) {
            return null;
        }
        ItemValidationBuilder<Void, Defect> vb = ItemValidationBuilder.of(null);
        vb.list(counterIds, "counterIds")
                .checkEachBy(counterId -> validateCounterId(counterId, validCounterIds));
        return validationResultConverter.buildGridValidationResult(vb.getResult());
    }
}
