package ru.yandex.direct.grid.processing.service.goal;

import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.metrika.service.validation.MetrikaCountersValidationService;
import ru.yandex.direct.grid.processing.model.goal.GdMetrikaGoalsFilterUnion;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoal;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalType;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValue;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValueType;
import ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoals;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValueType.ALL_VALUES;
import static ru.yandex.direct.grid.processing.model.goal.mutation.GdCreateMetrikaGoalValueType.SPECIFIED_VALUE;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.listSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.PhoneNumberConstraints.validPhoneNumber;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxElementsExceeded;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
@ParametersAreNonnullByDefault
public class GoalMutationValidationService {

    private static final int MAX_GOALS_TO_CREATE = 1;

    private static final Map<GdCreateMetrikaGoalType, Set<GdCreateMetrikaGoalValueType>> VALUE_TYPES_ALLOWED_FOR_GOAL_TYPE = Map.of(
            GdCreateMetrikaGoalType.PHONE, Set.of(SPECIFIED_VALUE, ALL_VALUES),
            GdCreateMetrikaGoalType.EMAIL, Set.of(SPECIFIED_VALUE, ALL_VALUES),
            GdCreateMetrikaGoalType.FORM, Set.of(ALL_VALUES)
    );

    public GoalMutationValidationService() {
    }

    public ValidationResult<GdCreateMetrikaGoals, Defect> getValidationResultForCreateGoalInput(Set<Long> counterIdsAvailableForEditing, GdCreateMetrikaGoals input) {
        ModelItemValidationBuilder<GdCreateMetrikaGoals> vr = ModelItemValidationBuilder.of(input);

        vr.item(GdCreateMetrikaGoals.GOALS)
                .check(notEmptyCollection(), When.notNull())
                .check(listSize(MAX_GOALS_TO_CREATE, MAX_GOALS_TO_CREATE)); //todo

        vr.list(input.getGoals(), "goals")
                .check(fromPredicate(list -> list.size() == MAX_GOALS_TO_CREATE,
                        maxElementsExceeded(MAX_GOALS_TO_CREATE)))
                .checkEachBy(validateCreateGoal(counterIdsAvailableForEditing));

        return vr.getResult();
    }

    public ValidationResult<GdMetrikaGoalsFilterUnion, Defect> validateMetrikaGoalsFilterUnion(GdMetrikaGoalsFilterUnion union) {
        var validator = metrikaGoalsFilterUnionValidator();
        return validator.apply(union);
    }

    private static Validator<GdMetrikaGoalsFilterUnion, Defect> metrikaGoalsFilterUnionValidator() {
        return union -> {
            ModelItemValidationBuilder<GdMetrikaGoalsFilterUnion> vb = ModelItemValidationBuilder.of(union);
            vb.item(GdMetrikaGoalsFilterUnion.METRIKA_GOALS_BY_STRATEGY_FILTER)
                    .check(notNull());
            return vb.getResult();
        };
    }

    private static final Validator<GdCreateMetrikaGoal, Defect> validateCreateGoal(Set<Long> counterIdsAvailableForEditing) {
        return goal -> {
            ModelItemValidationBuilder<GdCreateMetrikaGoal> vb = ModelItemValidationBuilder.of(goal);

            vb.item(GdCreateMetrikaGoal.COUNTER_ID)
                    .checkBy(counterId -> MetrikaCountersValidationService.validateCounterId(counterId,
                            counterIdsAvailableForEditing));

            vb.item(GdCreateMetrikaGoal.TYPE)
                    .check(fromPredicate(VALUE_TYPES_ALLOWED_FOR_GOAL_TYPE::containsKey, invalidValue()));

            vb.item(GdCreateMetrikaGoal.GOAL_VALUE)
                    .checkBy(validateGoalValue(goal.getType()));

            return vb.getResult();
        };
    }

    private static Validator<GdCreateMetrikaGoalValue, Defect> validateGoalValue(GdCreateMetrikaGoalType goalType) {
        return goalValue -> {
            ModelItemValidationBuilder<GdCreateMetrikaGoalValue> vb = ModelItemValidationBuilder.of(goalValue);

            vb.item(GdCreateMetrikaGoalValue.VALUE_TYPE)
                    .check(Constraint.fromPredicate(type -> VALUE_TYPES_ALLOWED_FOR_GOAL_TYPE.get(goalType).contains(type), inconsistentState()),
                            When.isTrue(goalType.equals(GdCreateMetrikaGoalType.FORM)));

            boolean specifiedValueIsSet = goalValue.getValueType().equals(SPECIFIED_VALUE);
            vb.item(GdCreateMetrikaGoalValue.VALUE)
                    .check(notNull(), When.isTrue(specifiedValueIsSet))
                    .check(isNull(), When.isFalse(specifiedValueIsSet))
                    .check(validEmail(), When.isValidAnd(When.isTrue(goalType.equals(GdCreateMetrikaGoalType.EMAIL))))
                    .check(validPhoneNumber(),
                            When.isValidAnd(When.isTrue(goalType.equals(GdCreateMetrikaGoalType.PHONE))));

            return vb.getResult();
        };
    }
}
