package ru.yandex.direct.grid.processing.service.goal

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Scope
import org.springframework.context.annotation.ScopedProxyMode
import org.springframework.stereotype.Component
import org.springframework.web.context.WebApplicationContext
import ru.yandex.direct.core.entity.metrika.model.GoalCampUsages
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.service.dataloader.GridBatchingDataLoader
import ru.yandex.direct.grid.processing.service.dataloader.GridContextProvider
import java.util.concurrent.CompletableFuture


@Component
// DataLoader'ы хранят состояние, поэтому жить должны в рамках запроса
@Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
class GoalUsageParamsDataLoader(
    gridContextProvider: GridContextProvider,
    metrikaGoalService: MetrikaGoalsService,
) : GridBatchingDataLoader<Long, GoalCampUsages>() {

    companion object {
        private val LOGGER: Logger = LoggerFactory.getLogger(GoalUsageParamsDataLoader::class.java)
    }

    init {
        val unusedGoalParams = GoalCampUsages(isMeaningful = false, isUsedInCampaignStrategy = false)
        dataLoader = mappedDataLoader(gridContextProvider) { goalIds: Set<Long>, environment ->
            val context = environment.getContext<GridGraphQLContext>()
            val queriedClient = context.queriedClient
            if (queriedClient == null) {
                LOGGER.info("Omit goals params calculation. Return nulls")
                // резолвер вызвался вне контекста клиента -- возвращаем null'ы
                return@mappedDataLoader CompletableFuture.completedFuture(goalIds.associateWith { null })
            }
            val clientId = ClientId.fromLong(queriedClient.id)
            val goals = metrikaGoalService.getGoalsUsedInCampaignsByClientId(clientId)
            val result = CompletableFuture.completedFuture(goalIds.associateWith { goals[it] ?: unusedGoalParams })
            return@mappedDataLoader result
        }
    }
}
