package ru.yandex.direct.grid.processing.service.goal

import io.leangen.graphql.annotations.GraphQLArgument
import io.leangen.graphql.annotations.GraphQLContext
import io.leangen.graphql.annotations.GraphQLMutation
import io.leangen.graphql.annotations.GraphQLQuery
import io.leangen.graphql.annotations.GraphQLRootContext
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext
import ru.yandex.direct.grid.processing.model.client.GdClient
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalSharingMainRep
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalSharingPayload
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalSharingReq
import ru.yandex.direct.grid.processing.service.goal.validation.MobileGoalSharingValidationService
import ru.yandex.direct.grid.processing.service.goal.validation.MobileGoalSharingValidationService.MutationInputType
import ru.yandex.direct.grid.processing.service.validation.GridValidationResultConversionService
import ru.yandex.direct.rbac.RbacClientsRelations
import ru.yandex.direct.utils.PassportUtils.normalizeLogin

/**
 * Сервис для поддержки сценариев с мобильными in-app целями
 */
@GridGraphQLService
open class MobileGoalGraphQlService(
    private val validationResultConverter: GridValidationResultConversionService,
    private val rbacClientsRelations: RbacClientsRelations,
    private val userService: UserService,
    private val validationService: MobileGoalSharingValidationService,
) {
    /**
     * Кому предоставлен доступ к мобильным in-app целям текущего клиента.
     * Возвращает логины главных представителей клиентов, у кого сейчас имеется доступ.
     */
    @GraphQLQuery(name = "mobileGoalsSharing")
    open fun getMobileSharingInfo(@GraphQLContext client: GdClient): List<GdMobileGoalSharingMainRep> {
        val clientId = ClientId.fromLong(client.info.id)
        val consumersOfMobileGoals = rbacClientsRelations.getConsumersOfMobileGoals(clientId)
        return userService.getChiefsLoginsByClientIds(consumersOfMobileGoals).values.map {
            GdMobileGoalSharingMainRep(it)
        }
    }

    /**
     * Предоставить доступ нескольких клиентов (идентифицируемых главными представителями) к мобильным
     * in-app целям текущего клиента
     */
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addMobileGoalSharing")
    open fun addMobileGoalSharing(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdMobileGoalSharingReq
    ): GdMobileGoalSharingPayload {
        val clientId = context.subjectUser?.clientId
            ?: throw IllegalStateException("No subjectUser in graphql context")

        val vr = validationService.validateMutationInput(clientId, input, MutationInputType.ADD)
        if (vr.hasAnyErrors()) {
            return GdMobileGoalSharingPayload().apply {
                validationResult = validationResultConverter.buildGridValidationResult(vr)
            }
        }
        val loginsFromRequest = input.mainReps.map { normalizeLogin(it.login) }
        val consumersFromRequest = userService.massGetUserByLogin(loginsFromRequest).map { it.clientId }.toSet()

        // добавляем доступ, только если доступа ещё нет
        val consumersWithAccess = rbacClientsRelations.getConsumersOfMobileGoals(clientId).toSet()
        val consumersToAddRelation: Set<ClientId> = consumersFromRequest - consumersWithAccess

        rbacClientsRelations.addMobileGoalsAccessRelations(consumersToAddRelation, clientId)
        return GdMobileGoalSharingPayload()
    }

    /**
     * Отозвать доступ нескольких клиентов (идентифицируемых главными представителями) к мобильным
     * in-app целям текущего клиента
     */
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "removeMobileGoalSharing")
    open fun removeMobileGoalSharing(
        @GraphQLRootContext context: GridGraphQLContext,
        @GraphQLArgument(name = "input") input: GdMobileGoalSharingReq
    ): GdMobileGoalSharingPayload {
        val clientId = context.subjectUser?.clientId
            ?: throw IllegalStateException("No subjectUser in graphql context")

        val vr = validationService.validateMutationInput(clientId, input, MutationInputType.REMOVE)
        if (vr.hasAnyErrors()) {
            return GdMobileGoalSharingPayload().apply {
                validationResult = validationResultConverter.buildGridValidationResult(vr)
            }
        }
        val loginsFromRequest = input.mainReps.map { normalizeLogin(it.login) }
        val consumersFromRequest = userService.massGetUserByLogin(loginsFromRequest).map { it.clientId }.toSet()
        rbacClientsRelations.removeMobileGoalsAccessRelations(consumersFromRequest, clientId)
        return GdMobileGoalSharingPayload()
    }
}
