package ru.yandex.direct.grid.processing.service.goal.validation

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalSharingMainRep
import ru.yandex.direct.grid.processing.model.goal.GdMobileGoalSharingReq
import ru.yandex.direct.rbac.RbacClientsRelations
import ru.yandex.direct.utils.PassportUtils.normalizeLogin
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CollectionConstraints
import ru.yandex.direct.validation.defect.CollectionDefects
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.D
import ru.yandex.direct.validation.util.listProperty
import ru.yandex.direct.validation.util.validateObject

const val MAX_MOBILE_GOAL_CONSUMERS = 200

@Service
open class MobileGoalSharingValidationService(
    private val userService: UserService,
    private val rbacClientsRelations: RbacClientsRelations,
) {
    fun validateMutationInput(
        clientId: ClientId,
        input: GdMobileGoalSharingReq,
        type: MutationInputType
    ): ValidationResult<GdMobileGoalSharingReq, D> {
        val requestedLogins = input.mainReps.map { normalizeLogin(it.login) }
        val foundUsers = userService.massGetUserByLogin(requestedLogins).associateBy { normalizeLogin(it.login) }

        val consumerToOwners = rbacClientsRelations.getOwnersOfMobileGoals(foundUsers.values.map { it.clientId })

        val consumersOfMobileGoals = rbacClientsRelations.getConsumersOfMobileGoals(clientId)
        return validateMutationInput(clientId, input, foundUsers, consumersOfMobileGoals, consumerToOwners, type)
    }

    fun validateMutationInput(
        clientId: ClientId,
        input: GdMobileGoalSharingReq,
        foundUsers: Map<String, User>,
        consumersOfMobileGoals: List<ClientId>,
        consumerToOwners: Map<ClientId, List<ClientId>>,
        type: MutationInputType
    ): ValidationResult<GdMobileGoalSharingReq, D> {
        val allConsumers: Set<ClientId> = foundUsers.values.map { it.clientId }.toSet() + consumersOfMobileGoals

        val validationResult = validateObject(input) {
            listProperty(GdMobileGoalSharingReq::mainReps) {
                check(
                    CollectionConstraints.maxListSize<GdMobileGoalSharingMainRep>(MAX_MOBILE_GOAL_CONSUMERS)
                        .overrideDefect(CollectionDefects.maxElementsPerRequest(MAX_MOBILE_GOAL_CONSUMERS))
                )
                if (!result.hasAnyErrors()) {
                    checkEach(CollectionConstraints.unique())
                    checkEach(userWithLoginIsExist(foundUsers), When.isValid())
                    checkEach(userIsMainRep(foundUsers), When.isValid())
                    checkEach(noMoreThanOneSharing(clientId, foundUsers, consumerToOwners), When.isValid())

                    if (type == MutationInputType.ADD && allConsumers.size > MAX_MOBILE_GOAL_CONSUMERS) {
                        result.addError(CollectionDefects.maxElementsExceeded(MAX_MOBILE_GOAL_CONSUMERS))
                    }
                }
            }
        }
        return validationResult
    }

    private fun userWithLoginIsExist(foundUsers: Map<String, User>): Constraint<GdMobileGoalSharingMainRep, D> =
        Constraint.fromPredicate(
            { rep: GdMobileGoalSharingMainRep -> normalizeLogin(rep.login) in foundUsers },
            CommonDefects.objectNotFound()
        )

    private fun userIsMainRep(foundUsers: Map<String, User>): Constraint<GdMobileGoalSharingMainRep, D> =
        Constraint.fromPredicate({ rep: GdMobileGoalSharingMainRep ->
            val user = foundUsers[normalizeLogin(rep.login)] ?: return@fromPredicate true
            user.chiefUserId == user.uid
        }, CommonDefects.objectNotFound())

    private fun noMoreThanOneSharing(
        ownerId: ClientId,
        foundUsers: Map<String, User>,
        consumerToOwners: Map<ClientId, List<ClientId>>
    ): Constraint<GdMobileGoalSharingMainRep, D> {
        return Constraint.fromPredicate({ rep: GdMobileGoalSharingMainRep ->
            val consumerId = foundUsers[normalizeLogin(rep.login)]?.clientId ?: return@fromPredicate true
            // никто кроме текущего владельца приложений больше не шарит цели потребителю
            val owners = consumerToOwners.getOrDefault(consumerId, emptyList()).toSet() - ownerId
            owners.isEmpty()
        }, CommonDefects.invalidValue())
    }

    enum class MutationInputType {
        ADD,
        REMOVE
    }
}
