package ru.yandex.direct.grid.processing.service.group;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLArgument;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.leangen.graphql.annotations.GraphQLRootContext;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.model.GdStatRequirements;
import ru.yandex.direct.grid.model.campaign.GdCampaignTruncated;
import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.model.banner.GdAd;
import ru.yandex.direct.grid.processing.model.client.GdClient;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupAccess;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupFeatures;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupFilter;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupTruncated;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupWithTotals;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupsContainer;
import ru.yandex.direct.grid.processing.model.group.GdAdGroupsContext;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddAdGroupsMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddContentPromotionAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddDynamicAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddInternalAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMcBannerAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddMobileContentAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddSmartAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdAddTextAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdChangeAdGroupsRelevanceMatch;
import ru.yandex.direct.grid.processing.model.group.mutation.GdChangeAdGroupsRelevanceMatchPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdCopyAdGroupsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoderateAdsCallouts;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoderateAdsCalloutsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoveAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRemoveAdGroupsMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdRepalceAdGroupMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdReplaceAdGroupMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdReplaceAdGroupsMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdReplaceAdGroupsMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupMinusKeywords;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupMinusKeywordsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegions;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateAdGroupRegionsPayload;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateContentPromotionAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateCpmAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateDynamicAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateInternalAdGroups;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMcBannerAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateMobileContentAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdatePerformanceAdGroup;
import ru.yandex.direct.grid.processing.model.group.mutation.GdUpdateTextAdGroup;
import ru.yandex.direct.grid.processing.model.showcondition.GdAdGroupGetKeywordRecommendationInput;
import ru.yandex.direct.grid.processing.model.showcondition.GdKeywordsByCategory;
import ru.yandex.direct.grid.processing.service.cache.GridCacheService;
import ru.yandex.direct.grid.processing.service.group.container.GroupsCacheRecordInfo;
import ru.yandex.direct.grid.processing.service.group.loader.AdGroupsMainAdDataLoader;
import ru.yandex.direct.grid.processing.service.group.mutation.AddInternalAdGroupsMutationService;
import ru.yandex.direct.grid.processing.service.group.mutation.UpdateAdGroupRegionsMutationService;
import ru.yandex.direct.grid.processing.service.group.mutation.UpdateInternalAdGroupsMutationService;
import ru.yandex.direct.grid.processing.service.shortener.GridShortenerService;
import ru.yandex.direct.grid.processing.service.validation.GridValidationService;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacService;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isLimitedSupport;
import static ru.yandex.direct.feature.FeatureName.CPC_AND_CPM_ON_ONE_GRID_ENABLED;
import static ru.yandex.direct.grid.processing.service.cache.util.CacheUtils.normalizeLimitOffset;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter.toGdAdGroupsContext;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupDataConverter.toGroupsCacheRecordInfo;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.createEmptyGdAddAdGroupPayload;
import static ru.yandex.direct.grid.processing.service.group.converter.AdGroupsMutationDataConverter.createEmptyGdUpdateAdGroupPayload;
import static ru.yandex.direct.grid.processing.util.StatHelper.normalizeStatRequirements;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис, возвращающий данные о группах баннеров клиента
 */
@GridGraphQLService
@ParametersAreNonnullByDefault
public class AdGroupGraphQlService {
    public static final String AD_GROUPS_RESOLVER_NAME = "adGroups";
    static final String CAN_DELETED_AD_GROUP_RESOLVER_NAME = "canBeDeleted";
    static final String CAN_BE_DELETED_AD_GROUPS_COUNT_RESOLVER_NAME = "canBeDeletedAdGroupsCount";
    static final String CAN_REMODERATE_ADS_CALLOUTS_RESOLVER_NAME = "canRemoderateAdsCallouts";
    static final String CAN_REMODERATE_ADS_CALLOUTS_COUNT_RESOLVER_NAME = "canRemoderateAdsCalloutsCount";
    static final String CAN_ACCEPT_ADS_CALLOUTS_MODERATION_RESOLVER_NAME = "canAcceptAdsCalloutsModeration";
    static final String CAN_ACCEPT_ADS_CALLOUTS_MODERATION_COUNT_RESOLVER_NAME = "canAcceptAdsCalloutsModerationCount";
    static final String CAN_BE_SENT_TO_BS_RESOLVER_NAME = "canBeSentToBS";
    static final String CAN_BE_SENT_TO_BS_AD_GROUPS_COUNT_RESOLVER_NAME = "canBeSentToBSAdGroupsCount";
    static final String CAN_BE_SENT_TO_MODERATION_RESOLVER_NAME = "canBeSentToModeration";
    static final String CAN_BE_SENT_TO_MODERATION_AD_GROUPS_COUNT_RESOLVER_NAME = "canBeSentToModerationAdGroupsCount";
    static final String CAN_BE_SENT_TO_REMODERATION_RESOLVER_NAME = "canBeSentToRemoderation";
    static final String CAN_BE_SENT_TO_REMODERATION_AD_GROUPS_COUNT_RESOLVER_NAME =
            "canBeSentToRemoderationAdGroupsCount";
    static final String CAN_ACCEPT_MODERATION_RESOLVER_NAME = "canAcceptModeration";
    static final String CAN_ACCEPT_MODERATION_AD_GROUPS_COUNT_RESOLVER_NAME = "canAcceptModerationAdGroupsCount";
    static final String REMODERATE_ADS_CALLOUTS_MUTATION_NAME = "remoderateAdsCallouts";
    static final String COPY_AD_GROUPS_MUTATION_NAME = "copyAdGroups";
    static final String CHANGE_AD_GROUPS_RELEVANCE_MATCH_MUTATION_NAME = "changeAdGroupsRelevanceMatch";
    static final String UPDATE_AD_GROUP_REGIONS_MUTATION_NAME = "updateAdGroupRegions";
    static final String ADS_COUNT_RESOLVER_NAME = "adsCount";
    static final String KEYWORDS_COUNT_RESOLVER_NAME = "keywordsCount";
    static final String KEYWORDS_BY_CATEGORY_NAME = "keywordsByCategory";
    static final String MAIN_AD_RESOLVER_NAME = "mainAd";
    private static final String DEFAULT_REGION_IDS = "defaultRegionIds";
    private static final String GEO_SUGGEST = "geoSuggest";

    private final GridCacheService gridCacheService;
    private final GroupDataService groupDataService;
    private final GridValidationService gridValidationService;
    private final GridShortenerService gridShortenerService;
    private final AdGroupMutationService adGroupMutationService;
    private final AdGroupMassActionsMutationService adGroupMassActionsMutationService;
    private final AddInternalAdGroupsMutationService addInternalAdGroupsMutationService;
    private final UpdateInternalAdGroupsMutationService updateInternalAdGroupsMutationService;
    private final UpdateAdGroupRegionsMutationService updateAdGroupRegionsMutationService;
    private final AdGroupsMainAdDataLoader adGroupsMainAdDataLoader;
    private final RbacService rbacService;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public AdGroupGraphQlService(GridCacheService gridCacheService,
                                 GroupDataService groupDataService,
                                 GridShortenerService gridShortenerService,
                                 AdGroupMutationService adGroupMutationService,
                                 AdGroupMassActionsMutationService adGroupMassActionsMutationService,
                                 GridValidationService gridValidationService,
                                 AddInternalAdGroupsMutationService addInternalAdGroupsMutationService,
                                 UpdateInternalAdGroupsMutationService updateInternalAdGroupsMutationService,
                                 UpdateAdGroupRegionsMutationService updateAdGroupRegionsMutationService,
                                 AdGroupsMainAdDataLoader adGroupsMainAdDataLoader,
                                 FeatureService featureService,
                                 RbacService rbacService,
                                 ShardHelper shardHelper) {
        this.gridCacheService = gridCacheService;
        this.groupDataService = groupDataService;
        this.gridShortenerService = gridShortenerService;
        this.gridValidationService = gridValidationService;
        this.adGroupMutationService = adGroupMutationService;
        this.adGroupMassActionsMutationService = adGroupMassActionsMutationService;
        this.addInternalAdGroupsMutationService = addInternalAdGroupsMutationService;
        this.updateInternalAdGroupsMutationService = updateInternalAdGroupsMutationService;
        this.updateAdGroupRegionsMutationService = updateAdGroupRegionsMutationService;
        this.adGroupsMainAdDataLoader = adGroupsMainAdDataLoader;
        this.featureService = featureService;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
    }

    /**
     * GraphQL подзапрос. Получает информацию о группах объявлений клиента, полученного из контекста выполнения запроса
     */
    @GraphQLNonNull
    @GraphQLQuery(name = AD_GROUPS_RESOLVER_NAME)
    public GdAdGroupsContext getAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @SuppressWarnings("unused") @GraphQLContext GdClient gdClient,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAdGroupsContainer input) {
        gridValidationService.validateGdAdGroupsContainer(input);
        GdClientInfo client = context.getQueriedClient();

        if (input.getFilterKey() != null) {
            GdAdGroupFilter savedFilter = gridShortenerService.getSavedFilter(input.getFilterKey(),
                    ClientId.fromLong(client.getId()),
                    GdAdGroupFilter.class,
                    () -> new GdAdGroupFilter().withCampaignIdIn(emptySet()));
            input.setFilter(savedFilter);
        }

        GdStatRequirements statRequirements = normalizeStatRequirements(input.getStatRequirements(),
                context.getInstant(), input.getFilter().getRecommendations());
        input.setStatRequirements(statRequirements);

        LimitOffset range = normalizeLimitOffset(input.getLimitOffset());

        // пытаемся прочитать из кеша нужный диапазон строк
        GroupsCacheRecordInfo recordInfo = toGroupsCacheRecordInfo(client.getId(), input);
        Optional<GdAdGroupsContext> fromCache = gridCacheService.getFromCache(recordInfo, range);
        GdAdGroupsContext result = fromCache.orElseGet(() -> {
            boolean cpcAndCpmOnOneGridEnabled = featureService
                    .isEnabledForClientId(ClientId.fromLong(client.getId()), CPC_AND_CPM_ON_ONE_GRID_ENABLED);

            // в кеше данные не нашлись, читаем из mysql/YT
            GdAdGroupWithTotals adGroupWithTotals = groupDataService.getAdGroups(client, input, context);
            GdAdGroupsContext adGroupsContext =
                    toGdAdGroupsContext(adGroupWithTotals, input.getFilter(), cpcAndCpmOnOneGridEnabled);
            adGroupsContext.setFilter(input.getFilter());

            // сохраняем в кеш и возвращаем нужный диапазон строк в результате
            return gridCacheService.getResultAndSaveToCacheIfRequested(recordInfo, adGroupsContext,
                    adGroupWithTotals.getGdAdGroups(), range,
                    context.getFetchedFieldsReslover().getAdGroup().getCacheKey());
        });

        context.setGdAdGroups(result.getRowset());
        return result;
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности удаления группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_DELETED_AD_GROUP_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanBeDeletedAdGroup(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanBeDeletedAdGroup(gdAdGroupAccess);
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, которые можно удалить
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_DELETED_AD_GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanBeDeletedAdGroupsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        return groupDataService.getCanBeDeletedAdGroupsCount(gdAdGroupFeatures.getAdGroupAccesses());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности отправить группу в БК
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_BS_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanBeSentToBS(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanBeSentToBSAdGroup(gdAdGroupAccess);
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, которые можно отправить в БК
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_BS_AD_GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanBeSentToBSAdGroupsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        return groupDataService.getCanBeSentToBSAdGroupsCount(gdAdGroupFeatures.getAdGroupAccesses());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности отправить группу на Модерацию
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_MODERATION_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanBeSentToModeration(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanBeSentToModerationAdGroup(gdAdGroupAccess);
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, которые можно отправить на Модерацию
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_MODERATION_AD_GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanBeSentToModerationAdGroupsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        return groupDataService.getCanBeSentToModerationAdGroupsCount(gdAdGroupFeatures.getAdGroupAccesses());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности перемодерировать группу
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_REMODERATION_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanBeSentToRemoderation(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanBeSentToRemoderationAdGroup(gdAdGroupAccess);
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, которые можно перемодерировать
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_BE_SENT_TO_REMODERATION_AD_GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanBeSentToRemoderationAdGroupsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        return groupDataService.getCanBeSentToRemoderationAdGroupsCount(gdAdGroupFeatures.getAdGroupAccesses());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности принять на Модерации группу
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_ACCEPT_MODERATION_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanAcceptModeration(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanAcceptModerationAdGroup(gdAdGroupAccess);
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, которые можно принять на Модерации
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_ACCEPT_MODERATION_AD_GROUPS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanAcceptModerationAdGroupsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        return groupDataService.getCanAcceptModerationAdGroupsCount(gdAdGroupFeatures.getAdGroupAccesses());
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности перемодерации уточнений объявлений для группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_REMODERATE_ADS_CALLOUTS_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanRemoderateAdsCallouts(@GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanRemoderateAdsCallouts(gdAdGroupAccess.getAdGroupId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, у которых можно перемодерировать уточнения объявлений
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_REMODERATE_ADS_CALLOUTS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanRemoderateAdsCalloutsCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        List<Long> adGroupIds = mapList(gdAdGroupFeatures.getAdGroupAccesses(), GdAdGroupAccess::getAdGroupId);
        return groupDataService.getCanRemoderateAdsCalloutsCount(adGroupIds);
    }

    /**
     * GraphQL подзапрос. Получает информацию о возможности принять на модерации уточнений объявлений для группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_ACCEPT_ADS_CALLOUTS_MODERATION_RESOLVER_NAME)
    public CompletableFuture<Boolean> getCanAcceptAdsCalloutsModeration(
            @GraphQLContext GdAdGroupAccess gdAdGroupAccess) {
        return groupDataService.getCanAcceptAdsCalloutsModeration(gdAdGroupAccess.getAdGroupId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во групп, у которых можно принять на модерации уточнения объявлений
     */
    @GraphQLNonNull
    @GraphQLQuery(name = CAN_ACCEPT_ADS_CALLOUTS_MODERATION_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getCanAcceptAdsCalloutsModerationCount(
            @GraphQLContext GdAdGroupFeatures gdAdGroupFeatures) {
        List<Long> adGroupIds = mapList(gdAdGroupFeatures.getAdGroupAccesses(), GdAdGroupAccess::getAdGroupId);
        return groupDataService.getCanAcceptAdsCalloutsModerationCount(adGroupIds);
    }

    /**
     * GraphQL подзапрос. Получает кол-во объявлений для группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = ADS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getAdsCount(@GraphQLContext GdAdGroupTruncated gdAdGroupTruncated) {
        return groupDataService.getAdsCount(gdAdGroupTruncated.getId());
    }

    /**
     * GraphQL подзапрос. Получает кол-во ключевых фраз для группы
     */
    @GraphQLNonNull
    @GraphQLQuery(name = KEYWORDS_COUNT_RESOLVER_NAME)
    public CompletableFuture<Integer> getKeywordsCount(@GraphQLContext GdAdGroupTruncated gdAdGroupTruncated) {
        return groupDataService.getKeywordsCount(gdAdGroupTruncated.getCampaign().getId(), gdAdGroupTruncated.getId());
    }

    /**
     * GraphQL подзапрос. Генерирует ключевые фразы по категориям.
     */
    @GraphQLNonNull
    @GraphQLQuery(name = KEYWORDS_BY_CATEGORY_NAME)
    public GdKeywordsByCategory getKeywordsByCategory(@GraphQLRootContext GridGraphQLContext context,
                                                      @GraphQLArgument(name = "input") GdAdGroupGetKeywordRecommendationInput input) {
        var clientId = context.getSubjectUser().getClientId();
        return groupDataService.recommendedKeywordsByCategory(shardHelper.getShardByClientId(clientId), clientId, input);
    }

    /**
     * Резолвер для "главного баннера" группы
     */
    @GraphQLQuery(name = MAIN_AD_RESOLVER_NAME)
    public CompletableFuture<GdAd> getMainAd(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLContext GdAdGroupTruncated gdAdGroup) {
        // в GridGraphQLContext сейчас есть поле Map<Long, Banner> mainAdsByAdGroupId, но полагаться на него нельзя,
        // так как оно заполняется из resolver'а adGroups, который может не вызываться
        return adGroupsMainAdDataLoader.get().load(gdAdGroup);
    }

    /**
     * GraphQL подзапрос. Возвращает дефолтный регион для групп создаваемых в кампании
     */
    @GraphQLNonNull
    @GraphQLQuery(name = DEFAULT_REGION_IDS)
    public List<Long> getDefaultRegionIds(
            @GraphQLRootContext GridGraphQLContext context, @GraphQLContext GdCampaignTruncated campaignTruncated) {
        return groupDataService.getDefaultRegionIds(context, campaignTruncated.getId());
    }

    @GraphQLNonNull
    @GraphQLQuery(name = GEO_SUGGEST)
    public List<@GraphQLNonNull Long> getGeoSuggest(
            @GraphQLRootContext GridGraphQLContext context, @GraphQLContext GdClient client) {
        return groupDataService.getGeoSuggest();
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = UPDATE_AD_GROUP_REGIONS_MUTATION_NAME)
    public GdUpdateAdGroupRegionsPayload updateAdGroupRegions(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAdGroupRegions input) {
        //noinspection ConstantConditions
        return updateAdGroupRegionsMutationService.updateAdGroupRegions(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addAdGroupMinusKeywords")
    public GdAddAdGroupMinusKeywordsPayload addAdGroupMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddAdGroupMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addMinusKeywords(input, context.getOperator().getUid(), context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addAdGroupsMinusKeywords")
    public GdAddAdGroupsMinusKeywordsPayload addAdGroupsMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddAdGroupsMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .massAddMinusKeywords(input, context.getOperator().getUid(), context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "removeAdGroupsMinusKeywords")
    public GdRemoveAdGroupsMinusKeywordsPayload removeAdGroupsMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRemoveAdGroupsMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .massRemoveMinusKeywords(input, context.getOperator().getUid(), context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "replaceAdGroupsMinusKeywords")
    public GdReplaceAdGroupsMinusKeywordsPayload replaceAdGroupsMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdReplaceAdGroupsMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .massReplaceMinusKeywords(input, context.getOperator().getUid(),
                        context.getSubjectUser().getClientId());
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addInternalAdGroups")
    public GdAddAdGroupPayload addInternalAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddInternalAdGroups input) {
        if (input.getAddItems().isEmpty()) {
            return createEmptyGdAddAdGroupPayload();
        }

        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        Long clientUid = checkNotNull(context.getSubjectUser()).getUid();
        return addInternalAdGroupsMutationService.addInternalAdGroups(
                context.getOperator().getUid(), UidAndClientId.of(clientUid, clientId), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateInternalAdGroups")
    public GdUpdateAdGroupPayload updateInternalAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateInternalAdGroups input) {
        if (input.getUpdateItems().isEmpty()) {
            return createEmptyGdUpdateAdGroupPayload();
        }

        ClientId clientId = checkNotNull(context.getSubjectUser()).getClientId();
        Long clientUid = checkNotNull(context.getSubjectUser()).getUid();
        var owner = UidAndClientId.of(clientUid, clientId);
        return updateInternalAdGroupsMutationService.updateInternalAdGroups(
                owner, context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateTextAdGroup")
    public GdUpdateAdGroupPayload updateTextAdGroup(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateTextAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updateTextAdGroup(
                UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addTextAdGroups")
    public GdAddAdGroupPayload addTextAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddTextAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addTextAdGroups(context.getSubjectUser().getClientId(), context.getOperator().getUid(),
                        context.getSubjectUser().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addSmartAdGroups")
    public GdAddAdGroupPayload addPerformanceAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddSmartAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addPerformanceAdGroups(context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateSmartAdGroups")
    public GdUpdateAdGroupPayload updateSmartAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdatePerformanceAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updatePerformanceAdGroups(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addContentPromotionAdGroups")
    public GdAddAdGroupPayload addContentPromotionAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddContentPromotionAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addContentPromotionAdGroups(context.getSubjectUser(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateContentPromotionAdGroups")
    public GdUpdateAdGroupPayload updateContentPromotionAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateContentPromotionAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .updateContentPromotionAdGroups(
                        UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                        context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateAdGroupMinusKeywords")
    public GdUpdateAdGroupMinusKeywordsPayload updateAdGroupMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateAdGroupMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updateAdGroupMinusKeywords(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "replaceAdGroupMinusKeywords")
    public GdReplaceAdGroupMinusKeywordsPayload replaceAdGroupMinusKeywords(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRepalceAdGroupMinusKeywords input) {
        //noinspection ConstantConditions
        return adGroupMutationService.replaceAdGroupMinusKeywords(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = COPY_AD_GROUPS_MUTATION_NAME)
    public GdCopyAdGroupsPayload copyAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdCopyAdGroups input) {
        //noinspection ConstantConditions
        UidAndClientId uidAndClientId =
                UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId());
        return adGroupMassActionsMutationService.copyAdGroups(context.getOperator(), uidAndClientId, input);
    }

    @GraphQLNonNull
    @GraphQLMutation(name = REMODERATE_ADS_CALLOUTS_MUTATION_NAME)
    public GdRemoderateAdsCalloutsPayload remoderateAdsCallouts(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdRemoderateAdsCallouts input) {
        User operator = context.getOperator();
        User client = checkNotNull(context.getSubjectUser());
        checkAdsCalloutsRemoderationRights(operator, client);
        return adGroupMassActionsMutationService.remoderateAdsCallouts(operator, client.getClientId(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = CHANGE_AD_GROUPS_RELEVANCE_MATCH_MUTATION_NAME)
    public GdChangeAdGroupsRelevanceMatchPayload changeAdGroupsRelevanceMatch(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdChangeAdGroupsRelevanceMatch input) {
        //noinspection ConstantConditions
        ClientId clientId = context.getSubjectUser().getClientId();
        return adGroupMassActionsMutationService.changeAdGroupsRelevanceMatch(context.getOperator(), clientId, input);
    }

    /**
     * Из-за limited_support нельзя использовать просто @PreAuthorizeWrite
     */
    private void checkAdsCalloutsRemoderationRights(User operator, User client) {
        boolean hasRights = hasOneOfRoles(operator, SUPER, PLACER, SUPPORT)
                || (isLimitedSupport(operator) && rbacService.isOwner(operator.getUid(), client.getUid()));
        if (!hasRights) {
            throw new AccessDeniedException(
                    String.format("У оператора [uid=%s, role=%s] нет прав на выполнение операции",
                            operator.getUid(), operator.getRole()));
        }
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "saveCpmAdGroups")
    public GdUpdateAdGroupPayload saveCpmAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateCpmAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.saveCpmAdGroups(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateMcBannerAdGroup")
    public GdUpdateAdGroupPayload updateMcBannerAdGroup(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateMcBannerAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updateMcBannerAdGroup(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addMcBannerAdGroups")
    public GdAddAdGroupPayload addMcBannerAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddMcBannerAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addMcBannerAdGroups(
                        UidAndClientId.of(context.getSubjectUser().getUid(), context.getSubjectUser().getClientId()),
                        context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "updateMobileContentAdGroup")
    public GdUpdateAdGroupPayload updateMobileContentAdGroup(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateMobileContentAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updateMobileContentAdGroup(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @GraphQLMutation(name = "addMobileContentAdGroups")
    public GdAddAdGroupPayload addMobileContentAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddMobileContentAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService
                .addMobileContentAdGroups(context.getSubjectUser(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "addDynamicAdGroups")
    public GdAddAdGroupPayload addDynamicAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdAddDynamicAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.addDynamicAdGroups(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }

    @GraphQLNonNull
    @PreAuthorizeWrite
    @EnableLoggingOnValidationIssues
    @GraphQLMutation(name = "updateDynamicAdGroups")
    public GdUpdateAdGroupPayload updateDynamicAdGroups(
            @GraphQLRootContext GridGraphQLContext context,
            @GraphQLNonNull @GraphQLArgument(name = "input") GdUpdateDynamicAdGroup input) {
        //noinspection ConstantConditions
        return adGroupMutationService.updateDynamicAdGroups(
                context.getSubjectUser().getClientId(), context.getOperator().getUid(), input);
    }
}
